const express = require('express');
const path = require('path');
const fs = require('fs');
const multer = require('multer');
const router = express.Router();
const userModel = require('../models/userModel');

const uploadsPath = path.join(__dirname, '..', 'uploads');
try {
  fs.mkdirSync(uploadsPath, { recursive: true });
} catch (err) {
  console.error('Failed to ensure uploads directory', err.message);
}

const imageUpload = multer({
  storage: multer.diskStorage({
    destination: (_req, _file, cb) => cb(null, uploadsPath),
    filename: (_req, file, cb) => cb(null, `${Date.now()}-${file.originalname}`)
  }),
  limits: { fileSize: 8 * 1024 * 1024 },
  fileFilter: (_req, file, cb) => {
    if (!file.mimetype.startsWith('image/')) return cb(new Error('Only image uploads are allowed'));
    cb(null, true);
  }
});

router.use((req, res, next) => {
  if (!req.session || !req.session.user) return res.redirect('/login');
  next();
});

function handleUpload(field) {
  return (req, res, next) => {
    imageUpload.single(field)(req, res, (err) => {
      if (err) {
        console.error('Account upload error:', err.message);
        req.session.accountFlash = { type: 'error', message: err.message || 'Upload failed.' };
        return res.redirect('/account');
      }
      next();
    });
  };
}

function pullAccountFlash(req) {
  const flash = req.session.accountFlash || null;
  delete req.session.accountFlash;
  return flash;
}

async function renderAccountView(req, res, overrides = {}) {
  const dbUser = await userModel.findById(req.session.user.id);
  if (dbUser && req.session.user) {
    req.session.user.name = dbUser.name;
  }
  const viewUser = { ...(req.session.user || {}), role: req.session.role };
  const flash = Object.prototype.hasOwnProperty.call(overrides, 'flash') ? overrides.flash : pullAccountFlash(req);
  const force = Object.prototype.hasOwnProperty.call(overrides, 'force')
    ? Boolean(overrides.force)
    : req.query.force === '1';

  res.render('account', {
    user: viewUser,
    accountUser: dbUser,
    error: overrides.error || null,
    flash,
    force,
    success: overrides.success || false
  });
}

router.get('/account', (req, res) => {
  renderAccountView(req, res);
});

router.post('/account/profile', handleUpload('photo'), async (req, res) => {
  try {
    const id = req.session.user.id;
    const { name, signatureName } = req.body;
    const profileUpdates = {};

    if (typeof signatureName === 'string') {
      profileUpdates.signatureName = signatureName.trim();
    }
    if (req.file) {
      profileUpdates.photo = {
        url: `/uploads/${req.file.filename}`,
        originalName: req.file.originalname
      };
    }

    const signatureDataUrl = typeof req.body.signatureDataUrl === 'string' ? req.body.signatureDataUrl.trim() : '';
    if (signatureDataUrl && signatureDataUrl.startsWith('data:image/')) {
      profileUpdates.signature = { url: signatureDataUrl };
    }

    if (Object.keys(profileUpdates).length) {
      await userModel.updateProfile(id, profileUpdates);
    }

    if (typeof name === 'string' && name.trim()) {
      await userModel.updateBasicInfo(id, { name: name.trim() });
      if (req.session.user) req.session.user.name = name.trim();
    }

    req.session.accountFlash = { type: 'success', message: 'Account details updated.' };
  } catch (err) {
    console.error('Failed to update account profile', err);
    req.session.accountFlash = { type: 'error', message: 'Could not update account details.' };
  }
  res.redirect('/account');
});

router.post('/account/links', handleUpload('image'), async (req, res) => {
  try {
    const { url, title } = req.body;
    const trimmedUrl = typeof url === 'string' ? url.trim() : '';
    if (!trimmedUrl) {
      req.session.accountFlash = { type: 'error', message: 'Link URL is required.' };
      return res.redirect('/account');
    }
    const link = { url: trimmedUrl };
    const label = typeof title === 'string' ? title.trim() : '';
    if (label) link.title = label;
    if (req.file) {
      link.image = {
        url: `/uploads/${req.file.filename}`,
        originalName: req.file.originalname
      };
    }
    await userModel.addLinks(req.session.user.id, [link]);
    req.session.accountFlash = { type: 'success', message: 'Link added.' };
  } catch (err) {
    console.error('Failed to add teacher link', err);
    req.session.accountFlash = { type: 'error', message: 'Could not add link.' };
  }
  res.redirect('/account');
});

router.post('/account/links/:index/delete', async (req, res) => {
  try {
    await userModel.removeLink(req.session.user.id, req.params.index);
    req.session.accountFlash = { type: 'success', message: 'Link removed.' };
  } catch (err) {
    console.error('Failed to remove teacher link', err);
    req.session.accountFlash = { type: 'error', message: 'Could not remove link.' };
  }
  res.redirect('/account');
});

router.post('/account/password', async (req, res) => {
  const { current, password, confirm, force } = req.body;
  const user = await userModel.findById(req.session.user.id);
  if (!user || !userModel.verifyPassword(user, current)) {
    return renderAccountView(req, res, { error: 'Current password is incorrect', force });
  }
  if (!password || password !== confirm) {
    return renderAccountView(req, res, { error: 'Passwords do not match', force });
  }
  try {
    await userModel.updatePassword(user.username, password);
    await userModel.updateProfile(user.id, { mustChangePassword: false });
    req.session.accountFlash = { type: 'success', message: 'Password updated.' };
  } catch (err) {
    console.error('Password update failed', err);
    return renderAccountView(req, res, { error: 'Could not update password. Try again.', force });
  }

  if (force) {
    return res.redirect('/dashboard');
  }
  res.redirect('/account');
});

module.exports = router;
