const db = require('./db');

async function init() {
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_calendar_events (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    description TEXT NULL,
    start_at DATETIME NOT NULL,
    end_at DATETIME NULL,
    all_day TINYINT(1) DEFAULT 0,
    created_by INT NULL,
    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4`);
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_calendar_event_attendees (
    event_id INT NOT NULL,
    user_id INT NOT NULL,
    PRIMARY KEY (event_id, user_id)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4`);
}

init().catch(err => {
  console.error('calendarEventModel init failed', err);
});

function formatDateTime(value) {
  if (!value) return null;
  const d = value instanceof Date ? value : new Date(value);
  if (Number.isNaN(d.getTime())) return null;
  const pad = (num) => String(num).padStart(2, '0');
  return `${d.getFullYear()}-${pad(d.getMonth() + 1)}-${pad(d.getDate())} ${pad(d.getHours())}:${pad(d.getMinutes())}:${pad(d.getSeconds())}`;
}

function mapRow(row) {
  if (!row) return null;
  const toDate = (value) => {
    if (!value) return null;
    if (value instanceof Date) return new Date(value.getTime());
    const normalized = typeof value === 'string' ? value.replace(' ', 'T') : String(value).replace(' ', 'T');
    const parsed = new Date(normalized);
    if (!Number.isNaN(parsed.getTime())) return parsed;
    const fallback = new Date(value);
    return Number.isNaN(fallback.getTime()) ? null : fallback;
  };
  return {
    id: row.id,
    title: row.title,
    description: row.description,
    start: toDate(row.start_at),
    end: toDate(row.end_at),
    allDay: Boolean(row.all_day),
    createdBy: row.created_by,
    createdAt: toDate(row.created_at)
  };
}

async function create({ title, description, start, end, allDay = false, createdBy = null }) {
  const startValue = formatDateTime(start);
  const endValue = formatDateTime(end);
  if (!startValue) {
    throw new Error('Invalid start datetime');
  }
  const [result] = await db.query(
    'INSERT INTO mdtslms_calendar_events (title, description, start_at, end_at, all_day, created_by) VALUES (?,?,?,?,?,?)',
    [title, description || null, startValue, endValue, allDay ? 1 : 0, createdBy]
  );
  return getById(result.insertId);
}

async function getById(id) {
  const [rows] = await db.query('SELECT * FROM mdtslms_calendar_events WHERE id = ?', [id]);
  return mapRow(rows[0]);
}

async function update(id, fields = {}) {
  const updates = [];
  const params = [];

  if (fields.title !== undefined) {
    updates.push('title = ?');
    params.push(fields.title);
  }
  if (fields.description !== undefined) {
    updates.push('description = ?');
    params.push(fields.description || null);
  }
  if (fields.start !== undefined) {
    const startValue = formatDateTime(fields.start);
    if (!startValue) throw new Error('Invalid start datetime');
    updates.push('start_at = ?');
    params.push(startValue);
  }
  if (fields.end !== undefined) {
    const endValue = fields.end ? formatDateTime(fields.end) : null;
    if (fields.end && !endValue) throw new Error('Invalid end datetime');
    updates.push('end_at = ?');
    params.push(endValue);
  }
  if (fields.allDay !== undefined) {
    updates.push('all_day = ?');
    params.push(fields.allDay ? 1 : 0);
  }

  if (!updates.length) return getById(id);
  params.push(id);
  await db.query(`UPDATE mdtslms_calendar_events SET ${updates.join(', ')} WHERE id = ?`, params);
  return getById(id);
}

async function remove(id) {
  await db.query('DELETE FROM mdtslms_calendar_event_attendees WHERE event_id = ?', [id]);
  const [result] = await db.query('DELETE FROM mdtslms_calendar_events WHERE id = ?', [id]);
  return result.affectedRows > 0;
}

async function listBetween(start, end) {
  const startValue = formatDateTime(start || new Date(0));
  const endValue = formatDateTime(end || new Date('2999-12-31T23:59:59Z'));
  const [rows] = await db.query(
    `SELECT * FROM mdtslms_calendar_events
     WHERE start_at <= ?
       AND (end_at IS NULL OR end_at >= ?)
     ORDER BY start_at ASC`,
    [endValue, startValue]
  );
  return rows.map(mapRow);
}

async function getAttendeeIds(eventId) {
  const [rows] = await db.query('SELECT user_id FROM mdtslms_calendar_event_attendees WHERE event_id = ?', [eventId]);
  return rows.map((row) => Number(row.user_id)).filter((id) => Number.isInteger(id));
}

async function replaceAttendees(eventId, userIds = []) {
  const unique = [...new Set(userIds.filter((id) => Number.isInteger(id) && id > 0))];
  await db.query('DELETE FROM mdtslms_calendar_event_attendees WHERE event_id = ?', [eventId]);
  if (!unique.length) return unique;
  const placeholders = unique.map(() => '(?, ?)').join(',');
  const values = unique.flatMap((id) => [eventId, id]);
  await db.query(`INSERT INTO mdtslms_calendar_event_attendees (event_id, user_id) VALUES ${placeholders}`, values);
  return unique;
}

async function getAttendeeIdsForEvents(eventIds = []) {
  if (!Array.isArray(eventIds) || !eventIds.length) return new Map();
  const uniqueIds = [...new Set(eventIds.filter((id) => Number.isInteger(id) && id > 0))];
  if (!uniqueIds.length) return new Map();
  const [rows] = await db.query(
    `SELECT event_id, user_id FROM mdtslms_calendar_event_attendees WHERE event_id IN (${uniqueIds.map(() => '?').join(',')})`,
    uniqueIds
  );
  const map = new Map(uniqueIds.map((id) => [id, []]));
  for (const row of rows) {
    const eventId = Number(row.event_id);
    const userId = Number(row.user_id);
    if (!map.has(eventId)) map.set(eventId, []);
    if (Number.isInteger(userId)) map.get(eventId).push(userId);
  }
  return map;
}

async function listAll() {
  const [rows] = await db.query('SELECT * FROM mdtslms_calendar_events ORDER BY start_at ASC');
  return rows.map(mapRow);
}

module.exports = {
  create,
  getById,
  update,
  remove,
  getAttendeeIds,
  replaceAttendees,
  getAttendeeIdsForEvents,
  listBetween,
  listAll
};
