/**
 * Alumni Auto-Transition Middleware
 * 
 * This middleware automatically checks if students should be transitioned to alumni status
 * when their profile is accessed. This eliminates the need for a cron job.
 * 
 * Logic:
 * - Runs on every student profile load (low overhead - only checks one student)
 * - If student is 'approved' and has no active classes
 * - And last class ended >7 days ago, transition to alumni
 * 
 * Usage: Add to routes that display student profiles
 */

const classModel = require('../models/classModel');
const userModel = require('../models/userModel');

/**
 * Check if a specific student should be transitioned to alumni
 * @param {number} studentId - The student's user ID
 * @returns {Promise<boolean>} - True if transition occurred, false otherwise
 */
async function checkStudentAlumniStatus(studentId) {
  try {
    // Get student details
    const allUsers = await userModel.getAll();
    const student = allUsers.find(u => u.id === studentId);
    
    if (!student || student.role !== 'student' || student.status !== 'approved') {
      return false; // Not eligible for auto-transition
    }

    // Get all classes
    const allClasses = await classModel.getAllClasses();
    
    // Find enrolled classes for this student
    const enrolledClasses = allClasses.filter(cls => {
      if (!cls.studentIds || !Array.isArray(cls.studentIds)) return false;
      return cls.studentIds.includes(Number(studentId)) || cls.studentIds.includes(String(studentId));
    });

    // Check if student has any active classes
    const now = new Date();
    const hasActiveClass = enrolledClasses.some(cls => {
      if (!cls.endDate) return true; // No end date = active
      const endDate = new Date(cls.endDate);
      return endDate >= now;
    });

    if (hasActiveClass) {
      return false; // Student has active classes, don't transition
    }

    // Find most recent class end date
    let mostRecentEndDate = null;
    for (const cls of enrolledClasses) {
      if (cls.endDate) {
        const endDate = new Date(cls.endDate);
        if (!mostRecentEndDate || endDate > mostRecentEndDate) {
          mostRecentEndDate = endDate;
        }
      }
    }

    // Check if 7-day grace period has passed
    if (!mostRecentEndDate) {
      return false; // No class end dates found
    }

    const sevenDaysAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
    if (mostRecentEndDate >= sevenDaysAgo) {
      return false; // Grace period not over yet (class ended less than 7 days ago)
    }

    // Transition student to alumni
    console.log(`[Auto-Transition] Moving student ${studentId} (${student.username}) to alumni`);
    
    await userModel.setStatus(studentId, 'alumni');
    
    // Update profile with transition metadata
    const profile = student.profile || {};
    profile.becameAlumniAt = now.toISOString();
    profile.autoTransitionedToAlumni = true;
    profile.lastClassEndDate = mostRecentEndDate.toISOString();
    await userModel.updateProfile(studentId, profile);

    return true;
  } catch (error) {
    console.error('[Auto-Transition] Error checking student alumni status:', error);
    return false;
  }
}

/**
 * Express middleware to check alumni status before rendering student profile
 * Usage: router.get('/students/:id', checkAlumniMiddleware, (req, res) => {...})
 */
async function checkAlumniMiddleware(req, res, next) {
  const studentId = parseInt(req.params.id);
  
  if (!studentId || isNaN(studentId)) {
    return next();
  }

  try {
    await checkStudentAlumniStatus(studentId);
  } catch (error) {
    console.error('[Alumni Middleware] Error:', error);
    // Don't block the request if middleware fails
  }
  
  next();
}

/**
 * Bulk check all students (for manual/scheduled runs)
 * Can be called from an admin route or periodically
 */
async function checkAllStudentsAlumniStatus() {
  try {
    const allUsers = await userModel.getAll();
    const approvedStudents = allUsers.filter(u => 
      u.role === 'student' && u.status === 'approved'
    );

    console.log(`[Bulk Check] Checking ${approvedStudents.length} approved students`);
    
    let transitionCount = 0;
    for (const student of approvedStudents) {
      const transitioned = await checkStudentAlumniStatus(student.id);
      if (transitioned) {
        transitionCount++;
      }
    }

    console.log(`[Bulk Check] Transitioned ${transitionCount} students to alumni`);
    return { checked: approvedStudents.length, transitioned: transitionCount };
  } catch (error) {
    console.error('[Bulk Check] Error:', error);
    throw error;
  }
}

module.exports = {
  checkStudentAlumniStatus,
  checkAlumniMiddleware,
  checkAllStudentsAlumniStatus
};
