# Official vs Practice Tests Feature

## Overview
Implemented a system that allows teachers to mark tests as "Official" (graded) or "Practice" (not graded). Only official tests are recorded in the gradebook and affect student grades, while practice tests allow students to study and practice without impacting their official scores.

## Implementation Date
October 13, 2025

---

## Features Implemented

### 1. Test Creation with Official Flag
**File:** `views/create_test.ejs`

Added a checkbox to the test creation form that allows teachers to designate tests as official or practice:

```html
<div class="mb-3">
  <div class="form-check">
    <input type="checkbox" class="form-check-input" id="official" name="official" checked>
    <label class="form-check-label" for="official">
      <strong>Official Test (Graded)</strong>
      <div class="form-text">
        Official tests are recorded in the grade book. Uncheck this for practice tests that won't affect student grades.
      </div>
    </label>
  </div>
</div>
```

**Default:** Tests are official (graded) by default (checkbox is checked)

### 2. Test Storage with Official Flag
**Files:** `routes/teacher.js`, `routes/admin.js`, `models/classModel.js`

**Updated Routes:**
- POST `/teacher/classes/:id/tests` - Generate new test
- POST `/teacher/classes/:id/tests/library` - Add test from library
- POST `/teacher/classes/:id/tests/upload` - Upload CSV test
- POST `/teacher/classes/:id/tests/:testId/update` - Update test details
- POST `/admin/classes/:id/tests/:testId/update` - Admin update test

All routes now accept and save the `official` field.

**Model Update - `classModel.js addTest()`:**
```javascript
async function addTest(classId, test) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  klass.tests = klass.tests || [];
  const newTest = {
    id: (klass.tests.reduce((m, t) => Math.max(m, t.id), 0) || 0) + 1,
    title: test.title,
    timeLimit: test.timeLimit,
    dueDate: test.dueDate,
    official: test.official !== false // Default to true if not specified
  };
  klass.tests.push(newTest);
  await db.query('UPDATE mdtslms_classes SET tests=? WHERE id=?', [JSON.stringify(klass.tests), classId]);
  return newTest;
}
```

**Model Update - `classModel.js updateTest()`:**
```javascript
async function updateTest(classId, testId, fields) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  const list = Array.isArray(klass.tests) ? klass.tests : [];
  const item = list.find(t => Number(t.id) === Number(testId));
  if (!item) return null;
  if (fields.title !== undefined) item.title = fields.title;
  if (fields.dueDate !== undefined) item.dueDate = fields.dueDate;
  if (fields.official !== undefined) item.official = fields.official;
  await db.query('UPDATE mdtslms_classes SET tests=? WHERE id=?', [JSON.stringify(list), classId]);
  return item;
}
```

### 3. Conditional Grade Recording
**File:** `routes/student.js` - POST `/classes/:id/tests/:testId`

Modified test submission to only record grades for official tests:

```javascript
const pct = Math.round((score / test.questions.length) * 100);

// Only record grade if test is official (graded)
if (test.official !== false) {
  await classModel.recordGrade(id, testId, req.session.user.id, pct);
  console.log('Grade recorded for official test', { classId: id, testId, userId: req.session.user.id, score: pct });
} else {
  console.log('Practice test completed (not graded)', { classId: id, testId, userId: req.session.user.id, score: pct });
}
```

**Behavior:**
- **Official tests:** Grade is recorded in `klass.grades` array
- **Practice tests:** Score is calculated and shown, but NOT recorded in gradebook

### 4. Visual Indicators
**File:** `views/view_class.ejs`

**Student View:**
```html
<span><%= t.title %></span>
<% if (t.official === false) { %>
  <span class="badge bg-secondary ms-2">Practice</span>
<% } else { %>
  <span class="badge bg-primary ms-2">Official</span>
<% } %>
```

**Teacher/Admin View:**
```html
<%= t.title %>
<% if (t.official === false) { %>
  <span class="badge bg-secondary ms-2">Practice</span>
<% } else { %>
  <span class="badge bg-success ms-2">Official (Graded)</span>
<% } %>
```

**Badge Colors:**
- **Official tests:** Green badge ("Official (Graded)" for teachers, "Official" for students)
- **Practice tests:** Gray badge ("Practice")

### 5. Toggle Official Status
**File:** `views/view_class.ejs`

Teachers and admins can change a test's official status after creation:

```html
<form id="test-edit-<%= t.id %>" method="post" action="/admin/classes/<%= klass.id %>/tests/<%= t.id %>/update" class="d-none mt-1 d-flex gap-2 flex-wrap align-items-center">
  <input type="text" name="title" value="<%- t.title %>" class="form-control" style="max-width:200px">
  <input type="date" name="dueDate" value="<%= t.dueDate %>" class="form-control" style="max-width:200px">
  <div class="form-check">
    <input type="checkbox" class="form-check-input" id="official-<%= t.id %>" name="official" <%= t.official !== false ? 'checked' : '' %>>
    <label class="form-check-label small" for="official-<%= t.id %>">Official (Graded)</label>
  </div>
  <button class="btn btn-sm btn-primary" type="submit">Save</button>
</form>
```

**Usage:**
1. Click "Edit" button next to test
2. Toggle the "Official (Graded)" checkbox
3. Click "Save"

### 6. Grade Report Filtering
**File:** `views/grade_report.ejs`

Grade reports now only display official tests:

**Grade Calculation:**
```javascript
// Test average (only official tests)
const officialTests = tests.filter(t => t.official !== false);
const testGrades = officialTests.map(t => {
  const gradeRecord = allGrades.find(gr => gr.testId === t.id && gr.studentId === studentId);
  return gradeRecord && gradeRecord.score !== undefined && gradeRecord.score !== null && gradeRecord.score !== '' 
    ? parseFloat(gradeRecord.score) 
    : null;
}).filter(g => g !== null);
const testAvg = testGrades.length > 0 ? testGrades.reduce((a, b) => a + b, 0) / testGrades.length : 0;
```

**Display:**
```html
<% 
// Only show official tests in grade report
const officialTests = klass.tests.filter(t => t.official !== false);
officialTests.forEach((test, idx) => { 
  // ... display test with "Official" badge
%>
```

---

## User Workflows

### Teacher Creates Official Test

1. Navigate to class page
2. Click "Create Test" or use upload/library
3. Fill in test details
4. **Check "Official Test (Graded)" checkbox** (checked by default)
5. Submit test
6. Test appears with **green "Official (Graded)" badge**
7. When students take test, **grades are recorded**

### Teacher Creates Practice Test

1. Navigate to class page
2. Click "Create Test" or use upload/library
3. Fill in test details
4. **Uncheck "Official Test (Graded)" checkbox**
5. Submit test
6. Test appears with **gray "Practice" badge**
7. When students take test, **grades are NOT recorded**

### Teacher Changes Test Status

1. Navigate to class page
2. Find test in tests list
3. Click "Edit" button
4. Toggle "Official (Graded)" checkbox
5. Click "Save"
6. Badge updates immediately
7. Future test submissions respect new status

### Student Takes Official Test

1. Navigate to class page
2. See test with **blue "Official" badge**
3. Click "Take" to start test
4. Complete and submit test
5. Grade is **recorded in gradebook**
6. Grade appears in:
   - Test results page
   - Grade report
   - Class view (badge shows score)

### Student Takes Practice Test

1. Navigate to class page
2. See test with **gray "Practice" badge**
3. Click "Take" to start test
4. Complete and submit test
5. Score is shown, but **NOT recorded**
6. Can retake unlimited times for practice
7. Does not appear in grade calculations

---

## Benefits

### For Teachers
✅ **Flexible Assessment:** Create both graded tests and practice tests
✅ **Easy Management:** Toggle test status anytime with edit button
✅ **Clear Labeling:** Visual badges show test type at a glance
✅ **No Database Changes:** Works with existing database structure

### For Students
✅ **Practice Safely:** Take practice tests without grade impact
✅ **Clear Expectations:** Know which tests count toward grades
✅ **Unlimited Retakes:** Practice tests can be taken multiple times
✅ **Study Effectively:** Use practice tests for exam preparation

### For Administrators
✅ **Accurate Grading:** Only official tests affect grade calculations
✅ **Grade Report Clarity:** Grade reports show only official assessments
✅ **Data Integrity:** Practice test scores don't pollute grade data
✅ **Audit Trail:** Can see all test attempts (with future logging)

---

## Technical Details

### Database Storage

Tests are stored in the `mdtslms_classes` table in the `tests` JSON column:

```json
{
  "id": 1,
  "title": "Module 1 Exam",
  "timeLimit": 90,
  "dueDate": "2025-10-20",
  "official": true
}
```

**Field:** `official` (Boolean)
- `true` or `undefined` = Official (graded) test
- `false` = Practice (not graded) test

### Grade Recording Logic

**Location:** `routes/student.js` - POST test submission

```javascript
if (test.official !== false) {
  await classModel.recordGrade(id, testId, req.session.user.id, pct);
}
```

**Condition:** `test.official !== false`
- Records grade if `official` is `true` or `undefined` (for backward compatibility)
- Skips grade recording if `official` is explicitly `false`

### Backward Compatibility

**Existing Tests:**
- Tests created before this feature have no `official` field
- These tests are treated as official (graded) by default
- Condition `test.official !== false` evaluates to `true` for `undefined`

**Migration:** No database migration required - existing tests work as official tests

---

## Testing Checklist

- [x] Create official test - checkbox checked by default
- [x] Create practice test - uncheck checkbox
- [x] Official test records grade in database
- [x] Practice test does NOT record grade
- [x] Visual badges appear correctly (blue/gray)
- [x] Edit button allows toggling official status
- [x] Grade report shows only official tests
- [x] Grade calculations exclude practice tests
- [x] Students can see test type (badges)
- [x] Teachers can see "Official (Graded)" label
- [x] Upload CSV test respects official flag
- [x] Library test respects official flag
- [x] AI-generated test respects official flag
- [x] Existing tests treated as official

---

## Future Enhancements (Optional)

### Attempt Tracking
Track all test attempts (official and practice) in a separate table:
```sql
CREATE TABLE test_attempts (
  id INT AUTO_INCREMENT PRIMARY KEY,
  student_id INT,
  class_id INT,
  test_id INT,
  score INT,
  official BOOLEAN,
  taken_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### Practice Test Analytics
Show teachers:
- How many students practiced before taking official test
- Average practice scores vs official scores
- Time spent on practice tests

### Test Mode Selection
Allow students to explicitly choose:
- "Practice Mode" (score not recorded, see answers after)
- "Official Mode" (score recorded, limited attempts)

### Scheduled Status Changes
Auto-convert practice tests to official on specific date:
```javascript
{
  "official": false,
  "becomesOfficialOn": "2025-10-20T00:00:00Z"
}
```

### Conditional Official Status
Test is official only after completing prerequisite:
```javascript
{
  "official": true,
  "officialOnlyAfter": {
    "type": "lecture",
    "id": 5
  }
}
```

---

## Related Files

- `views/create_test.ejs` - Test creation form with official checkbox
- `routes/teacher.js` - Test creation and update routes
- `routes/admin.js` - Admin test update route
- `routes/student.js` - Test submission with conditional grade recording
- `models/classModel.js` - addTest and updateTest functions
- `views/view_class.ejs` - Test list with badges and edit form
- `views/grade_report.ejs` - Filtered grade calculations
- `views/test_result.ejs` - Test results display

---

## Summary

The Official vs Practice Tests feature provides teachers with flexible assessment tools while maintaining clean, accurate grade data. Students benefit from unlimited practice opportunities without grade penalties, and administrators get precise grade reports that reflect only official assessments. The implementation is backward-compatible, requires no database migrations, and integrates seamlessly with existing test workflows.
