# Drag-and-Drop Field Placement Fix

## Problem
The click-and-drag feature for placing signature fields on PDF documents wasn't working because the `<embed>` tag doesn't support overlaying interactive HTML elements.

## Solution
Converted the PDF viewer from embed-based to image-based rendering:

### 1. Installed PDF Processing Libraries
```bash
npm install pdf-lib pdfjs-dist canvas
```

### 2. Created PDF-to-Image Utility (`utils/pdfToImage.js`)
- `convertPdfPageToImage(pdfPath, pageNumber, scale)` - Converts a single PDF page to PNG
- `getPdfPageCount(pdfPath)` - Gets total page count of a PDF
- `convertPdfToImages(pdfPath, scale)` - Converts all pages to PNG images

### 3. Added Image Rendering Route
**Route:** `GET /document-signing/pdf-page/:documentId/:pageNumber`

This route:
- Reads the PDF file from the database
- Converts the requested page to PNG format
- Serves it as an image with 1-day cache headers
- Defaults to scale 2.0 for high-quality rendering

### 4. Updated Edit View (`views/admin_signature_document_edit.ejs`)
Changed from:
```html
<embed src="/uploads/document.pdf" type="application/pdf">
```

To:
```html
<img src="/document-signing/pdf-page/<%= document.id %>/1" 
     id="pdfImage"
     style="width: 100%; display: block; user-select: none;"
     draggable="false">
```

Added page navigation buttons that:
- Switch the image source to different pages
- Call `loadFieldsForPage()` to show only relevant fields
- Update the active button state

### 5. Updated Field Editor JavaScript (`public/js/signature-field-editor.js`)

**Modified `initDragAndDrop()`:**
- Changed from `pdfPage.querySelector('embed')` to `document.getElementById('pdfImage')`
- Calculate drop positions relative to the image element
- Prevent default image drag behavior

**Modified `confirmField()`:**
- Added `page_number: parseInt(pdfPage.dataset.page)` to field data
- Tracks which page each field belongs to

**Added `loadFieldsForPage(pageNumber)` function:**
- Removes all currently rendered field elements
- Filters `placedFields` by `page_number`
- Renders only fields for the current page

**Updated `loadExistingFields()`:**
- Only renders fields for page 1 initially
- All fields are still stored in `placedFields` array
- Page switching will show fields for other pages

## How It Works

1. **PDF Upload**: Admin uploads a PDF document
2. **Field Placement**: Admin opens the edit page
   - PDF is converted to PNG on-the-fly (server-side)
   - Image is displayed with absolute positioning container
   - Admin drags field types onto the image
   - Fields are positioned absolutely over the image
   - Each field stores its page number
3. **Multi-Page Support**:
   - Page buttons switch the image source
   - Only fields for current page are shown
   - All fields are saved to database with page numbers
4. **Signing Process**:
   - Signers see fields overlaid on PDF images
   - Only fields for their role (student/staff) are shown
   - Fields appear on correct pages

## Performance
- Server-side rendering with pdfjs-dist ensures consistent output
- 1-day cache headers reduce repeated conversions
- Images cached by browser for fast page switching
- Scale 2.0 provides high quality without excessive file size

## Testing Checklist
- [ ] Single-page PDF drag-and-drop works
- [ ] Multi-page PDF page switching works
- [ ] Fields save with correct page numbers
- [ ] Fields only show on their assigned pages
- [ ] Signature and text fields render correctly
- [ ] Student can sign documents with field overlays
- [ ] Staff can sign documents with field overlays
- [ ] Email notifications still work

## Files Modified
1. `utils/pdfToImage.js` - NEW
2. `routes/documentSigning.js` - Added `/pdf-page` route, added `pageCount` to edit route
3. `views/admin_signature_document_edit.ejs` - Changed embed to img tag
4. `public/js/signature-field-editor.js` - Updated for image-based positioning
5. `package.json` - Added pdf-lib, pdfjs-dist, canvas dependencies
