# Document View Page Improvements

## Issues Fixed

### 1. PDF Not Showing Signatures
**Problem:** The document view page was not displaying signatures on the PDF preview.

**Root Cause:** The `getDocumentById` method in the model was not including field responses (signature data) when fetching the document. The fields were retrieved, but without the `response_value` column that contains the actual signatures.

**Solution:** Updated the `getDocumentById` query to LEFT JOIN with `signature_field_responses` table:

```sql
SELECT sf.*, sfr.response_value, sfr.response_type
FROM signature_fields sf
LEFT JOIN signature_field_responses sfr ON sf.id = sfr.field_id
WHERE sf.document_id = ?
```

Now when `generateSignedPDF` is called, each field has its `response_value` attached, allowing signatures to be embedded in the PDF.

### 2. Admin Cannot Sign Staff Documents from View Page
**Problem:** Admins could only sign staff documents from the dashboard modal or email links. When viewing a document, they couldn't sign pending staff signatures.

**Solution:** Added action buttons in the Signing Parties table:
- New "Action" column (visible only to admins)
- "Sign as Staff" button for pending/viewed staff parties
- Button links directly to signing page
- Shows completion status for already-signed parties

### 3. PDF Display Enhancement
**Problem:** Embed tags don't always work reliably across browsers for PDFs.

**Solution:** Improved PDF display with multiple fallback options:
1. Primary: iframe with PDF
2. Fallback: object tag
3. Final fallback: Links to open in new tab or download
4. Added "Open in New Tab" button for easier viewing

## Technical Changes

### models/signatureDocumentModel.js

#### Updated getDocumentById
**Before:**
```javascript
const [fields] = await pool.query(
  'SELECT * FROM signature_fields WHERE document_id = ? ORDER BY page_number, y_position',
  [documentId]
);
```

**After:**
```javascript
const [fields] = await pool.query(`
  SELECT sf.*, sfr.response_value, sfr.response_type
  FROM signature_fields sf
  LEFT JOIN signature_field_responses sfr ON sf.id = sfr.field_id
  WHERE sf.document_id = ?
  ORDER BY sf.page_number, sf.y_position
`, [documentId]);
```

#### Fixed getPendingStaffSignatures
Changed `sp.email` to `u.email` (email comes from user table, not parties table).

### views/signature_document_view.ejs

#### Added Admin Action Column
```html
<% if (role === 'admin') { %>
  <th>Action</th>
<% } %>
```

In table rows:
```html
<% if (role === 'admin') { %>
  <td>
    <% if (party.status !== 'signed' && party.party_type === 'staff') { %>
      <a href="/document-signing/sign/<%= document.id %>/<%= party.id %>" 
         class="btn btn-sm btn-warning">
        <i class="bi bi-pen-fill"></i> Sign as Staff
      </a>
    <% } %>
  </td>
<% } %>
```

#### Enhanced PDF Display
```html
<iframe 
  src="/document-signing/pdf-signed/<%= document.id %>" 
  type="application/pdf" 
  width="100%" 
  height="700px">
  <object data="/document-signing/pdf-signed/<%= document.id %>" type="application/pdf">
    <p>Unable to display PDF. 
      <a href="/document-signing/pdf-signed/<%= document.id %>" target="_blank">Open PDF</a>
      or <a href="/document-signing/download/<%= document.id %>">Download PDF</a>
    </p>
  </object>
</iframe>
```

## How It Works Now

### Signature Display Flow

1. User opens `/document-signing/view/:documentId`
2. Route calls `signatureDocumentModel.getDocumentById(documentId)`
3. Model fetches document, fields with responses, and parties
4. View renders with PDF embed
5. Browser requests `/document-signing/pdf-signed/:documentId`
6. Route calls `documentSigningUtils.generateSignedPDF(documentId)`
7. Utility function:
   - Loads original PDF with pdf-lib
   - Iterates through fields
   - For each field with a response_value:
     - If signature: Embeds image at coordinates
     - If text: Draws text with background
   - Saves modified PDF
8. PDF buffer sent to browser
9. Browser displays PDF with signatures visible

### Admin Signing from View Page

1. Admin views document at `/document-signing/view/:documentId`
2. Sees table of signing parties
3. For staff parties with status != 'signed':
   - "Sign as Staff" button appears
4. Clicks button
5. Redirects to `/document-signing/sign/:documentId/:partyId`
6. Admin can sign on behalf of staff
7. After signing, redirects back to view page
8. Status updates, signature appears on PDF

## Benefits

1. **Signatures Visible**: PDFs now show actual signatures, not blank fields
2. **Admin Efficiency**: Can sign staff documents from any view, not just dashboard
3. **Better UX**: Multiple PDF viewing options with fallbacks
4. **Transparency**: Clear visual indication of who has/hasn't signed
5. **Convenient Actions**: Direct links to sign from view page

## Testing Checklist

- [x] PDF displays with signatures embedded
- [x] Download includes signatures
- [x] Admin sees "Action" column in parties table
- [x] "Sign as Staff" button appears for pending staff parties
- [x] Button does not appear for student parties
- [x] Button does not appear for already-signed parties
- [x] Clicking button opens signing page
- [x] Admin can sign successfully
- [x] After signing, PDF updates with new signature
- [x] PDF fallback works if browser doesn't support iframe
- [x] "Open in New Tab" button works
- [x] Non-admin users don't see action column

## Files Modified

1. **models/signatureDocumentModel.js**
   - Updated `getDocumentById` to LEFT JOIN with field responses
   - Fixed `getPendingStaffSignatures` query (u.email not sp.email)

2. **views/signature_document_view.ejs**
   - Added Action column for admins
   - Added "Sign as Staff" buttons
   - Enhanced PDF embed with iframe/object/fallback
   - Added "Open in New Tab" button

## Known Limitations

- Multiple responses per field (if a field is signed by multiple people) will show only one response. This is expected behavior as each signer should have their own fields.
- PDF.js in the signing view shows fields as overlays; the view page shows the final embedded PDF - these are intentionally different interfaces.

## Future Enhancements

- Add ability to preview document before sending for signature
- Show signature thumbnails in parties table
- Add bulk signing capability for admins
- Email notification when admin signs on behalf of staff
