/**
 * Document Signing Utilities
 * Helper functions for PDF processing, emails, and signature tracking
 */

const nodemailer = require('nodemailer');
const fs = require('fs').promises;
const path = require('path');
const signatureDocumentModel = require('../models/signatureDocumentModel');
const branding = require('../branding.json');
const PDFDocument = require('pdfkit');

// Email configuration - adjust based on your setup
let transporter = null;

/**
 * Initialize email transporter
 */
function initializeEmailTransporter() {
  if (!transporter) {
    transporter = nodemailer.createTransport({
      host: process.env.SMTP_HOST || 'smtp.gmail.com',
      port: process.env.SMTP_PORT || 587,
      secure: false,
      auth: {
        user: process.env.SMTP_USER,
        pass: process.env.SMTP_PASS
      }
    });
  }
  return transporter;
}

/**
 * Generate HTML email template
 */
function generateEmailHtml(subject, bodyHtml) {
  const logo = branding?.primaryLogo || '';
  const schoolName = branding?.schoolName || 'MD Technical School';
  
  return `
    <div style="background:#f6f8fb; padding:24px; font-family:Segoe UI,Roboto,Arial,sans-serif;">
      <div style="max-width:680px; margin:0 auto; background:#ffffff; border:1px solid #e5e7eb; border-radius:12px; overflow:hidden;">
        ${logo ? `<div style="padding:24px; text-align:center; border-bottom:1px solid #e5e7eb;">
          <img src="${logo}" alt="${schoolName}" style="max-height:60px;">
        </div>` : ''}
        <div style="padding:32px;">
          <h2 style="margin:0 0 16px; color:#1e293b; font-size:24px;">${subject}</h2>
          ${bodyHtml}
        </div>
        <div style="padding:24px; background:#f8fafc; border-top:1px solid #e5e7eb; text-align:center;">
          <p style="margin:0; color:#64748b; font-size:13px;">
            &copy; ${new Date().getFullYear()} ${schoolName}. All rights reserved.
          </p>
        </div>
      </div>
    </div>
  `;
}

/**
 * Send signature request email
 */
async function sendSignatureRequestEmail(partyInfo, documentInfo, signUrl) {
  // Check if SMTP is configured
  if (!process.env.SMTP_USER || !process.env.SMTP_PASS) {
    console.warn('SMTP credentials not configured. Email not sent to:', partyInfo.email);
    console.log('Sign URL:', signUrl);
    return false;
  }

  const transporter = initializeEmailTransporter();
  
  if (!transporter) {
    console.warn('Email transporter not available. Email not sent to:', partyInfo.email);
    return false;
  }
  
  const subject = `Action Required: Sign "${documentInfo.title}"`;
  const bodyHtml = `
    <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
      Hello ${partyInfo.firstName} ${partyInfo.lastName},
    </p>
    <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
      You have been requested to review and sign the document: <strong>${documentInfo.title}</strong>
    </p>
    ${documentInfo.description ? `
      <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
        <em>${documentInfo.description}</em>
      </p>
    ` : ''}
    <p style="color:#475569; line-height:1.6; margin:0 0 24px;">
      Please click the button below to view and sign the document.
    </p>
    <div style="text-align:center; margin:0 0 24px;">
      <a href="${signUrl}" 
         style="display:inline-block; padding:12px 32px; background:#3b82f6; color:#ffffff; 
                text-decoration:none; border-radius:8px; font-weight:600;">
        Review & Sign Document
      </a>
    </div>
    <p style="color:#64748b; font-size:13px; line-height:1.6; margin:0;">
      If the button doesn't work, copy and paste this link into your browser:<br>
      <a href="${signUrl}" style="color:#3b82f6; word-break:break-all;">${signUrl}</a>
    </p>
  `;

  const mailOptions = {
    from: process.env.SMTP_FROM || process.env.SMTP_USER,
    to: partyInfo.email,
    subject: subject,
    html: generateEmailHtml(subject, bodyHtml)
  };

  try {
    await transporter.sendMail(mailOptions);
    return true;
  } catch (error) {
    console.error('Error sending email:', error);
    console.log('Sign URL for', partyInfo.email, ':', signUrl);
    return false;
  }
}

/**
 * Send signature request emails to all parties
 */
async function sendSignatureRequestEmails(documentId) {
  const document = await signatureDocumentModel.getDocumentById(documentId);
  
  if (!document || !document.parties) {
    throw new Error('Document or parties not found');
  }

  const baseUrl = process.env.BASE_URL || 'http://localhost:3012';
  
  for (const party of document.parties) {
    const signUrl = `${baseUrl}/document-signing/sign/${documentId}/${party.id}`;
    
    const partyInfo = {
      firstName: party.firstName,
      lastName: party.lastName,
      email: party.email
    };
    
    const documentInfo = {
      title: document.title,
      description: document.description
    };
    
    const sent = await sendSignatureRequestEmail(partyInfo, documentInfo, signUrl);
    
    if (sent) {
      await signatureDocumentModel.markEmailSent(party.id);
    }
  }
}

/**
 * Send completion email with signed document
 */
async function sendCompletionEmail(partyInfo, documentInfo, downloadUrl, pdfBuffer) {
  const transporter = initializeEmailTransporter();
  
  const subject = `✅ Signatures Complete: "${documentInfo.title}"`;
  const bodyHtml = `
    <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
      Hello ${partyInfo.firstName} ${partyInfo.lastName},
    </p>
    <div style="background:#dcfce7; border-left:4px solid #10b981; padding:16px; margin:0 0 24px; border-radius:4px;">
      <p style="color:#166534; font-weight:600; margin:0 0 8px; font-size:16px;">
        ✅ All Signatures Complete!
      </p>
      <p style="color:#15803d; margin:0; line-height:1.5;">
        The document "<strong>${documentInfo.title}</strong>" has been signed by all required parties and is now finalized.
      </p>
    </div>
    <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
      <strong>Your signed copy is attached to this email.</strong>
    </p>
    <p style="color:#475569; line-height:1.6; margin:0 0 24px;">
      You can also view and download the document anytime from your account dashboard.
    </p>
    <div style="text-align:center; margin:0 0 24px;">
      <a href="${downloadUrl}" 
         style="display:inline-block; padding:12px 32px; background:#10b981; color:#ffffff; 
                text-decoration:none; border-radius:8px; font-weight:600; font-size:16px;">
        📄 View Final Document
      </a>
    </div>
    <p style="color:#64748b; font-size:13px; line-height:1.6; margin:0; padding-top:16px; border-top:1px solid #e5e7eb;">
      <strong>Important:</strong> Please save this document for your records. This is the official signed copy.
    </p>
  `;

  const attachmentFilename = `${documentInfo.title.replace(/[^a-z0-9]/gi, '_')}_signed.pdf`;

  const mailOptions = {
    from: process.env.SMTP_FROM || process.env.SMTP_USER,
    to: partyInfo.email,
    subject: subject,
    html: generateEmailHtml(subject, bodyHtml),
    attachments: pdfBuffer ? [{
      filename: attachmentFilename,
      content: pdfBuffer
    }] : []
  };

  try {
    await transporter.sendMail(mailOptions);
    return true;
  } catch (error) {
    console.error('Error sending completion email:', error);
    return false;
  }
}

/**
 * Send completion emails to all parties
 */
async function sendCompletionEmails(documentId) {
  const document = await signatureDocumentModel.getDocumentById(documentId);
  
  if (!document || !document.parties) {
    throw new Error('Document or parties not found');
  }

  const baseUrl = process.env.BASE_URL || 'http://localhost:3012';
  const downloadUrl = `${baseUrl}/document-signing/download/${documentId}`;
  
  // Generate signed PDF once
  let pdfBuffer = null;
  try {
    pdfBuffer = await generateSignedPDF(documentId);
  } catch (error) {
    console.error('Error generating PDF for completion emails:', error);
  }
  
  const documentInfo = {
    title: document.title,
    description: document.description
  };
  
  for (const party of document.parties) {
    const partyInfo = {
      firstName: party.firstName,
      lastName: party.lastName,
      email: party.email
    };
    
    await sendCompletionEmail(partyInfo, documentInfo, downloadUrl, pdfBuffer);
  }
}

/**
 * Send progress notification email when a signature is added
 */
async function sendProgressNotificationEmail(documentId, signerName, signerType) {
  // Check if SMTP is configured
  if (!process.env.SMTP_USER || !process.env.SMTP_PASS) {
    console.warn('SMTP credentials not configured. Progress notification not sent.');
    return false;
  }

  const document = await signatureDocumentModel.getDocumentById(documentId);
  
  if (!document || !document.parties) {
    return false;
  }

  const baseUrl = process.env.BASE_URL || 'http://localhost:3012';
  const viewUrl = `${baseUrl}/document-signing/view/${documentId}`;
  
  const transporter = initializeEmailTransporter();
  
  if (!transporter) {
    console.warn('Email transporter not available. Progress notification not sent.');
    return false;
  }

  // Count signatures
  const totalParties = document.parties.length;
  const signedCount = document.parties.filter(p => p.status === 'signed').length;
  const remainingCount = totalParties - signedCount;
  
  // Send notification to ALL parties (signed and unsigned)
  for (const party of document.parties) {
    const isThisSigner = party.firstName === signerName.split(' ')[0] && 
                         party.lastName === signerName.split(' ')[1];
    
    const subject = isThisSigner ? 
      `Confirmation: You signed "${document.title}"` :
      `Update: "${document.title}" has been signed by ${signerName}`;
    
    let bodyHtml;
    
    if (isThisSigner) {
      // Email to the person who just signed
      bodyHtml = `
        <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
          Hello ${party.firstName} ${party.lastName},
        </p>
        <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
          Thank you for signing "<strong>${document.title}</strong>". Your signature has been recorded.
        </p>
        <div style="background:#f0fdf4; border-left:4px solid #10b981; padding:16px; margin:0 0 16px; border-radius:4px;">
          <p style="color:#166534; font-weight:600; margin:0 0 8px;">
            📊 Progress Update
          </p>
          <p style="color:#166534; margin:0;">
            ${signedCount} of ${totalParties} signature${totalParties > 1 ? 's' : ''} completed
          </p>
        </div>
        ${remainingCount > 0 ? `
          <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
            <strong>Next Steps:</strong> We're waiting for ${remainingCount} more signature${remainingCount > 1 ? 's' : ''}. 
            You'll receive another notification when the document is fully executed.
          </p>
        ` : `
          <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
            🎉 <strong>All signatures have been collected!</strong> The document is now complete.
          </p>
        `}
        <div style="text-align:center; margin:0 0 24px;">
          <a href="${viewUrl}" 
             style="display:inline-block; padding:12px 32px; background:#3b82f6; color:#ffffff; 
                    text-decoration:none; border-radius:8px; font-weight:600;">
            View Document Status
          </a>
        </div>
        <p style="color:#64748b; font-size:13px; line-height:1.6; margin:0;">
          If the button doesn't work, copy and paste this link into your browser:<br>
          <a href="${viewUrl}" style="color:#3b82f6; word-break:break-all;">${viewUrl}</a>
        </p>
      `;
    } else {
      // Email to other parties
      bodyHtml = `
        <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
          Hello ${party.firstName} ${party.lastName},
        </p>
        <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
          We wanted to update you on the progress of "<strong>${document.title}</strong>".
        </p>
        <div style="background:#eff6ff; border-left:4px solid #3b82f6; padding:16px; margin:0 0 16px; border-radius:4px;">
          <p style="color:#1e40af; font-weight:600; margin:0 0 8px;">
            ✍️ Recent Activity
          </p>
          <p style="color:#1e40af; margin:0;">
            ${signerName} (${signerType}) has signed the document
          </p>
        </div>
        <div style="background:#f0fdf4; border-left:4px solid #10b981; padding:16px; margin:0 0 16px; border-radius:4px;">
          <p style="color:#166534; font-weight:600; margin:0 0 8px;">
            📊 Progress Update
          </p>
          <p style="color:#166534; margin:0;">
            ${signedCount} of ${totalParties} signature${totalParties > 1 ? 's' : ''} completed
          </p>
        </div>
        ${party.status !== 'signed' ? `
          <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
            <strong>Your Action:</strong> Your signature is ${party.status === 'pending' ? 'still' : 'also'} required. 
            Please review and sign the document at your earliest convenience.
          </p>
          <div style="text-align:center; margin:0 0 24px;">
            <a href="${baseUrl}/document-signing/sign/${documentId}/${party.id}" 
               style="display:inline-block; padding:12px 32px; background:#f59e0b; color:#ffffff; 
                      text-decoration:none; border-radius:8px; font-weight:600;">
              Sign Now
            </a>
          </div>
        ` : `
          <p style="color:#475569; line-height:1.6; margin:0 0 16px;">
            You have already signed this document. ${remainingCount > 0 ? 
              `We're waiting for ${remainingCount} more signature${remainingCount > 1 ? 's' : ''}.` :
              'All signatures have been collected!'}
          </p>
          <div style="text-align:center; margin:0 0 24px;">
            <a href="${viewUrl}" 
               style="display:inline-block; padding:12px 32px; background:#3b82f6; color:#ffffff; 
                      text-decoration:none; border-radius:8px; font-weight:600;">
              View Document
            </a>
          </div>
        `}
        <p style="color:#64748b; font-size:13px; line-height:1.6; margin:0;">
          If the button doesn't work, copy and paste this link into your browser:<br>
          <a href="${party.status !== 'signed' ? 
            `${baseUrl}/document-signing/sign/${documentId}/${party.id}` : 
            viewUrl}" 
            style="color:#3b82f6; word-break:break-all;">
            ${party.status !== 'signed' ? 
              `${baseUrl}/document-signing/sign/${documentId}/${party.id}` : 
              viewUrl}
          </a>
        </p>
      `;
    }

    const mailOptions = {
      from: process.env.SMTP_FROM || process.env.SMTP_USER,
      to: party.email,
      subject: subject,
      html: generateEmailHtml(subject, bodyHtml)
    };

    try {
      await transporter.sendMail(mailOptions);
      console.log(`Progress notification sent to: ${party.email}`);
    } catch (error) {
      console.error(`Error sending progress notification to ${party.email}:`, error);
    }
  }
  
  return true;
}

/**
 * Check if a party can sign based on signing order
 */
async function canPartySign(document, party) {
  if (party.status === 'signed') {
    return false; // Already signed
  }

  if (document.signing_order === 'any_order') {
    return true; // Anyone can sign anytime
  }

  // Check if this party's turn based on signing_order
  const sortedParties = document.parties.sort((a, b) => a.signing_order - b.signing_order);
  
  for (const p of sortedParties) {
    if (p.id === party.id) {
      return true; // It's this party's turn
    }
    if (p.status !== 'signed') {
      return false; // Someone before this party hasn't signed yet
    }
  }

  return false;
}

/**
 * Notify the next signer after someone signs
 */
async function notifyNextSigner(documentId, justSignedParty) {
  const document = await signatureDocumentModel.getDocumentById(documentId);
  
  if (document.signing_order === 'any_order') {
    return; // No specific order, so no need to notify next
  }

  const sortedParties = document.parties.sort((a, b) => a.signing_order - b.signing_order);
  
  // Find the next unsigned party
  for (const party of sortedParties) {
    if (party.signing_order > justSignedParty.signing_order && party.status === 'pending') {
      const baseUrl = process.env.BASE_URL || 'http://localhost:3012';
      const signUrl = `${baseUrl}/document-signing/sign/${documentId}/${party.id}`;
      
      const partyInfo = {
        firstName: party.firstName,
        lastName: party.lastName,
        email: party.email
      };
      
      const documentInfo = {
        title: document.title,
        description: document.description
      };
      
      await sendSignatureRequestEmail(partyInfo, documentInfo, signUrl);
      break; // Only notify the next person
    }
  }
}

/**
 * Generate a signed PDF with all signatures and field data overlaid
 * This is a simplified version - for production, consider using pdf-lib
 */
async function generateSignedPDF(documentId) {
  const { PDFDocument, rgb } = require('pdf-lib');
  
  const document = await signatureDocumentModel.getDocumentById(documentId);
  
  if (!document) {
    throw new Error('Document not found');
  }

  // Read the original PDF
  const originalPdfBuffer = await fs.readFile(document.file_path);
  
  // Load the PDF document
  const pdfDoc = await PDFDocument.load(originalPdfBuffer);
  const pages = pdfDoc.getPages();
  
  // Get all field responses
  const fields = document.fields || [];
  const responses = {};
  
  // Organize responses by field_id
  for (const field of fields) {
    if (field.response_value) {
      responses[field.id] = field.response_value;
    }
  }
  
  // Add signatures and field data to PDF
  for (const field of fields) {
    const pageIndex = (field.page_number || 1) - 1;
    
    if (pageIndex < 0 || pageIndex >= pages.length) {
      continue;
    }
    
    const page = pages[pageIndex];
    const { width, height } = page.getSize();
    
    // Convert percentage positions to actual coordinates
    // Note: PDF coordinates start from bottom-left, canvas from top-left
    const x = (field.x_position / 100) * width;
    const y = height - ((field.y_position / 100) * height) - ((field.height / 100) * height);
    const fieldWidth = (field.width / 100) * width;
    const fieldHeight = (field.height / 100) * height;
    
    const responseValue = responses[field.id];
    
    if (responseValue) {
      // Handle signature and initials fields (image data)
      if ((field.field_type === 'signature' || field.field_type === 'initials') && responseValue.startsWith('data:image')) {
        try {
          // Extract base64 data from data URL
          const base64Data = responseValue.split(',')[1];
          const imageBytes = Buffer.from(base64Data, 'base64');
          
          // Embed the signature/initials image
          let image;
          if (responseValue.includes('image/png')) {
            image = await pdfDoc.embedPng(imageBytes);
          } else if (responseValue.includes('image/jpeg') || responseValue.includes('image/jpg')) {
            image = await pdfDoc.embedJpg(imageBytes);
          }
          
          if (image) {
            // Draw the signature/initials image
            page.drawImage(image, {
              x: x,
              y: y,
              width: fieldWidth,
              height: fieldHeight,
            });
          }
        } catch (error) {
          console.error('Error embedding signature/initials image:', error);
          // Fallback to text
          page.drawText(field.field_type === 'initials' ? '[Initials]' : '[Signature]', {
            x: x + 5,
            y: y + fieldHeight / 2,
            size: 10,
            color: rgb(0, 0, 0),
          });
        }
      } 
      // Handle checkbox fields
      else if (field.field_type === 'checkbox') {
        if (responseValue === 'true' || responseValue === true) {
          // Draw a checkmark using simple shapes (X pattern for compatibility)
          // Draw two diagonal lines to form an X/check pattern
          const padding = fieldWidth * 0.2;
          const lineWidth = Math.max(2, fieldHeight * 0.1);
          
          // Draw check-like mark using rectangles
          // First stroke of check (bottom left to middle)
          page.drawLine({
            start: { x: x + padding, y: y + fieldHeight / 2 },
            end: { x: x + fieldWidth / 2, y: y + padding },
            thickness: lineWidth,
            color: rgb(0, 0.5, 0), // Green color
          });
          
          // Second stroke of check (middle to top right)
          page.drawLine({
            start: { x: x + fieldWidth / 2, y: y + padding },
            end: { x: x + fieldWidth - padding, y: y + fieldHeight - padding },
            thickness: lineWidth,
            color: rgb(0, 0.5, 0), // Green color
          });
        }
      }
      // Handle text fields (name, date, email, etc.)
      else if (responseValue && typeof responseValue === 'string') {
        // Draw the text directly without background or border
        const fontSize = Math.min(12, fieldHeight * 0.6);
        page.drawText(responseValue, {
          x: x + 5,
          y: y + (fieldHeight / 2) - (fontSize / 3),
          size: fontSize,
          color: rgb(0, 0, 0),
        });
      }
    }
  }
  
  // Save the modified PDF
  const pdfBytes = await pdfDoc.save();
  return Buffer.from(pdfBytes);
}

/**
 * Get the number of pages in a PDF
 */
async function getPdfPageCount(filePath) {
  try {
    // This would require a PDF parsing library
    // For now, return a default value
    // In production, use pdf-parse or pdf-lib:
    // const pdfParse = require('pdf-parse');
    // const dataBuffer = await fs.readFile(filePath);
    // const data = await pdfParse(dataBuffer);
    // return data.numpages;
    
    return 1; // Default fallback
  } catch (error) {
    console.error('Error getting PDF page count:', error);
    return 1;
  }
}

/**
 * Validate signature field positions
 */
function validateFieldPositions(fields) {
  const errors = [];
  
  for (const field of fields) {
    if (field.x_position < 0 || field.x_position > 100) {
      errors.push(`Invalid x position for field ${field.field_label}`);
    }
    if (field.y_position < 0 || field.y_position > 100) {
      errors.push(`Invalid y position for field ${field.field_label}`);
    }
    if (field.width <= 0 || field.width > 100) {
      errors.push(`Invalid width for field ${field.field_label}`);
    }
    if (field.height <= 0 || field.height > 100) {
      errors.push(`Invalid height for field ${field.field_label}`);
    }
  }
  
  return errors;
}

module.exports = {
  initializeEmailTransporter,
  sendSignatureRequestEmail,
  sendSignatureRequestEmails,
  sendCompletionEmail,
  sendCompletionEmails,
  sendProgressNotificationEmail,
  canPartySign,
  notifyNextSigner,
  generateSignedPDF,
  getPdfPageCount,
  validateFieldPositions
};
