const express = require('express');
const router = express.Router();
const nodemailer = require('nodemailer');
const multer = require('multer');
const path = require('path');
const crypto = require('crypto');
const emailTemplates = require('../utils/emailTemplates');
const dropdowns = require('../utils/dropdownStore');
const db = require('../models/db');

const userModel = require('../models/userModel');

// Multer setup for file uploads (docs)
const upload = multer({
  storage: multer.diskStorage({
    destination: (req, file, cb) => {
      cb(null, path.join(__dirname, '../docs/stxd'));
    },
    filename: (req, file, cb) => {
      cb(null, Date.now() + '-' + file.originalname);
    }
  }),
  limits: { fileSize: 15 * 1024 * 1024 }, // 15 MB
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.jpg', '.jpeg', '.png', '.doc', '.docx'];
    if (allowed.includes(path.extname(file.originalname).toLowerCase())) {
      cb(null, true);
    } else {
      cb(new Error('Invalid file type'));
    }
  }
}).fields([
  { name: 'gedDoc', maxCount: 1 },
  { name: 'govIdDoc', maxCount: 1 }
]);

// Email transporter
const transporter = nodemailer.createTransport({
  host: 'mdts-apps.com',
  port: 465,
  secure: true,
  auth: {
    user: 'noreply@mdts-apps.com',
    pass: 'c@r,5ysPI@&s'
  }
});

const DOC_VERSION = 'v1.0-2025-08-14';
const DOC_TEXT = 'By registering, you agree to the school’s Acceptable Use Policy, Honor Code, and Privacy Notice. You consent to electronic records and provide a digital signature affirming your identity.';

function ensureStr(v) {
  if (typeof v === 'string') return v;
  if (Array.isArray(v)) return String(v[0] ?? '');
  if (v && typeof v === 'object') {
    if (typeof v.password === 'string') return v.password;
    if (typeof v.value === 'string') return v.value;
  }
  return String(v ?? '');
}

router.get('/login', (req, res) => {
  // Check for returnTo in session (set by requireAuth middleware) or query param
  const returnTo = req.session.returnTo || req.query.next;
  const next = ensureStr(returnTo);
  const safeNext = next.startsWith('/') ? next : '';
  res.render('login', { error: null, next: safeNext });
});
router.get('/forgot-password', (req, res) => {
  res.render('forgot_password', { error: null, sent: false });
});

router.post('/forgot-password', async (req, res) => {
  const email = ensureStr(req.body.email).toLowerCase();
  try {
    const user = await userModel.findByEmail(email);
    if (user) {
      const token = crypto.randomBytes(32).toString('hex');
      const expires = Date.now() + 1000 * 60 * 60; // 1 hour
      await userModel.setResetToken(user.username, token, expires);
      const resetLink = `${req.protocol}://${req.get('host')}/reset-password?token=${token}`;
      const { subject, html, text } = emailTemplates.render('passwordResetLink', { resetLink });
      await transporter.sendMail({
        from: 'no-reply@mdts-apps.com',
        to: user.email,
        subject,
        html,
        text
      });
    }
    return res.render('forgot_password', { sent: true, error: null });
  } catch (e) {
    console.error(e);
    return res.status(500).render('forgot_password', { sent: false, error: 'Unable to send reset request' });
  }
});

router.get('/reset-password', async (req, res) => {
  const token = ensureStr(req.query.token);
  if (!token) {
    return res.status(400).render('reset_password', { error: 'Invalid or expired link', success: false, token: null });
  }
  const user = await userModel.findByResetToken(token);
  if (!user) {
    return res.status(400).render('reset_password', { error: 'Invalid or expired link', success: false, token: null });
  }
  return res.render('reset_password', { error: null, success: false, token });
});

router.post('/reset-password', async (req, res) => {
  const token = ensureStr(req.body.token);
  const password = ensureStr(req.body.password);
  const confirm = ensureStr(req.body.confirm);

  if (!token || !password || password !== confirm) {
    return res.status(400).render('reset_password', {
      error: 'Invalid request',
       success: false,
      token
    });
  }

  const user = await userModel.findByResetToken(token);
  if (!user) {
    return res.status(400).render('reset_password', {
      error: 'Invalid or expired link',
      success: false,
      token: null
    });
  }
  await userModel.updatePassword(user.username, password);
  await userModel.clearResetToken(user.id);
  return res.render('reset_password', { error: null, success: true, token: null });
});

router.get('/step2', async (req, res) => {
  const token = ensureStr(req.query.token);
  if (!token) {
    return res.status(400).render('login', { error: 'Invalid or expired link', next: '' });
  }
  const user = await userModel.findByStep2Token(token);
  if (!user) {
    return res.status(400).render('login', { error: 'Invalid or expired link', next: '' });
  }
  req.session.user = {
    id: user.id,
    name: user.name,
    username: user.username,
    firstName: user.firstName,
    lastName: user.lastName
  };
  req.session.role = user.role;
  await userModel.clearStep2Token(user.id);
  return res.redirect('/student/profile');
});

router.post('/login', async (req, res) => {
  const username = ensureStr(req.body.username);
  const password = ensureStr(req.body.password);
  // Check both form input and session returnTo
  const returnTo = req.session.returnTo || req.body.next;
  const redirectTo = ensureStr(returnTo);
  const safeRedirect = redirectTo.startsWith('/') ? redirectTo : '';

  const user = await userModel.findByUsername(username);
  if (!user || !userModel.verifyPassword(user, password)) {
    return res.status(401).render('login-error', { error: 'Invalid username or password', next: safeRedirect });
  }
  if (user.active === false) {
    return res.status(403).render('login-error', { error: 'Account deactivated - Please contact administrator', next: safeRedirect });
  }
  req.session.user = {
    id: user.id,
    name: user.name,
    username: user.username,
    firstName: user.firstName,
    lastName: user.lastName
  };
  req.session.role = user.role;
  req.session.loginAt = Date.now();
  req.session.pageViews = [];
  req.session.lastAction = Date.now();
  
  // Clear returnTo from session after using it
  delete req.session.returnTo;
  
  await userModel.updateProfile(user.id, { lastLogin: new Date().toISOString() });
  
  // If admin user, run the set_user_status_inactive function
  // if (user.role === 'admin') {
  //   try {
  //     await db.query('SELECT set_user_status_inactive()');
  //     console.log('Admin login: set_user_status_inactive() executed successfully');
  //   } catch (err) {
  //     console.error('Error executing set_user_status_inactive() on admin login:', err);
  //   }
  // }
  if (user.profile?.mustChangePassword) {
    return res.redirect('/account?force=1');
  }
  if (user.role === 'student' && (user.status === 'pending' || !user.status) && !safeRedirect) {
    // Get financial information from user profile
    const financialAid = user.profile?.financialAid === 'yes' || false;
    const selfPay = user.profile?.selfPay === true || user.profile?.paymentPlan === 'self' || false;
    return res.render('pending', { user, financialAid, selfPay });
  }
  return res.redirect(safeRedirect || '/dashboard');
});

router.get('/register', async (req, res) => {
  const formData = req.session.preRegData || {};
  delete req.session.preRegData;
  try {
    formData.studentId = await userModel.generateStudentId();
  } catch (e) {
    console.error('Error generating student ID', e);
    formData.studentId = '';
  }
  const dd = dropdowns.getAll();
  res.render('register', {
    error: null,
    docVersion: DOC_VERSION,
    docText: DOC_TEXT,
    formData,
    courses: dd.courses,
    affiliatePrograms: dd.affiliatePrograms
  });
});

// Quick username availability check
router.get('/api/username-available', async (req, res) => {
  try {
    const username = String(req.query.username || '').trim();
    if (!username) return res.json({ available: false, reason: 'empty' });
    const existing = await userModel.findByUsername(username);
    res.json({ available: !existing });
  } catch (e) {
    res.json({ available: false });
  }
});

router.post('/register', (req, res) => {
  upload(req, res, async (err) => {
     const dd = dropdowns.getAll();
    if (err) {
      return res.status(400).render('register', {
        error: err.message,
        docVersion: DOC_VERSION,
        docText: DOC_TEXT,
        formData: req.body,
        courses: dd.courses,
        affiliatePrograms: dd.affiliatePrograms
      });
    }

    try {
      const firstName = ensureStr(req.body.firstName);
      const lastName = ensureStr(req.body.lastName);
      const username = ensureStr(req.body.username) || `${firstName}${lastName}`.replace(/\s+/g, '');
      const email = ensureStr(req.body.email);
      const confirmEmail = ensureStr(req.body.confirmEmail);
      const password = ensureStr(req.body.password);
 const studentId = ensureStr(req.body.studentId) || await userModel.generateStudentId();
      const suffix = ensureStr(req.body.suffix);
      const address = ensureStr(req.body.address);
      const city = ensureStr(req.body.city);
      const state = ensureStr(req.body.state);
      const zip = ensureStr(req.body.zip);
      const course = ensureStr(req.body.course);
      const affiliateProgram = ensureStr(req.body.affiliateProgram);
      const phoneHome = ensureStr(req.body.phoneHome);
      const phoneCell = ensureStr(req.body.phoneCell);
      const phoneWork = ensureStr(req.body.phoneWork);
  let gender = ensureStr(req.body.gender).toLowerCase();
  let race = ensureStr(req.body.race).toLowerCase();
  const allowedGender = ['female','male',''];
  const allowedRace = ['american-indian-alaska-native','asian','black-african-american','native-hawaiian-pacific-islander','white','two-or-more','other','prefer-not-to-say',''];
  if (!allowedGender.includes(gender)) gender = '';
  if (!allowedRace.includes(race)) race = '';
   
      const selfPay = affiliateProgram === 'Self Pay';

      const financialAid = ensureStr(req.body.financialAid);
      const referralName = ensureStr(req.body.referralName);
      const referralEmail = ensureStr(req.body.referralEmail);


      if (email !== confirmEmail) {
        return res.status(400).render('register', {
          error: 'Emails do not match.',
          docVersion: DOC_VERSION,
          docText: DOC_TEXT,
          formData: req.body,
          courses: dd.courses,
          affiliatePrograms: dd.affiliatePrograms
        });
      }

      // Enforce unique username
      const existingUser = await userModel.findByUsername(username.trim());
      if (existingUser) {
        return res.status(400).render('register', {
          error: 'Username already exists. Please choose another.',
          docVersion: DOC_VERSION,
          docText: DOC_TEXT,
          formData: req.body,
          courses: dd.courses,
          affiliatePrograms: dd.affiliatePrograms
        });
      }

        const user = await userModel.createStudent({
        username,
        firstName,
        lastName,
          suffix,
        address,
        city,
        state,
        zip,
        course,
        affiliateProgram,
        phones: { home: phoneHome, cell: phoneCell, work: phoneWork },
        ssn: '',
        emergencyContact: {},
        grievanceAck: false,
        name: `${firstName} ${lastName}`.trim(),
        email,
        password,
        studentId,
        gender,
        race,
        financialAid: financialAid === 'yes',
        referralName,
        referralEmail
      });
      // Update lead interest for registration (80%)
      try {
        const leadModel = require('../models/leadModel');
        const phoneAny = phoneCell || phoneHome || phoneWork || '';
        await leadModel.upsertByEmail({ name: `${firstName} ${lastName}`.trim(), email: email.trim(), phone: phoneAny.trim(), interestPercent: 80, source: 'register' });
      } catch (e) { console.error('Lead upsert (register) failed', e); }
     if (req.files) {
        const collected = [];
        if (Array.isArray(req.files.gedDoc)) collected.push(...req.files.gedDoc);
        if (Array.isArray(req.files.govIdDoc)) collected.push(...req.files.govIdDoc);
        if (collected.length) {
          const uploads = collected.map(f => ({
            originalName: f.originalname,
            mimeType: f.mimetype,
            size: f.size,
            url: `/docs/stxd/${f.filename}`
          }));
          await userModel.addUploads(user.id, uploads);
        }
      }

      const { subject, html, text } = emailTemplates.render('registrationSubmitted', { firstName, username });
      await transporter.sendMail({
        from: 'no-reply@mdts-apps.com',
        to: email,
        subject,
        html,
        text
      });

      return res.render('pending', { user: { firstName, lastName, name: `${firstName} ${lastName}` }, financialAid: financialAid === 'yes', selfPay });    } catch (e) {
      console.error(e);
      return res.status(500).render('register', {
        error: 'Registration failed. Please try again.',
        docVersion: DOC_VERSION,
        docText: DOC_TEXT,
        formData: req.body,
        courses: dd.courses,
        affiliatePrograms: dd.affiliatePrograms
      });
    }
  });
});

router.get('/logout', async (req, res) => {
  if (req.session && req.session.user) {
    try {
      const user = await userModel.findById(req.session.user.id);
      const pages = Array.isArray(req.session.pageViews) ? req.session.pageViews : [];
      const timeSpent = req.session.timeSpent || 0;
      const updates = {
        pagesVisited: Array.isArray(user.profile?.pagesVisited)
          ? user.profile.pagesVisited.concat(pages)
          : pages,
        timeSpent: (user.profile?.timeSpent || 0) + timeSpent
      };
      await userModel.updateProfile(user.id, updates);
    } catch (e) {
      console.error('logout audit save failed', e);
    }
  }
  req.session.destroy(() => res.redirect('/login'));
});

module.exports = router;
