const pool = require('./db');

/**
 * Save or update test progress for a student
 */
async function saveProgress(studentId, classId, testId, answers, timeRemaining) {
  const [rows] = await pool.query(
    `INSERT INTO test_progress (student_id, class_id, test_id, answers, time_remaining, completed)
     VALUES (?, ?, ?, ?, ?, FALSE)
     ON DUPLICATE KEY UPDATE 
       answers = VALUES(answers),
       time_remaining = VALUES(time_remaining),
       last_saved_at = CURRENT_TIMESTAMP`,
    [studentId, classId, testId, JSON.stringify(answers), timeRemaining]
  );
  return rows;
}

/**
 * Get in-progress test for a student
 */
async function getProgress(studentId, classId, testId) {
  const [rows] = await pool.query(
    `SELECT * FROM test_progress 
     WHERE student_id = ? AND class_id = ? AND test_id = ? AND completed = FALSE
     LIMIT 1`,
    [studentId, classId, testId]
  );
  if (rows.length > 0) {
    const progress = rows[0];
    // Parse JSON answers
    if (typeof progress.answers === 'string') {
      progress.answers = JSON.parse(progress.answers);
    }
    return progress;
  }
  return null;
}

/**
 * Get all incomplete tests for a student
 */
async function getIncompleteTests(studentId) {
  const [rows] = await pool.query(
    `SELECT * FROM test_progress 
     WHERE student_id = ? AND completed = FALSE
     ORDER BY last_saved_at DESC`,
    [studentId]
  );
  return rows.map(row => {
    if (typeof row.answers === 'string') {
      row.answers = JSON.parse(row.answers);
    }
    return row;
  });
}

/**
 * Mark test progress as completed
 */
async function markCompleted(studentId, classId, testId) {
  const [rows] = await pool.query(
    `UPDATE test_progress 
     SET completed = TRUE 
     WHERE student_id = ? AND class_id = ? AND test_id = ?`,
    [studentId, classId, testId]
  );
  return rows;
}

/**
 * Delete/reset test progress (for "Try Again" or "Start Over")
 */
async function deleteProgress(studentId, classId, testId) {
  const [rows] = await pool.query(
    `DELETE FROM test_progress 
     WHERE student_id = ? AND class_id = ? AND test_id = ?`,
    [studentId, classId, testId]
  );
  return rows;
}

module.exports = {
  saveProgress,
  getProgress,
  getIncompleteTests,
  markCompleted,
  deleteProgress
};
