const db = require('./db');

// Ensure tables exist
let tablesInitialized = null;

async function ensureTables() {
  if (tablesInitialized !== null) return tablesInitialized;
  try {
    // Teacher availability table
    await db.query(`
      CREATE TABLE IF NOT EXISTS mdtslms_teacher_availability (
        id INT AUTO_INCREMENT PRIMARY KEY,
        teacherId INT NOT NULL,
        dayOfWeek TINYINT NOT NULL COMMENT '0=Sunday, 6=Saturday',
        startTime TIME NOT NULL,
        endTime TIME NOT NULL,
        isRecurring BOOLEAN DEFAULT TRUE,
        specificDate DATE NULL COMMENT 'For non-recurring availability',
        notes TEXT NULL,
        createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
        updatedAt DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_teacher (teacherId),
        INDEX idx_day (dayOfWeek)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    `);

    // Scheduled sessions table
    await db.query(`
      CREATE TABLE IF NOT EXISTS mdtslms_scheduled_sessions (
        id INT AUTO_INCREMENT PRIMARY KEY,
        cohortName VARCHAR(255) NOT NULL COMMENT 'WIOA, Month name, or section number',
        courseId INT NULL,
        courseName VARCHAR(255) NOT NULL,
        sessionDate DATE NOT NULL,
        startTime TIME NOT NULL,
        endTime TIME NOT NULL,
        teacherId INT NOT NULL,
        locationOnline BOOLEAN DEFAULT TRUE,
        locationPhysical VARCHAR(100) NULL COMMENT 'Newport News, Fredericksburg, Virginia Beach, Maryland',
        colorCode VARCHAR(20) DEFAULT '#1e90ff' COMMENT 'Hex color for visualization',
        programTrack VARCHAR(100) NULL COMMENT 'e.g., Cyber II Track',
        notes TEXT NULL,
        createdBy INT NULL,
        createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
        updatedAt DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_date (sessionDate),
        INDEX idx_teacher (teacherId),
        INDEX idx_cohort (cohortName),
        INDEX idx_course (courseId)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    `);

    // Program tracks table
    await db.query(`
      CREATE TABLE IF NOT EXISTS mdtslms_program_tracks (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        colorCode VARCHAR(20) DEFAULT '#1e90ff',
        description TEXT NULL,
        isActive BOOLEAN DEFAULT TRUE,
        createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
        UNIQUE KEY uk_name (name)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    `);

    // Locations table
    await db.query(`
      CREATE TABLE IF NOT EXISTS mdtslms_locations (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        address TEXT NULL,
        isOnline BOOLEAN DEFAULT FALSE,
        isActive BOOLEAN DEFAULT TRUE,
        createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
        UNIQUE KEY uk_name (name)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    `);

    // Insert default locations if not present
    const defaultLocations = [
      { name: 'Online', isOnline: true },
      { name: 'Newport News', isOnline: false },
      { name: 'Fredericksburg', isOnline: false },
      { name: 'Virginia Beach', isOnline: false },
      { name: 'Maryland', isOnline: false }
    ];
    for (const loc of defaultLocations) {
      await db.query(
        'INSERT IGNORE INTO mdtslms_locations (name, isOnline) VALUES (?, ?)',
        [loc.name, loc.isOnline]
      );
    }

    // Teacher time-off / exceptions table (holidays, vacation, sick days)
    await db.query(`
      CREATE TABLE IF NOT EXISTS mdtslms_teacher_timeoff (
        id INT AUTO_INCREMENT PRIMARY KEY,
        teacherId INT NULL COMMENT 'NULL means applies to all teachers (school-wide holiday)',
        startDate DATE NOT NULL,
        endDate DATE NOT NULL,
        timeOffType ENUM('holiday', 'vacation', 'sick', 'personal', 'other') DEFAULT 'other',
        title VARCHAR(255) NOT NULL,
        notes TEXT NULL,
        isSchoolWide BOOLEAN DEFAULT FALSE COMMENT 'Applies to all teachers',
        createdBy INT NULL,
        createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_teacher (teacherId),
        INDEX idx_dates (startDate, endDate),
        INDEX idx_schoolwide (isSchoolWide)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    `);

    // Teacher weekly override table (for specific weeks with different schedules)
    await db.query(`
      CREATE TABLE IF NOT EXISTS mdtslms_teacher_week_override (
        id INT AUTO_INCREMENT PRIMARY KEY,
        teacherId INT NOT NULL,
        weekStartDate DATE NOT NULL COMMENT 'Monday of the week',
        dayOfWeek TINYINT NOT NULL COMMENT '0=Sunday, 6=Saturday',
        startTime TIME NULL COMMENT 'NULL means unavailable that day',
        endTime TIME NULL,
        isAvailable BOOLEAN DEFAULT TRUE COMMENT 'FALSE means blocked off',
        notes TEXT NULL,
        createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_teacher (teacherId),
        INDEX idx_week (weekStartDate),
        UNIQUE KEY uk_teacher_week_day (teacherId, weekStartDate, dayOfWeek)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    `);

    // School holidays table (pre-defined holidays)
    await db.query(`
      CREATE TABLE IF NOT EXISTS mdtslms_school_holidays (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        date DATE NOT NULL,
        isRecurringYearly BOOLEAN DEFAULT FALSE,
        createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
        UNIQUE KEY uk_date (date)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    `);

    // Insert US Federal holidays for 2025 and 2026
    // These are the actual observed dates for federal holidays
    const defaultHolidays = [
      // 2025 US Federal Holidays
      { name: "New Year's Day", date: "2025-01-01" },
      { name: "Martin Luther King Jr. Day", date: "2025-01-20" }, // 3rd Monday of January
      { name: "Presidents' Day", date: "2025-02-17" }, // 3rd Monday of February
      { name: "Memorial Day", date: "2025-05-26" }, // Last Monday of May
      { name: "Juneteenth", date: "2025-06-19" },
      { name: "Independence Day", date: "2025-07-04" },
      { name: "Labor Day", date: "2025-09-01" }, // 1st Monday of September
      { name: "Columbus Day", date: "2025-10-13" }, // 2nd Monday of October
      { name: "Veterans Day", date: "2025-11-11" },
      { name: "Thanksgiving Day", date: "2025-11-27" }, // 4th Thursday of November
      { name: "Day After Thanksgiving", date: "2025-11-28" },
      { name: "Christmas Eve", date: "2025-12-24" },
      { name: "Christmas Day", date: "2025-12-25" },
      { name: "New Year's Eve", date: "2025-12-31" },
      // 2026 US Federal Holidays
      { name: "New Year's Day", date: "2026-01-01" },
      { name: "Martin Luther King Jr. Day", date: "2026-01-19" }, // 3rd Monday of January
      { name: "Presidents' Day", date: "2026-02-16" }, // 3rd Monday of February
      { name: "Memorial Day", date: "2026-05-25" }, // Last Monday of May
      { name: "Juneteenth", date: "2026-06-19" },
      { name: "Independence Day (Observed)", date: "2026-07-03" }, // July 4 is Saturday, observed Friday
      { name: "Labor Day", date: "2026-09-07" }, // 1st Monday of September
      { name: "Columbus Day", date: "2026-10-12" }, // 2nd Monday of October
      { name: "Veterans Day", date: "2026-11-11" },
      { name: "Thanksgiving Day", date: "2026-11-26" }, // 4th Thursday of November
      { name: "Day After Thanksgiving", date: "2026-11-27" },
      { name: "Christmas Eve", date: "2026-12-24" },
      { name: "Christmas Day", date: "2026-12-25" },
      { name: "New Year's Eve", date: "2026-12-31" },
    ];
    for (const h of defaultHolidays) {
      await db.query(
        'INSERT IGNORE INTO mdtslms_school_holidays (name, date, isRecurringYearly) VALUES (?, ?, FALSE)',
        [h.name, h.date]
      );
    }

    tablesInitialized = true;
  } catch (e) {
    console.error('Schedule tables initialization error:', e.message || e);
    tablesInitialized = false;
  }
  return tablesInitialized;
}

// ============ TEACHER AVAILABILITY ============

async function getTeacherAvailability(teacherId) {
  await ensureTables();
  const [rows] = await db.query(
    'SELECT * FROM mdtslms_teacher_availability WHERE teacherId = ? ORDER BY dayOfWeek, startTime',
    [teacherId]
  );
  return rows;
}

async function getAllTeacherAvailability() {
  await ensureTables();
  const [rows] = await db.query(
    'SELECT * FROM mdtslms_teacher_availability ORDER BY teacherId, dayOfWeek, startTime'
  );
  return rows;
}

async function setTeacherAvailability(teacherId, availabilitySlots) {
  await ensureTables();
  // Delete existing recurring availability and re-insert
  await db.query(
    'DELETE FROM mdtslms_teacher_availability WHERE teacherId = ? AND isRecurring = TRUE',
    [teacherId]
  );
  for (const slot of availabilitySlots) {
    await db.query(
      `INSERT INTO mdtslms_teacher_availability 
       (teacherId, dayOfWeek, startTime, endTime, isRecurring, notes)
       VALUES (?, ?, ?, ?, TRUE, ?)`,
      [teacherId, slot.dayOfWeek, slot.startTime, slot.endTime, slot.notes || null]
    );
  }
  return getTeacherAvailability(teacherId);
}

async function addTeacherAvailabilitySlot(teacherId, slot) {
  await ensureTables();
  const [result] = await db.query(
    `INSERT INTO mdtslms_teacher_availability 
     (teacherId, dayOfWeek, startTime, endTime, isRecurring, specificDate, notes)
     VALUES (?, ?, ?, ?, ?, ?, ?)`,
    [
      teacherId,
      slot.dayOfWeek,
      slot.startTime,
      slot.endTime,
      slot.isRecurring !== false,
      slot.specificDate || null,
      slot.notes || null
    ]
  );
  return result.insertId;
}

async function deleteTeacherAvailabilitySlot(slotId) {
  await ensureTables();
  await db.query('DELETE FROM mdtslms_teacher_availability WHERE id = ?', [slotId]);
}

// ============ SCHEDULED SESSIONS ============

async function getAllScheduledSessions() {
  await ensureTables();
  const [rows] = await db.query(
    'SELECT * FROM mdtslms_scheduled_sessions ORDER BY sessionDate, startTime'
  );
  return rows;
}

async function getScheduledSessionsInRange(startDate, endDate) {
  await ensureTables();
  const [rows] = await db.query(
    'SELECT * FROM mdtslms_scheduled_sessions WHERE sessionDate BETWEEN ? AND ? ORDER BY sessionDate, startTime',
    [startDate, endDate]
  );
  return rows;
}

async function getScheduledSessionsByTeacher(teacherId, startDate = null, endDate = null) {
  await ensureTables();
  let query = 'SELECT * FROM mdtslms_scheduled_sessions WHERE teacherId = ?';
  const params = [teacherId];
  if (startDate && endDate) {
    query += ' AND sessionDate BETWEEN ? AND ?';
    params.push(startDate, endDate);
  }
  query += ' ORDER BY sessionDate, startTime';
  const [rows] = await db.query(query, params);
  return rows;
}

async function getScheduledSessionsByCohort(cohortName, startDate = null, endDate = null) {
  await ensureTables();
  let query = 'SELECT * FROM mdtslms_scheduled_sessions WHERE cohortName = ?';
  const params = [cohortName];
  if (startDate && endDate) {
    query += ' AND sessionDate BETWEEN ? AND ?';
    params.push(startDate, endDate);
  }
  query += ' ORDER BY sessionDate, startTime';
  const [rows] = await db.query(query, params);
  return rows;
}

async function getScheduledSessionsByCourse(courseId, startDate = null, endDate = null) {
  await ensureTables();
  let query = 'SELECT * FROM mdtslms_scheduled_sessions WHERE courseId = ?';
  const params = [courseId];
  if (startDate && endDate) {
    query += ' AND sessionDate BETWEEN ? AND ?';
    params.push(startDate, endDate);
  }
  query += ' ORDER BY sessionDate, startTime';
  const [rows] = await db.query(query, params);
  return rows;
}

async function createScheduledSession(session) {
  await ensureTables();
  const [result] = await db.query(
    `INSERT INTO mdtslms_scheduled_sessions 
     (cohortName, courseId, courseName, sessionDate, startTime, endTime, 
      teacherId, locationOnline, locationPhysical, colorCode, programTrack, notes, createdBy)
     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
    [
      session.cohortName,
      session.courseId || null,
      session.courseName,
      session.sessionDate,
      session.startTime,
      session.endTime,
      session.teacherId,
      session.locationOnline !== false,
      session.locationPhysical || null,
      session.colorCode || '#1e90ff',
      session.programTrack || null,
      session.notes || null,
      session.createdBy || null
    ]
  );
  return result.insertId;
}

async function updateScheduledSession(id, session) {
  await ensureTables();
  await db.query(
    `UPDATE mdtslms_scheduled_sessions SET
     cohortName = ?, courseId = ?, courseName = ?, sessionDate = ?, startTime = ?, endTime = ?,
     teacherId = ?, locationOnline = ?, locationPhysical = ?, colorCode = ?, programTrack = ?, notes = ?
     WHERE id = ?`,
    [
      session.cohortName,
      session.courseId || null,
      session.courseName,
      session.sessionDate,
      session.startTime,
      session.endTime,
      session.teacherId,
      session.locationOnline !== false,
      session.locationPhysical || null,
      session.colorCode || '#1e90ff',
      session.programTrack || null,
      session.notes || null,
      id
    ]
  );
}

async function deleteScheduledSession(id) {
  await ensureTables();
  await db.query('DELETE FROM mdtslms_scheduled_sessions WHERE id = ?', [id]);
}

async function bulkCreateScheduledSessions(sessions) {
  await ensureTables();
  const ids = [];
  for (const session of sessions) {
    const id = await createScheduledSession(session);
    ids.push(id);
  }
  return ids;
}

// ============ CONFLICT DETECTION ============

async function checkTeacherConflicts(teacherId, sessionDate, startTime, endTime, excludeSessionId = null) {
  await ensureTables();
  let query = `
    SELECT * FROM mdtslms_scheduled_sessions 
    WHERE teacherId = ? 
    AND sessionDate = ?
    AND (
      (startTime < ? AND endTime > ?) OR
      (startTime >= ? AND startTime < ?) OR
      (endTime > ? AND endTime <= ?)
    )
  `;
  const params = [teacherId, sessionDate, endTime, startTime, startTime, endTime, startTime, endTime];
  
  if (excludeSessionId) {
    query += ' AND id != ?';
    params.push(excludeSessionId);
  }
  
  const [rows] = await db.query(query, params);
  return rows;
}

async function isTeacherAvailableOnDate(teacherId, sessionDate, startTime, endTime) {
  await ensureTables();
  const date = new Date(sessionDate);
  const dayOfWeek = date.getDay();
  
  // Check if teacher has availability set for this day of week
  const [availability] = await db.query(
    `SELECT * FROM mdtslms_teacher_availability 
     WHERE teacherId = ? 
     AND (
       (isRecurring = TRUE AND dayOfWeek = ?) OR
       (isRecurring = FALSE AND specificDate = ?)
     )
     AND startTime <= ? AND endTime >= ?`,
    [teacherId, dayOfWeek, sessionDate, startTime, endTime]
  );
  
  return availability.length > 0;
}

// ============ PROGRAM TRACKS ============

async function getAllProgramTracks() {
  await ensureTables();
  const [rows] = await db.query(
    'SELECT * FROM mdtslms_program_tracks WHERE isActive = TRUE ORDER BY name'
  );
  return rows;
}

async function createProgramTrack(track) {
  await ensureTables();
  const [result] = await db.query(
    'INSERT INTO mdtslms_program_tracks (name, colorCode, description) VALUES (?, ?, ?)',
    [track.name, track.colorCode || '#1e90ff', track.description || null]
  );
  return result.insertId;
}

async function updateProgramTrack(id, track) {
  await ensureTables();
  await db.query(
    'UPDATE mdtslms_program_tracks SET name = ?, colorCode = ?, description = ? WHERE id = ?',
    [track.name, track.colorCode || '#1e90ff', track.description || null, id]
  );
}

async function deleteProgramTrack(id) {
  await ensureTables();
  await db.query('UPDATE mdtslms_program_tracks SET isActive = FALSE WHERE id = ?', [id]);
}

// ============ LOCATIONS ============

async function getAllLocations() {
  await ensureTables();
  const [rows] = await db.query(
    'SELECT * FROM mdtslms_locations WHERE isActive = TRUE ORDER BY isOnline DESC, name'
  );
  return rows;
}

async function createLocation(location) {
  await ensureTables();
  const [result] = await db.query(
    'INSERT INTO mdtslms_locations (name, address, isOnline) VALUES (?, ?, ?)',
    [location.name, location.address || null, location.isOnline || false]
  );
  return result.insertId;
}

// ============ UNIQUE VALUES FOR FILTERS ============

async function getUniqueCohorts() {
  await ensureTables();
  const [rows] = await db.query(
    'SELECT DISTINCT cohortName FROM mdtslms_scheduled_sessions ORDER BY cohortName'
  );
  return rows.map(r => r.cohortName);
}

async function getUniqueCourses() {
  await ensureTables();
  const [rows] = await db.query(
    'SELECT DISTINCT courseId, courseName FROM mdtslms_scheduled_sessions ORDER BY courseName'
  );
  return rows;
}

// ============ TEACHER TIME-OFF / EXCEPTIONS ============

async function getTeacherTimeOff(teacherId, startDate = null, endDate = null) {
  await ensureTables();
  let query = `
    SELECT * FROM mdtslms_teacher_timeoff 
    WHERE (teacherId = ? OR isSchoolWide = TRUE)
  `;
  const params = [teacherId];
  
  if (startDate && endDate) {
    query += ' AND startDate <= ? AND endDate >= ?';
    params.push(endDate, startDate);
  }
  
  query += ' ORDER BY startDate';
  const [rows] = await db.query(query, params);
  return rows;
}

async function getAllTimeOff(startDate = null, endDate = null) {
  await ensureTables();
  let query = 'SELECT * FROM mdtslms_teacher_timeoff WHERE 1=1';
  const params = [];
  
  if (startDate && endDate) {
    query += ' AND startDate <= ? AND endDate >= ?';
    params.push(endDate, startDate);
  }
  
  query += ' ORDER BY startDate';
  const [rows] = await db.query(query, params);
  return rows;
}

async function addTimeOff(timeOff) {
  await ensureTables();
  const [result] = await db.query(
    `INSERT INTO mdtslms_teacher_timeoff 
     (teacherId, startDate, endDate, timeOffType, title, notes, isSchoolWide, createdBy)
     VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
    [
      timeOff.isSchoolWide ? null : timeOff.teacherId,
      timeOff.startDate,
      timeOff.endDate,
      timeOff.timeOffType || 'other',
      timeOff.title,
      timeOff.notes || null,
      timeOff.isSchoolWide || false,
      timeOff.createdBy || null
    ]
  );
  return result.insertId;
}

async function updateTimeOff(id, timeOff) {
  await ensureTables();
  await db.query(
    `UPDATE mdtslms_teacher_timeoff SET
     teacherId = ?, startDate = ?, endDate = ?, timeOffType = ?, title = ?, notes = ?, isSchoolWide = ?
     WHERE id = ?`,
    [
      timeOff.isSchoolWide ? null : timeOff.teacherId,
      timeOff.startDate,
      timeOff.endDate,
      timeOff.timeOffType || 'other',
      timeOff.title,
      timeOff.notes || null,
      timeOff.isSchoolWide || false,
      id
    ]
  );
}

async function deleteTimeOff(id) {
  await ensureTables();
  await db.query('DELETE FROM mdtslms_teacher_timeoff WHERE id = ?', [id]);
}

// ============ WEEKLY OVERRIDES ============

function getWeekStart(date) {
  const d = new Date(date);
  const day = d.getDay();
  const diff = d.getDate() - day + (day === 0 ? -6 : 1); // Adjust for Monday start
  return new Date(d.setDate(diff)).toISOString().split('T')[0];
}

async function getWeekOverrides(teacherId, weekStartDate) {
  await ensureTables();
  const [rows] = await db.query(
    'SELECT * FROM mdtslms_teacher_week_override WHERE teacherId = ? AND weekStartDate = ? ORDER BY dayOfWeek',
    [teacherId, weekStartDate]
  );
  return rows;
}

async function getAllWeekOverridesForTeacher(teacherId) {
  await ensureTables();
  const [rows] = await db.query(
    'SELECT * FROM mdtslms_teacher_week_override WHERE teacherId = ? ORDER BY weekStartDate, dayOfWeek',
    [teacherId]
  );
  return rows;
}

async function setWeekOverride(teacherId, weekStartDate, dayOfWeek, override) {
  await ensureTables();
  await db.query(
    `INSERT INTO mdtslms_teacher_week_override 
     (teacherId, weekStartDate, dayOfWeek, startTime, endTime, isAvailable, notes)
     VALUES (?, ?, ?, ?, ?, ?, ?)
     ON DUPLICATE KEY UPDATE 
     startTime = VALUES(startTime), endTime = VALUES(endTime), 
     isAvailable = VALUES(isAvailable), notes = VALUES(notes)`,
    [
      teacherId,
      weekStartDate,
      dayOfWeek,
      override.isAvailable ? override.startTime : null,
      override.isAvailable ? override.endTime : null,
      override.isAvailable !== false,
      override.notes || null
    ]
  );
}

async function deleteWeekOverride(id) {
  await ensureTables();
  await db.query('DELETE FROM mdtslms_teacher_week_override WHERE id = ?', [id]);
}

async function deleteWeekOverrides(teacherId, weekStartDate) {
  await ensureTables();
  await db.query(
    'DELETE FROM mdtslms_teacher_week_override WHERE teacherId = ? AND weekStartDate = ?',
    [teacherId, weekStartDate]
  );
}

async function copyWeekToOverride(teacherId, weekStartDate) {
  await ensureTables();
  // Get the base recurring availability
  const availability = await getTeacherAvailability(teacherId);
  const recurringSlots = availability.filter(a => a.isRecurring);
  
  // Create overrides for each day based on recurring schedule
  for (const slot of recurringSlots) {
    await setWeekOverride(teacherId, weekStartDate, slot.dayOfWeek, {
      startTime: slot.startTime,
      endTime: slot.endTime,
      isAvailable: true,
      notes: 'Copied from regular schedule'
    });
  }
  
  return getWeekOverrides(teacherId, weekStartDate);
}

// ============ SCHOOL HOLIDAYS ============

async function getSchoolHolidays(year = null) {
  await ensureTables();
  let query = 'SELECT * FROM mdtslms_school_holidays';
  const params = [];
  
  if (year) {
    query += ' WHERE YEAR(date) = ?';
    params.push(year);
  }
  
  query += ' ORDER BY date';
  const [rows] = await db.query(query, params);
  return rows;
}

async function addSchoolHoliday(holiday) {
  await ensureTables();
  const [result] = await db.query(
    'INSERT INTO mdtslms_school_holidays (name, date, isRecurringYearly) VALUES (?, ?, ?)',
    [holiday.name, holiday.date, holiday.isRecurringYearly || false]
  );
  return result.insertId;
}

async function updateSchoolHoliday(id, holiday) {
  await ensureTables();
  await db.query(
    'UPDATE mdtslms_school_holidays SET name = ?, date = ?, isRecurringYearly = ? WHERE id = ?',
    [holiday.name, holiday.date, holiday.isRecurringYearly || false, id]
  );
}

async function deleteSchoolHoliday(id) {
  await ensureTables();
  await db.query('DELETE FROM mdtslms_school_holidays WHERE id = ?', [id]);
}

// ============ EFFECTIVE AVAILABILITY (combines all sources) ============

async function getEffectiveAvailability(teacherId, date) {
  await ensureTables();
  const dateStr = typeof date === 'string' ? date : date.toISOString().split('T')[0];
  const dayOfWeek = new Date(dateStr).getDay();
  const weekStart = getWeekStart(dateStr);
  
  // 1. Check for school holidays
  const [holidays] = await db.query(
    'SELECT * FROM mdtslms_school_holidays WHERE date = ?',
    [dateStr]
  );
  if (holidays.length > 0) {
    return { available: false, reason: 'holiday', holiday: holidays[0] };
  }
  
  // 2. Check for time-off (vacation, sick, etc.)
  const [timeOff] = await db.query(
    `SELECT * FROM mdtslms_teacher_timeoff 
     WHERE (teacherId = ? OR isSchoolWide = TRUE)
     AND startDate <= ? AND endDate >= ?`,
    [teacherId, dateStr, dateStr]
  );
  if (timeOff.length > 0) {
    return { available: false, reason: 'timeoff', timeOff: timeOff[0] };
  }
  
  // 3. Check for week-specific override
  const [overrides] = await db.query(
    'SELECT * FROM mdtslms_teacher_week_override WHERE teacherId = ? AND weekStartDate = ? AND dayOfWeek = ?',
    [teacherId, weekStart, dayOfWeek]
  );
  if (overrides.length > 0) {
    const override = overrides[0];
    if (!override.isAvailable) {
      return { available: false, reason: 'override', override };
    }
    return {
      available: true,
      startTime: override.startTime,
      endTime: override.endTime,
      source: 'override',
      notes: override.notes
    };
  }
  
  // 4. Fall back to recurring availability
  const [recurring] = await db.query(
    'SELECT * FROM mdtslms_teacher_availability WHERE teacherId = ? AND dayOfWeek = ? AND isRecurring = TRUE',
    [teacherId, dayOfWeek]
  );
  if (recurring.length > 0) {
    return {
      available: true,
      slots: recurring,
      source: 'recurring'
    };
  }
  
  return { available: false, reason: 'no-schedule' };
}

// ============ CSV IMPORT ============

/**
 * Parse and import schedule data from CSV
 * Expected CSV structure based on the Cybersecurity II Track Schedule format:
 * - Row 1: Track name header
 * - Row 2: Course names in columns (HELPDESK, FUNDAMENTALS, STIGS, etc.)
 * - Row 3: Time slots for each course
 * - Row 4+: Schedule data with Day, Date, and instructor assignments in each column
 * 
 * @param {Array} parsedData - 2D array of CSV rows
 * @param {Object} options - Import options
 * @param {Object} options.teacherMapping - Map of instructor names to teacher IDs
 * @param {Object} options.courseConfig - Map of course column names to {colorCode, programTrack}
 * @param {String} options.defaultTrack - Default program track name
 * @param {Number} options.createdBy - User ID creating the import
 * @param {Boolean} options.skipConflicts - Whether to skip conflicting sessions
 * @returns {Object} - { created: [], skipped: [], errors: [] }
 */
async function importScheduleFromCSV(parsedData, options = {}) {
  await ensureTables();
  
  const results = {
    created: [],
    skipped: [],
    errors: [],
    stats: {
      totalRows: 0,
      sessionsFound: 0,
      sessionsCreated: 0,
      conflictsSkipped: 0,
      holidaysSkipped: 0
    }
  };
  
  const {
    teacherMapping = {},
    courseConfig = {},
    defaultTrack = 'Cyber II Track',
    createdBy = null,
    skipConflicts = true,
    year = new Date().getFullYear()
  } = options;
  
  // Default course configurations (column index -> config)
  const defaultCourseConfig = {
    'HELPDESK': { colorCode: '#1e90ff', times: '2-10PM' },
    'FUNDAMENTALS': { colorCode: '#28a745', times: '2-10PM' },
    'STIGS': { colorCode: '#fd7e14', times: '2-10PM' },
    'SECURITY+': { colorCode: '#6f42c1', times: '6-10PM' },
    'SecurityX-CASP': { colorCode: '#dc3545', times: '6-10PM' },
    'CEH': { colorCode: '#20c997', times: '6-10PM' },
    'RMF': { colorCode: '#e83e8c', times: '6-9PM' },
    'Junior Network': { colorCode: '#17a2b8', times: '' },
    'CCNA': { colorCode: '#ffc107', times: '' },
    'Internship': { colorCode: '#6c757d', times: '6-9PM' }
  };
  
  // Merge with provided config
  const mergedCourseConfig = { ...defaultCourseConfig, ...courseConfig };
  
  // Find the header row with course names (usually row 2 in the CSV)
  let headerRowIndex = -1;
  let courseColumns = {};
  
  for (let i = 0; i < Math.min(5, parsedData.length); i++) {
    const row = parsedData[i];
    if (row && row.some(cell => cell && (cell.includes('HELPDESK') || cell.includes('FUNDAMENTALS') || cell.includes('Security')))) {
      headerRowIndex = i;
      // Map column indices to course names
      row.forEach((cell, colIdx) => {
        const cellText = (cell || '').trim().toUpperCase();
        Object.keys(mergedCourseConfig).forEach(courseName => {
          if (cellText.includes(courseName.toUpperCase()) || cellText === courseName.toUpperCase()) {
            courseColumns[colIdx] = courseName;
          }
        });
      });
      break;
    }
  }
  
  // Helper to parse date from row (handles formats like "9/22", "MON 9/22", etc.)
  function parseDate(dayCell, dateCell, currentYear) {
    let dateStr = dateCell || dayCell || '';
    
    // Extract date part (M/D or MM/DD format)
    const dateMatch = dateStr.match(/(\d{1,2})\/(\d{1,2})/);
    if (!dateMatch) return null;
    
    const month = parseInt(dateMatch[1], 10);
    const day = parseInt(dateMatch[2], 10);
    
    if (month < 1 || month > 12 || day < 1 || day > 31) return null;
    
    // Determine year based on month (school year logic)
    let sessionYear = currentYear;
    // If month is before August and we're looking at a schedule that starts in fall,
    // assume it's the next year
    if (month < 8) {
      sessionYear = currentYear + 1;
    }
    
    const date = new Date(sessionYear, month - 1, day);
    return date.toISOString().split('T')[0];
  }
  
  // Helper to parse time range
  function parseTimeRange(timeStr) {
    if (!timeStr) return { startTime: '09:00', endTime: '17:00' };
    
    const normalized = timeStr.toLowerCase().replace(/\s/g, '');
    
    // Match patterns like "2-10PM", "6:00pm-10:00pm", "9am-11am"
    const timeMatch = normalized.match(/(\d{1,2}):?(\d{2})?\s*(am|pm)?[-–to]+(\d{1,2}):?(\d{2})?\s*(am|pm)?/i);
    
    if (timeMatch) {
      let startHour = parseInt(timeMatch[1], 10);
      const startMin = timeMatch[2] ? parseInt(timeMatch[2], 10) : 0;
      const startAmPm = timeMatch[3];
      let endHour = parseInt(timeMatch[4], 10);
      const endMin = timeMatch[5] ? parseInt(timeMatch[5], 10) : 0;
      const endAmPm = timeMatch[6] || startAmPm; // Use end ampm or inherit from start
      
      // Convert to 24h format
      if (endAmPm && endAmPm.toLowerCase() === 'pm' && endHour < 12) {
        endHour += 12;
      }
      if (startAmPm && startAmPm.toLowerCase() === 'pm' && startHour < 12) {
        startHour += 12;
      }
      // If no AM/PM specified but end hour is less than start, assume PM
      if (!startAmPm && !endAmPm) {
        if (endHour < startHour) {
          endHour += 12;
        }
        if (startHour < 12 && endHour >= 12) {
          startHour += 12; // Both are PM
        }
      }
      
      return {
        startTime: `${startHour.toString().padStart(2, '0')}:${startMin.toString().padStart(2, '0')}`,
        endTime: `${endHour.toString().padStart(2, '0')}:${endMin.toString().padStart(2, '0')}`
      };
    }
    
    return { startTime: '09:00', endTime: '17:00' };
  }
  
  // Helper to extract instructor name and additional info from cell
  function parseInstructorCell(cellValue) {
    if (!cellValue || typeof cellValue !== 'string') return null;
    
    const trimmed = cellValue.trim();
    if (!trimmed || trimmed.toLowerCase().includes('holiday')) return null;
    
    // Skip cells that are just placeholders
    if (trimmed === 'TBD' || trimmed === '-' || trimmed === '') return null;
    
    // Extract instructor name (usually first word or before parenthesis)
    let instructorName = trimmed;
    let timeOverride = null;
    let cohortInfo = null;
    
    // Check for time override in parentheses like "(6:00pm-10:00pm)"
    const timeInParens = trimmed.match(/\(([^)]*\d{1,2}[:-]\d{0,2}[^)]*)\)/);
    if (timeInParens) {
      timeOverride = parseTimeRange(timeInParens[1]);
      instructorName = trimmed.replace(timeInParens[0], '').trim();
    }
    
    // Check for WIOA or week info
    if (trimmed.toLowerCase().includes('wioa')) {
      cohortInfo = 'WIOA';
    }
    
    // Extract just the name (first word that looks like a name)
    const nameMatch = instructorName.match(/^([A-Za-z]+(?:\s*&\s*[A-Za-z]+)?)/);
    if (nameMatch) {
      instructorName = nameMatch[1].trim();
    }
    
    // Skip if still looks like a non-name
    if (instructorName.length < 2) return null;
    
    return {
      instructor: instructorName,
      timeOverride,
      cohortInfo,
      rawValue: trimmed
    };
  }
  
  // Process data rows (skip header rows)
  const dataStartRow = headerRowIndex >= 0 ? headerRowIndex + 2 : 4;
  
  for (let rowIdx = dataStartRow; rowIdx < parsedData.length; rowIdx++) {
    const row = parsedData[rowIdx];
    if (!row || row.length < 2) continue;
    
    results.stats.totalRows++;
    
    // First column is usually day (MON, TU, W, etc.)
    // Second column is date (9/22, 10/1, etc.)
    const dayCell = (row[0] || '').toString().trim();
    const dateCell = (row[1] || '').toString().trim();
    
    // Skip holiday rows
    if (dayCell.toLowerCase().includes('holiday') || dateCell.toLowerCase().includes('holiday')) {
      results.skipped.push({ row: rowIdx + 1, reason: 'Holiday row' });
      continue;
    }
    
    const sessionDate = parseDate(dayCell, dateCell, year);
    if (!sessionDate) continue;
    
    // Process each course column
    for (const [colIdxStr, courseName] of Object.entries(courseColumns)) {
      const colIdx = parseInt(colIdxStr, 10);
      const cellValue = row[colIdx];
      
      const parsed = parseInstructorCell(cellValue);
      if (!parsed) continue;
      
      results.stats.sessionsFound++;
      
      // Get course config
      const config = mergedCourseConfig[courseName] || { colorCode: '#1e90ff' };
      
      // Determine time
      let times = parsed.timeOverride || parseTimeRange(config.times);
      
      // Find teacher ID from mapping
      let teacherId = null;
      const instructorLower = parsed.instructor.toLowerCase();
      
      for (const [name, id] of Object.entries(teacherMapping)) {
        if (name.toLowerCase().includes(instructorLower) || instructorLower.includes(name.toLowerCase())) {
          teacherId = id;
          break;
        }
      }
      
      if (!teacherId) {
        results.errors.push({
          row: rowIdx + 1,
          course: courseName,
          instructor: parsed.instructor,
          error: `Unknown instructor: ${parsed.instructor}`
        });
        continue;
      }
      
      // Build cohort name
      const monthName = new Date(sessionDate).toLocaleString('en-US', { month: 'long', year: 'numeric' });
      const cohortName = parsed.cohortInfo || monthName;
      
      // Check for conflicts if needed
      if (skipConflicts) {
        const conflicts = await checkTeacherConflicts(teacherId, sessionDate, times.startTime, times.endTime);
        if (conflicts.length > 0) {
          results.stats.conflictsSkipped++;
          results.skipped.push({
            row: rowIdx + 1,
            course: courseName,
            date: sessionDate,
            reason: 'Teacher conflict'
          });
          continue;
        }
      }
      
      // Create the session
      try {
        const session = {
          cohortName,
          courseId: null,
          courseName,
          sessionDate,
          startTime: times.startTime,
          endTime: times.endTime,
          teacherId,
          locationOnline: true,
          locationPhysical: null,
          colorCode: config.colorCode,
          programTrack: defaultTrack,
          notes: parsed.rawValue !== parsed.instructor ? parsed.rawValue : null,
          createdBy
        };
        
        const id = await createScheduledSession(session);
        results.created.push({ ...session, id });
        results.stats.sessionsCreated++;
      } catch (err) {
        results.errors.push({
          row: rowIdx + 1,
          course: courseName,
          date: sessionDate,
          error: err.message
        });
      }
    }
  }
  
  return results;
}

/**
 * Clear all scheduled sessions (use with caution!)
 */
async function clearAllScheduledSessions() {
  await ensureTables();
  const [result] = await db.query('DELETE FROM mdtslms_scheduled_sessions');
  return result.affectedRows;
}

module.exports = {
  ensureTables,
  // Teacher availability
  getTeacherAvailability,
  getAllTeacherAvailability,
  setTeacherAvailability,
  addTeacherAvailabilitySlot,
  deleteTeacherAvailabilitySlot,
  // Scheduled sessions
  getAllScheduledSessions,
  getScheduledSessionsInRange,
  getScheduledSessionsByTeacher,
  getScheduledSessionsByCohort,
  getScheduledSessionsByCourse,
  createScheduledSession,
  updateScheduledSession,
  deleteScheduledSession,
  bulkCreateScheduledSessions,
  // Conflict detection
  checkTeacherConflicts,
  isTeacherAvailableOnDate,
  // Program tracks
  getAllProgramTracks,
  createProgramTrack,
  updateProgramTrack,
  deleteProgramTrack,
  // Locations
  getAllLocations,
  createLocation,
  // Unique values
  getUniqueCohorts,
  getUniqueCourses,
  // Time-off / exceptions
  getTeacherTimeOff,
  getAllTimeOff,
  addTimeOff,
  updateTimeOff,
  deleteTimeOff,
  // Weekly overrides
  getWeekStart,
  getWeekOverrides,
  getAllWeekOverridesForTeacher,
  setWeekOverride,
  deleteWeekOverride,
  deleteWeekOverrides,
  copyWeekToOverride,
  // School holidays
  getSchoolHolidays,
  addSchoolHoliday,
  updateSchoolHoliday,
  deleteSchoolHoliday,
  // Effective availability
  getEffectiveAvailability,
  // CSV Import
  importScheduleFromCSV,
  clearAllScheduledSessions
};
