# Student Mobile API (Session-Based)

- Base URL: `https://<host>` (same as web app). All student routes require an authenticated session cookie from `/login`.
- Auth model: session cookie `connect.sid` issued after login; reuse it on every call.
- Most routes render HTML. JSON responses are called out; otherwise parse the HTML or add thin JSON wrappers if you extend the server.

## Auth
- `POST /login` — form fields `username`, `password` (optional `next`). On success follow 302 and store `connect.sid`. 401 renders error page.
- `GET /logout` — clears session, redirects to `/login`.

Example:
```bash
curl -c cookie.txt -d "username=alice&password=secret" https://<host>/login
curl -b cookie.txt https://<host>/student
```

## Dashboard and Announcements
- `GET /student` — dashboard for the logged-in student (classes, events, latest grades, pending tasks/tests, pending signature docs). HTML.
- `GET /student/announcements` — announcements for enrolled classes. HTML.
- `POST /student/announcements/:id/delete` — deletes an announcement. Redirects.

## Classes and Content
- `GET /student/classes/:id` — class detail (lectures, simulations under `/sims/...`, assignments, tests, discussions, drip-filtered). Includes per-test/assignment grades and in-progress test status. HTML.

## Assignments
- `POST /student/classes/:id/assignments/:assignmentId/submit` — `multipart/form-data` with:
  - `file` (png/jpg/jpeg/csv, ≤25MB)
  - `note` (optional text)
  Response: JSON `{ ok, url, submittedAt }` or `{ error }`. 70

## Discussions
- `POST /student/classes/:id/discussion` — form field `message`; posts to discussion. Redirects to class page.

## Tests (official/graded)
- `GET /student/classes/:id/tests/:testId` — loads test-taking page and any saved progress. Blocks after 5 attempts. HTML.
- `POST /student/classes/:id/tests/:testId` — submit answers; fields `q_0`, `q_1`, … (option numbers). Records grade if official and renders result page with breakdown.
- `POST /student/classes/:id/tests/:testId/save-progress` — JSON auto-save. Body:
  ```json
  { "answers": { "0": 2, "1": 3 }, "timeRemaining": 540 }
  ```
  Response: `{ "ok": true }` or `{ "error": "…" }`.
- `POST /student/classes/:id/tests/:testId/reset` — clears saved progress; redirects to test start.

## Tests (study/practice mode)
- `GET /student/classes/:id/tests/:testId/study` — study session view. HTML.
- `POST /student/classes/:id/tests/:testId/study` — form field `choice` (option number) per question step; returns next question/feedback. HTML.

## Grades and Certificates
- `GET /student/classes/:id/grade-report` — grade report for that class. HTML.
- `GET /student/certificates/:classId/:testId` — certificate for a passed test. HTML view.
- `GET /student/certificates/:classId/completion` — completion certificate for class. HTML view.

## Profile and Documents
- `GET /student/profile` — profile page with enrollment docs and signatures. HTML.
- `POST /student/profile/upload-documents` — `multipart/form-data`; fields `idDocument`, `transcriptDocument` (pdf/jpg/png/doc/docx, ≤10MB). Redirect.
- `POST /student/profile/complete` — form fields `ssn`, emergency contact, `agree`, `grievanceAck`; optional uploads `gedDoc`, `govIdDoc` (pdf/jpg/png/doc/docx, ≤15MB). Redirect.
- `POST /student/sign-doc` — form fields `docType`, `signatureDataUrl` (data URL). JSON `{ success, signedAt }` when XHR; otherwise redirect.
- `POST /student/documents/to-sign/:docId` — `multipart/form-data` with `signedFile` (pdf/jpg/png/doc/docx, ≤20MB). Redirect.

## Mailbox
- `GET /student/mailbox` — inbox/outbox view; includes teacher list. HTML.
- `POST /student/mailbox` — send message. Form fields `to` (user id), `subject`, `body`. Redirect.

## Career Services (optional)
- `GET /student/career-advisor` — scheduling page. HTML.
- `POST /student/career-advisor/schedule` — `multipart/form-data`; fields `name`, `email`, `phone`, `date`, `time`, `interestArea`, `summary` (optional), `resume` file (pdf/doc/docx/pages, ≤10MB). Renders confirmation/success page.

## Notes for Mobile Clients
- Always include the session cookie `connect.sid` after logging in.
- Handle 302 redirects to `/login` as session-expired signals.
- Only the endpoints listed as JSON return JSON; others return HTML and may require scraping or adding JSON-friendly wrappers if you extend the server.
- File uploads: ensure `multipart/form-data` with correct field names and size/type limits noted above.

## Teacher (high-level)
- `GET /teacher` — dashboard; classes taught, announcements, messages. HTML.
- `GET /teacher/announcements` | `POST /teacher/announcements` — list/create announcements (body: `message`, `classId`). HTML.
- `GET /teacher/classes/:id` — class view (roster, content, grades, attendance). HTML.
- `POST /teacher/classes/:id/assignments` — add assignment (fields: `title`, `url`, `date`, `dripDay`). Redirect.
- `POST /teacher/classes/:id/grades` — upsert grades. Body keys: `grade_{studentId}_{testId}` for tests, `asg_{studentId}_{assignmentId}` for assignments, `lab_{studentId}_{simId}` (checkbox) for labs. Redirect.
- `POST /teacher/classes/:id/tests` & related routes — create/edit tests; accepts CSV upload or generator (HTML forms).
- `POST /teacher/classes/:id/attendance` — record attendance for a date (HTML form keys per student). Redirect.
- Messaging: `GET/POST /teacher/mailbox` similar to student mailbox.

## Admin (high-level)
- `GET /admin` — admin dashboard. HTML.
- Student lifecycle: `GET /admin/approvals`, `POST /admin/approve/:id`, `POST /admin/decline/:id`, `GET /admin/students`, `GET /admin/students/:id`, `POST /admin/students/:id/enroll`, `POST /admin/students/:id/unenroll/:classId`, `POST /admin/students/:id/reset-password`, and numerous document/signature/certificate helpers.
- Classes: `POST /admin/classes` (fields: `schoolYear`, `cohort`, `name`, `shortName`, `description`, `teacherId`, `assistantTeacherId`, `startDate`, `endDate`, `weeks`, `clockHours`, schedule arrays `day[]`, `start[]`, `end[]`), plus many `POST /admin/classes/:id/*` routes for lectures, sims, assignments, tests, links, attendance, duplication, rename, delete. HTML forms.
- Announcements: `GET /admin/announcements`, `POST /admin/announcements` (fields: `message`, `audience`, optional class filter). HTML.
- Calendar/events, leads/pre-registrations, partners/mentors, marketing assets, charts/reports, career-advisor interview management — all HTML form-driven routes in `routes/admin.js`.

## Postman Collection
- Import `docs/MDTS_LMS.postman_collection.json` into Postman. It includes sample requests for auth, student, teacher, and admin flows with variables: `baseUrl`, `classId`, `testId`, `assignmentId`, `studentId`, `teacherId`.
- Workflow: run **Auth / Login** first to capture the session cookie in Postman’s cookie jar, then hit the role-specific requests. Update path variables and bodies as needed.
