# Test-Taking System Enhancements

## Summary
Comprehensive improvements to the test-taking experience with enhanced UX, state persistence, retry functionality, and progress tracking.

## Features Implemented

### 1. Clickable Answer Boxes ✅
**Location:** `views/take_test.ejs`

**Enhancement:** Students can now click anywhere on the answer box to select an answer, not just the radio button itself.

**Implementation:**
- Added click event listeners to `.form-check` divs
- Made boxes cursor pointer for better UX
- Prevents double-triggering when clicking radio/label directly

### 2. Test Progress State Saving ✅
**Location:** `models/testProgressModel.js`, `scripts/setupTestProgress.js`

**Enhancement:** Test progress is automatically saved every 10 seconds, allowing students to resume after disconnection or leaving the page.

**Database Schema:**
```sql
CREATE TABLE test_progress (
  id INT AUTO_INCREMENT PRIMARY KEY,
  student_id INT NOT NULL,
  class_id INT NOT NULL,
  test_id INT NOT NULL,
  answers JSON,
  time_remaining INT,
  started_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  last_saved_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  completed BOOLEAN DEFAULT FALSE,
  INDEX idx_student_test (student_id, class_id, test_id),
  UNIQUE KEY unique_in_progress (student_id, class_id, test_id, completed)
)
```

**Model Methods:**
- `saveProgress()` - Save/update test progress
- `getProgress()` - Get in-progress test for a student
- `getIncompleteTests()` - Get all incomplete tests for a student
- `markCompleted()` - Mark test as completed after submission
- `deleteProgress()` - Reset test progress (for Try Again/Start Over)

### 3. Resume Test Functionality ✅
**Location:** `routes/student.js`, `views/take_test.ejs`

**Enhancement:** When students return to a test, their previous answers and remaining time are restored.

**Features:**
- Auto-save progress every 10 seconds via `/student/classes/:id/tests/:testId/save-progress`
- Load saved progress when test page is accessed
- Restore selected answers and time remaining
- Progress marked as completed upon test submission

**Routes Added:**
- `POST /student/classes/:id/tests/:testId/save-progress` - Auto-save endpoint
- `POST /student/classes/:id/tests/:testId/reset` - Reset/restart test

### 4. Try Again Button ✅
**Location:** `views/test_result.ejs`

**Enhancement:** When students score below 70%, a "Try Again" button is displayed to reset and retake the test.

**Features:**
- Only shown when score < 70% (configurable via PASSING_SCORE env var)
- Resets test progress and redirects to fresh test
- Uses Bootstrap warning styling with retry icon
- Positioned prominently in alert box

### 5. Incomplete Test Dashboard Prompt ✅
**Location:** `routes/student.js`, `views/student_dashboard.ejs`

**Enhancement:** Students are notified on the dashboard if they have an incomplete test and can resume with one click.

**Features:**
- Shows alert for most recent incomplete test
- Displays class name, test title, and direct resume link
- Alert is dismissible
- Uses info styling with clock icon
- Positioned at top of dashboard

### 6. Test Progress on Class Page ✅
**Location:** `routes/student.js`, `views/view_class.ejs`

**Enhancement:** In the class view, students see their progress on incomplete tests with progress bars and Resume/Start Over buttons.

**Features:**
- Progress bar showing answered questions percentage
- "X of Y questions answered" text
- "Resume" button (info color) to continue test
- "Start Over" button to reset progress
- Only shows progress UI for incomplete tests
- Shows final grade badge for completed tests

## User Flow Examples

### Scenario 1: Test Interruption
1. Student starts taking a test
2. Connection drops or browser closes
3. Student logs back in
4. Dashboard shows "Resume Your Test" alert
5. Student clicks alert link or goes to class page
6. Test loads with all previous answers and remaining time

### Scenario 2: Failed Test
1. Student completes test and scores 65%
2. Results page shows "Try Again" button
3. Student clicks "Try Again"
4. Progress is reset and test reloads from start
5. Student can retake with fresh attempt

### Scenario 3: Multiple In-Progress Tests
1. Student starts a test but doesn't finish
2. Goes to class page
3. Sees progress bar showing "8 of 10 questions answered"
4. Can click "Resume" to continue or "Start Over" to reset
5. Dashboard also prompts for most recent incomplete test

## Technical Notes

### Auto-Save Mechanism
- Saves every 10 seconds while test is open
- Captures all selected answers as JSON object
- Saves remaining time in seconds
- Uses fetch API with error handling

### Progress Calculation
- Counts answered questions (questions with selected radio button)
- Calculates percentage: `(answered / total) * 100`
- Updates progress bar and stats in real-time

### Data Persistence
- Progress stored in MySQL with JSON column for answers
- Unique constraint prevents duplicate in-progress entries
- Completed flag allows multiple completed attempts while maintaining one in-progress

### Resume Logic
- On test page load, checks for saved progress
- If found, pre-selects radio buttons and sets timer
- Otherwise, starts fresh test with full time

## Files Modified

1. **views/take_test.ejs** - Added clickable boxes, auto-save, resume functionality
2. **views/test_result.ejs** - Added Try Again button for failing scores
3. **views/student_dashboard.ejs** - Added incomplete test alert
4. **views/view_class.ejs** - Added progress bars and Resume/Start Over buttons
5. **routes/student.js** - Added progress loading, save endpoint, reset endpoint, incomplete test detection
6. **models/testProgressModel.js** - New model for test progress operations
7. **scripts/setupTestProgress.js** - Database table setup script

## Environment Variables
- `PASSING_SCORE` - Minimum score to pass (default: 70)

## Database Changes
- New table: `test_progress`
- Indexes: `idx_student_test` for efficient lookups
- Unique constraint: `unique_in_progress` to prevent duplicates

## Testing Recommendations

1. **Test Interruption:** Start test, close browser, reopen, verify resume
2. **Auto-Save:** Take test, check database for progress updates every 10 seconds
3. **Try Again:** Score below 70%, click Try Again, verify test resets
4. **Dashboard Alert:** Leave test incomplete, go to dashboard, verify alert shows
5. **Class Progress:** View class with incomplete test, verify progress bar displays
6. **Multiple Tests:** Start multiple tests, verify dashboard shows most recent
7. **Completion:** Complete test, verify progress marked as completed in database
8. **Time Persistence:** Partially complete test, verify time remaining is saved/restored
