# Programs List Feature

## Overview
The student profile Program section has been restructured from a single-entry system to a dynamic list system, allowing admins to add multiple course enrollments for each student. Each program entry contains complete details including course name, affiliated program, dates, and class schedule.

## Feature Details

### Admin View (Edit Mode)
**Location:** `views/student_profile.ejs` (Admin section, lines ~413-490)

#### Features:
1. **Dynamic List Interface**
   - Section titled "Program Enrollments"
   - Green "Add Course" button in the header
   - Each program displayed as a bordered card with delete button
   - Empty state message when no programs exist

2. **Program Entry Fields**
   Each program entry contains:
   - **Course** (required): Course name
   - **Affiliated Program**: Associated program/organization
   - **Admission Date**: Date format input
   - **Start Date**: Date format input
   - **End Date**: Date format input
   - **Class Time**: Text input (e.g., "8:00pm")
   - **Class Days**: Text input (e.g., "33")

3. **Add/Remove Functionality**
   - "Add Course" button adds new program entry with empty fields
   - Each entry has a delete/trash button for removal
   - Confirmation dialog before deleting entries
   - Auto-focus on course name when adding new entry

4. **Backwards Compatibility**
   - Automatically converts old single-program structure to array format
   - Preserves existing data during migration

5. **Grievance Acknowledged**
   - Moved outside programs list (student-level field)
   - Dropdown: Yes/No

### Student View (Read-Only)
**Location:** `views/student_profile.ejs` (Student section, lines ~594-637)

#### Features:
- Section titled "Program Enrollments"
- Each program displayed in a light bordered card
- Course name shown as header with icon
- All fields displayed as read-only text
- Empty state message if no programs enrolled

## Data Structure

### New Structure (Programs Array)
```javascript
{
  profile: {
    programs: [
      {
        course: "Security+",
        affiliateProgram: "Army",
        admissionDate: "10/20/2025",
        startDate: "10/20/2025",
        endDate: "10/24/2025",
        classTime: "8:00pm",
        classDays: "33"
      },
      {
        course: "Network+",
        affiliateProgram: "Navy",
        admissionDate: "11/01/2025",
        startDate: "11/01/2025",
        endDate: "11/30/2025",
        classTime: "6:00pm",
        classDays: "20"
      }
    ],
    grievanceAcknowledged: true
  }
}
```

### Old Structure (Backwards Compatible)
```javascript
{
  profile: {
    course: "Security+",
    affiliateProgram: "Army",
    program: {
      admissionDate: "10/20/2025",
      startDate: "10/20/2025",
      endDate: "10/24/2025",
      classTime: "8:00pm",
      classDays: "33"
    },
    grievanceAcknowledged: true
  }
}
```

**Migration:** The view automatically converts old structure to new array format on render.

## Backend Implementation

### Route Handler
**File:** `routes/admin.js`
**Route:** `PUT /admin/students/:id`
**Lines:** ~2676-2770

#### Request Processing:
1. Accepts `programs` object from form data
2. Converts object to array format
3. Filters out empty entries (requires course name)
4. Trims whitespace from all fields
5. Stores as `profile.programs` array

#### Form Data Format:
```javascript
{
  programs: {
    0: {
      course: "Security+",
      affiliateProgram: "Army",
      admissionDate: "2025-10-20",
      startDate: "2025-10-20",
      endDate: "2025-10-24",
      classTime: "8:00pm",
      classDays: "33"
    },
    1: {
      course: "Network+",
      // ...
    }
  }
}
```

#### Backwards Compatibility:
- If `programs` array not provided, falls back to old structure
- Preserves old `course`, `affiliateProgram`, and `program` fields
- Allows gradual migration of existing data

### Database Update
Uses existing `userModel.updateProfile(id, profileUpdate)` method. No schema changes required as MySQL stores profile as JSON.

## JavaScript Functions

### Admin Functions
**Location:** `views/student_profile.ejs` (Script section, lines ~2154-2251)

#### `addProgramEntry()`
- Creates new program entry HTML
- Increments program counter for unique indices
- Removes empty state alert if present
- Appends new entry to container
- Auto-focuses on course input field

#### `removeProgramEntry(index)`
- Finds entry by data-index attribute
- Shows confirmation dialog (SweetAlert2)
- Removes entry from DOM on confirmation
- Shows empty state if no entries remain

#### Counter Management
```javascript
let programCounter = document.querySelectorAll('.program-entry').length;
```
Tracks total entries for generating unique field names.

## UI Components

### Program Entry Card
```html
<div class="program-entry border rounded p-3 mb-3" data-index="0">
  <div class="d-flex justify-content-between align-items-center mb-2">
    <h6 class="mb-0 text-primary">
      <i class="bi bi-mortarboard-fill"></i> Security+
    </h6>
    <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeProgramEntry(0)">
      <i class="bi bi-trash"></i>
    </button>
  </div>
  <div class="row g-2">
    <!-- Form fields -->
  </div>
</div>
```

### Empty State
```html
<div class="alert alert-info mb-0">
  <i class="bi bi-info-circle"></i> No program enrollments yet. Click "Add Course" to create one.
</div>
```

## Form Submission

### Field Naming Convention
Array-based naming for proper parsing:
```html
<input name="programs[0][course]" value="Security+">
<input name="programs[0][admissionDate]" value="2025-10-20">
<input name="programs[1][course]" value="Network+">
```

### Validation
- Course field marked as `required`
- Backend filters out entries without course name
- Empty entries automatically excluded

## Styling

### Bootstrap Classes
- **Cards:** `border rounded p-3 mb-3`
- **Headers:** `text-primary mb-2`
- **Buttons:** `btn-sm btn-success` (Add), `btn-sm btn-outline-danger` (Delete)
- **Grid:** `row g-2` with responsive columns (`col-md-4`, `col-md-6`)
- **Empty State:** `alert alert-info`

### Icons
- **Course Icon:** `bi-mortarboard-fill`
- **Add Icon:** `bi-plus-circle`
- **Delete Icon:** `bi-trash`
- **Info Icon:** `bi-info-circle`

## User Workflow

### Adding a Course
1. Admin clicks "Add Course" button
2. New entry card appears at bottom
3. Course field auto-focused
4. Admin fills in all desired fields
5. Submits main profile form
6. Programs array saved to database

### Editing a Course
1. Admin modifies any field in existing entry
2. Submits main profile form
3. Updated data saved to database

### Removing a Course
1. Admin clicks trash icon on entry
2. Confirmation dialog appears
3. On confirmation, entry removed from DOM
4. Submit form to persist changes

### Student View
1. Student views their profile
2. Sees all enrolled programs as read-only cards
3. Cannot add, edit, or delete programs
4. Can view all program details

## Benefits

1. **Multiple Enrollments:** Students can be enrolled in multiple courses simultaneously
2. **Complete History:** Track all past and future program enrollments
3. **Flexible Dates:** Each program has independent dates
4. **Better Organization:** Clear visual separation between programs
5. **Backwards Compatible:** Existing single-program data preserved
6. **User-Friendly:** Intuitive add/remove interface
7. **Data Integrity:** Required fields and validation

## Technical Notes

### Migration Strategy
- View-level migration: Old structure converted on render
- No database migration required
- Gradual transition as profiles are edited
- Both structures supported indefinitely

### Performance
- Client-side add/remove (instant feedback)
- Single form submission for all programs
- No AJAX calls for individual operations
- Efficient DOM manipulation

### Future Enhancements
- Individual edit modals per program
- Drag-and-drop reordering
- Bulk import from CSV
- Program status tracking (Active/Completed/Withdrawn)
- Completion certificates linked to specific programs
- Filtering and sorting in student list by program

## Testing Checklist

- [x] Add new program entry
- [x] Remove program entry with confirmation
- [x] Empty state displays correctly
- [x] Backwards compatibility with old structure
- [x] Form submission with multiple programs
- [x] Student read-only view displays programs
- [x] Course field validation (required)
- [x] Date input formats
- [x] Grievance acknowledged moved outside programs
- [x] Auto-focus on new entry
- [x] Program counter increments correctly
- [x] Empty entries filtered out on backend

## Files Modified

1. **views/student_profile.ejs**
   - Admin edit section (lines ~413-490)
   - Student read-only section (lines ~594-637)
   - JavaScript functions (lines ~2154-2251)

2. **routes/admin.js**
   - PUT /admin/students/:id route (lines ~2676-2770)
   - Programs array processing logic
   - Backwards compatibility handling

## Dependencies

- **Bootstrap 5.3.3:** UI components and grid system
- **Bootstrap Icons:** Icon set
- **SweetAlert2:** Confirmation dialogs
- **Express.js:** Backend routing
- **MySQL:** Database storage (JSON field)

---

**Last Updated:** 2025
**Status:** ✅ Implemented and Running
