# Email Link Authentication Flow Fix

## Issue
When students receive a signature request email and click the link, they were getting "Access denied" instead of being redirected to login.

## Problem Analysis

The authentication flow had these pieces in place:
1. ✅ `requireAuth` middleware stores `req.session.returnTo` with the original URL
2. ✅ Middleware redirects to `/login`
3. ❌ Login route doesn't check `req.session.returnTo`
4. ❌ After login, user goes to dashboard instead of original URL

## Solution Implemented

Updated the login routes to check for `req.session.returnTo` and redirect back to the original URL after successful authentication.

### Changes Made

#### 1. GET /login Route
**Before:**
```javascript
router.get('/login', (req, res) => {
  const next = ensureStr(req.query.next);
  const safeNext = next.startsWith('/') ? next : '';
  res.render('login', { error: null, next: safeNext });
});
```

**After:**
```javascript
router.get('/login', (req, res) => {
  // Check for returnTo in session (set by requireAuth middleware) or query param
  const returnTo = req.session.returnTo || req.query.next;
  const next = ensureStr(returnTo);
  const safeNext = next.startsWith('/') ? next : '';
  res.render('login', { error: null, next: safeNext });
});
```

#### 2. POST /login Route
**Before:**
```javascript
const redirectTo = ensureStr(req.body.next);
const safeRedirect = redirectTo.startsWith('/') ? redirectTo : '';
// ... authentication logic ...
return res.redirect(safeRedirect || '/dashboard');
```

**After:**
```javascript
// Check both form input and session returnTo
const returnTo = req.session.returnTo || req.body.next;
const redirectTo = ensureStr(returnTo);
const safeRedirect = redirectTo.startsWith('/') ? redirectTo : '';
// ... authentication logic ...

// Clear returnTo from session after using it
delete req.session.returnTo;

// ... rest of logic ...
return res.redirect(safeRedirect || '/dashboard');
```

## How It Works Now

### Complete Flow

1. **Student Receives Email**
   - Email contains link: `http://localhost:3012/document-signing/sign/12/34`
   - Student clicks the link

2. **Unauthenticated Access Attempt**
   - Browser requests `/document-signing/sign/12/34`
   - `requireAuth` middleware checks session
   - No user in session → stores URL in `req.session.returnTo`
   - Redirects to `/login`

3. **Login Page**
   - GET `/login` route checks `req.session.returnTo`
   - Finds: `/document-signing/sign/12/34`
   - Passes to login form as hidden field
   - Renders login page

4. **User Logs In**
   - Student enters username/password
   - POST `/login` route processes credentials
   - Checks both `req.session.returnTo` and `req.body.next`
   - Finds the return URL
   - Validates credentials

5. **Successful Login**
   - Session created with user info
   - `req.session.returnTo` deleted (cleanup)
   - Redirects to `/document-signing/sign/12/34`

6. **Access Document**
   - Request goes through `requireAuth` middleware
   - User is authenticated → passes through
   - Signing page loads successfully

### Security Features

1. **URL Validation**: Only relative URLs (starting with `/`) are accepted
   ```javascript
   const safeRedirect = redirectTo.startsWith('/') ? redirectTo : '';
   ```
   This prevents open redirect vulnerabilities.

2. **Session Cleanup**: After using `returnTo`, it's deleted from the session
   ```javascript
   delete req.session.returnTo;
   ```
   Prevents confusion on subsequent logins.

3. **Fallback**: If no return URL, defaults to dashboard
   ```javascript
   return res.redirect(safeRedirect || '/dashboard');
   ```

## Email Link Structure

The signature request emails contain links like:
```
${baseUrl}/document-signing/sign/${documentId}/${partyId}
```

Example:
```
http://localhost:3012/document-signing/sign/12/34
```

Where:
- `12` is the document ID
- `34` is the party ID

## Files Modified

### routes/auth.js

#### GET /login
- Added check for `req.session.returnTo`
- Prioritizes session returnTo over query param
- Passes to login form

#### POST /login
- Added check for `req.session.returnTo`
- Prioritizes session returnTo over form input
- Clears `returnTo` after use
- Redirects to original URL after successful login

## Testing Scenarios

### Scenario 1: Email Link (Unauthenticated)
```
1. User clicks email link
2. Sees login page
3. Enters credentials
4. ✓ Redirected to signing page
```

### Scenario 2: Email Link (Already Logged In)
```
1. User clicks email link
2. ✓ Goes directly to signing page (requireAuth passes)
```

### Scenario 3: Direct Login (No Return URL)
```
1. User goes to /login directly
2. Enters credentials
3. ✓ Redirects to /dashboard (default)
```

### Scenario 4: Login with Query Param
```
1. User goes to /login?next=/student/classes
2. Enters credentials
3. ✓ Redirects to /student/classes
```

### Scenario 5: Security Test (Malicious Redirect)
```
1. Attacker creates link: /login?next=http://evil.com
2. User logs in
3. ✓ Redirects to /dashboard (external URL rejected)
```

## Benefits

1. **Seamless UX**: Students can go directly from email to signing without manually navigating
2. **Reduced Friction**: One-click access to documents after authentication
3. **Better Conversion**: More likely to complete signing when process is smooth
4. **Secure**: Validates URLs and prevents open redirect attacks
5. **Flexible**: Works with both email links and manual navigation

## Related Middleware

### requireAuth (documentSigning.js)
```javascript
function requireAuth(req, res, next) {
  if (!req.session || !req.session.user || !req.session.user.id) {
    // Store the original URL to redirect back after login
    req.session.returnTo = req.originalUrl;
    return res.redirect('/login');
  }
  next();
}
```

This middleware:
- Checks for authenticated session
- Stores original URL if not authenticated
- Redirects to login page

## Edge Cases Handled

1. **Pending Students**: If student status is pending, they see pending page instead (existing logic preserved)
2. **Password Change Required**: Redirects to account page first (existing logic preserved)
3. **No Return URL**: Falls back to dashboard
4. **Invalid URL Format**: Rejects non-relative URLs
5. **Session Persistence**: returnTo survives page refresh before login

## Future Enhancements

Potential improvements:
- Add expiration time to returnTo (auto-clear after 30 minutes)
- Log authentication attempts from email links for analytics
- Add "return to document" breadcrumb after login
- Support deep linking with document title in URL
- Email reminder if user doesn't complete signing within X days
