# Document Signing Module - DocuSign-Style Integration

A **plug-and-play** document signing module for Node.js applications with similar structure to this LMS. This module provides digital signature capabilities similar to DocuSign, allowing administrators to send documents for signature to students and staff members.

## Features

✅ **PDF Document Upload** - Upload PDF documents that require signatures  
✅ **Interactive Field Placement** - Drag-and-drop signature fields, text fields, initials, dates, and checkboxes onto PDF pages  
✅ **Multiple Signers** - Support for both student and staff signatures on the same document  
✅ **Signing Order Control** - Define who signs first (student first, staff first, or any order)  
✅ **Email Notifications** - Automatic email notifications when documents are ready to sign and when completed  
✅ **Electronic Signatures** - Canvas-based signature pad for legal electronic signatures  
✅ **Audit Trail** - Complete activity log of all document actions  
✅ **Status Tracking** - Track document status (draft, awaiting signatures, partially signed, completed)  
✅ **Document Download** - Download completed signed documents as PDF  
✅ **Responsive Design** - Works on desktop, tablet, and mobile devices

---

## Installation & Setup

### 1. Database Initialization

First, initialize the database tables by calling the initialization endpoint **once** after integrating the module:

```bash
# Run this command once after installation
POST /document-signing/initialize-db
```

Or manually run the SQL from the model file to create the required tables:
- `signature_documents`
- `signature_fields`
- `signature_parties`
- `signature_field_responses`
- `signature_audit_log`

### 2. Required Dependencies

Ensure these npm packages are installed:

```bash
npm install multer nodemailer
```

Optional (for advanced PDF manipulation):
```bash
npm install pdf-lib pdf-parse
```

### 3. Environment Variables

Add these to your `.env` file:

```env
# Email Configuration (required for notifications)
SMTP_HOST=smtp.gmail.com
SMTP_PORT=587
SMTP_USER=your-email@domain.com
SMTP_PASS=your-app-password
SMTP_FROM=noreply@yourdomain.com

# Application URL (required for email links)
BASE_URL=http://localhost:3000
```

### 4. File Structure

The module includes these files:

```
Archive 2/
├── models/
│   └── signatureDocumentModel.js      # Database model
├── routes/
│   └── documentSigning.js             # API routes
├── utils/
│   └── documentSigning.js             # Helper functions
├── views/
│   ├── admin_signature_documents.ejs  # Admin document list
│   ├── admin_signature_document_create.ejs
│   ├── admin_signature_document_edit.ejs
│   ├── signature_document_sign.ejs    # Signing interface
│   ├── signature_documents_pending.ejs
│   └── signature_document_view.ejs
├── public/
│   └── js/
│       └── signature-field-editor.js  # Client-side field editor
└── uploads/
    └── signature-documents/           # Auto-created upload directory
```

### 5. Integration

The module is already integrated in `app.js`:

```javascript
const documentSigningRoutes = require('./routes/documentSigning');
app.use('/document-signing', documentSigningRoutes);
```

---

## Usage Guide

### For Administrators

#### 1. Access Document Signing from Student Profile

- Navigate to a student's profile page
- Click the **"Document Signing"** button (blue button near top)
- This opens the document management page for that student

#### 2. Create a New Document

1. Click **"New Document"** button
2. Fill in the form:
   - **Document Title**: e.g., "Enrollment Agreement"
   - **Description**: Optional brief description
   - **Signing Order**: Choose who signs first
     - Any Order: Both parties can sign anytime
     - Student First: Student must sign before staff
     - Staff First: Staff must sign before student
3. **Upload PDF**: Drag and drop or click to browse for PDF file (max 10MB)
4. Click **"Upload & Continue"**

#### 3. Place Signature Fields

After uploading, you'll see the PDF with a field palette on the left:

**Available Field Types:**
- **Signature** - Full signature field
- **Initials** - Initials field
- **Text Field** - For typed text input
- **Date** - Date picker field
- **Checkbox** - Agreement checkbox

**For Each Party:**
- **Student Fields** - Blue colored
- **Staff Fields** - Green colored

**To Add Fields:**
1. Drag a field type from the left panel
2. Drop it onto the PDF where you want it
3. Configure the field:
   - Field Label (e.g., "Student Signature")
   - Placeholder text (optional)
   - Required checkbox
4. Click **"Add Field"**
5. Reposition fields by dragging them on the PDF
6. Remove fields by clicking the × button

#### 4. Send for Signature

1. Click **"Save & Continue"** when all fields are placed
2. In the "Send for Signature" modal:
   - Student is automatically included
   - Select a staff member from the dropdown
   - Set the signing order (1st or 2nd)
3. Click **"Send for Signature"**
4. Email notifications are automatically sent to all parties

#### 5. Track Document Status

- **Draft** - Document created but not sent
- **Awaiting Signatures** - Sent to parties, none signed yet
- **Partially Signed** - Some parties have signed
- **Completed** - All parties have signed

### For Students & Staff (Signers)

#### 1. Receive Email Notification

When a document is ready for your signature, you'll receive an email with:
- Document title and description
- "Review & Sign Document" button
- Direct link to the signing page

#### 2. Review Document

1. Click the link in the email or go to your pending documents
2. Review the full PDF document
3. See all required fields you need to complete

#### 3. Complete Fields

For each field assigned to you:

- **Signature Fields**: Draw your signature with mouse or touch
  - Click "Clear" to start over if needed
  
- **Initial Fields**: Draw your initials
  
- **Text Fields**: Type the required information
  
- **Date Fields**: Select or enter a date
  
- **Checkboxes**: Check to agree/acknowledge

#### 4. Sign the Document

1. Complete all required fields
2. Check the electronic signature agreement checkbox
3. Click **"Sign Document"**
4. You'll see a confirmation message
5. A copy of the signed document will be emailed when all parties sign

---

## API Endpoints

### Admin Endpoints

| Method | Endpoint | Description |
|--------|----------|-------------|
| GET | `/document-signing/student/:studentId` | View all documents for a student |
| GET | `/document-signing/create/:studentId` | Show create document form |
| POST | `/document-signing/upload` | Upload a PDF document |
| GET | `/document-signing/edit/:documentId` | Edit document and place fields |
| POST | `/document-signing/fields/:documentId` | Save signature fields |
| POST | `/document-signing/parties/:documentId` | Send document for signature |
| DELETE | `/document-signing/:documentId` | Delete a document |
| POST | `/document-signing/initialize-db` | Initialize database tables (run once) |

### User Endpoints

| Method | Endpoint | Description |
|--------|----------|-------------|
| GET | `/document-signing/pending` | View documents pending your signature |
| GET | `/document-signing/sign/:documentId/:partyId` | Sign a document |
| POST | `/document-signing/sign/:documentId/:partyId` | Submit signature |
| GET | `/document-signing/view/:documentId` | View completed document |
| GET | `/document-signing/download/:documentId` | Download signed PDF |

---

## Database Schema

### signature_documents
Stores document metadata and status.

**Columns:**
- `id` - Primary key
- `student_id` - Reference to student
- `created_by` - Admin who created it
- `original_filename` - Original PDF filename
- `stored_filename` - Stored filename
- `file_path` - Full file path
- `title` - Document title
- `description` - Optional description
- `status` - Current status (draft, awaiting_signatures, etc.)
- `signing_order` - Signing order rule
- `created_at`, `updated_at`, `completed_at`

### signature_fields
Defines signature fields on the PDF.

**Columns:**
- `id` - Primary key
- `document_id` - Reference to document
- `field_type` - Type (signature, initials, text, date, checkbox)
- `signer_type` - Who fills this field (student, staff)
- `field_label` - Display label
- `page_number` - PDF page number
- `x_position`, `y_position` - Position on page (percentage)
- `width`, `height` - Field dimensions (percentage)
- `is_required` - Required flag
- `placeholder_text` - Placeholder text

### signature_parties
Tracks who needs to sign.

**Columns:**
- `id` - Primary key
- `document_id` - Reference to document
- `user_id` - Reference to user
- `party_type` - Party type (student, staff)
- `signing_order` - Order number
- `status` - Status (pending, viewed, signed, declined)
- `signed_at` - When signed
- `signature_data` - Signature metadata
- `ip_address`, `user_agent` - Audit info
- `email_sent`, `email_sent_at`, `viewed_at`

### signature_field_responses
Stores filled field data.

**Columns:**
- `id` - Primary key
- `field_id` - Reference to field
- `party_id` - Reference to party
- `response_value` - The actual data (text or base64 image)
- `response_type` - Type of response
- `responded_at` - When filled

### signature_audit_log
Complete audit trail of all actions.

**Columns:**
- `id` - Primary key
- `document_id` - Reference to document
- `user_id` - User who performed action
- `action` - Action type
- `details` - Action details
- `ip_address` - IP address
- `created_at` - Timestamp

---

## Customization

### Email Templates

Edit `utils/documentSigning.js` to customize email templates:
- `sendSignatureRequestEmail()` - Notification to sign
- `sendCompletionEmail()` - Document completed notification

### Field Types

Add custom field types by:
1. Adding to `fieldSizes` in `public/js/signature-field-editor.js`
2. Adding rendering logic in `views/signature_document_sign.ejs`
3. Adding to database ENUM in model

### Styling

Customize colors and branding in the view files:
- Update CSS variables (`:root` section)
- Modify Bootstrap classes
- Change icons (using Bootstrap Icons)

---

## Security Considerations

✅ **Authentication Required** - All endpoints require active session  
✅ **Role-Based Access** - Admin-only endpoints are protected  
✅ **File Upload Validation** - Only PDF files accepted, size limits enforced  
✅ **Audit Trail** - All actions logged with IP and timestamp  
✅ **SQL Injection Protection** - Parameterized queries used throughout  
✅ **XSS Protection** - EJS auto-escapes output  

**Recommendations:**
- Use HTTPS in production
- Enable CSRF protection
- Implement file virus scanning
- Set up backup for uploaded documents
- Configure proper file permissions on upload directory

---

## Troubleshooting

### Database Tables Not Created
**Solution**: Call `/document-signing/initialize-db` endpoint once (POST request as admin)

### Emails Not Sending
**Solution**: Check your SMTP credentials in `.env` file and test email server connection

### PDF Not Displaying
**Solution**: Ensure `/uploads/signature-documents/` directory exists and has write permissions

### Fields Not Saving
**Solution**: Check browser console for errors, ensure JavaScript is enabled

### Signatures Not Appearing in PDF
**Solution**: The basic implementation returns the original PDF. For production, integrate `pdf-lib` to overlay signatures (code stub provided in `generateSignedPDF()`)

---

## Advanced Features (Optional)

### PDF Overlay with Signatures

To add actual signature images to the PDF, install and use `pdf-lib`:

```bash
npm install pdf-lib
```

Then update the `generateSignedPDF()` function in `utils/documentSigning.js` to use pdf-lib to overlay the signature images onto the PDF pages.

### Multi-Page PDF Support

The current implementation assumes single-page PDFs. To support multi-page:
1. Add page navigation to the editor
2. Store page_number with each field
3. Render fields on appropriate pages during signing

### Digital Certificates

For legal compliance, integrate with certificate authorities to add digital certificates to signed PDFs.

---

## Module Isolation & Portability

This module is designed to be **plug-and-play**. To move it to another project:

1. **Copy Files**:
   - `models/signatureDocumentModel.js`
   - `routes/documentSigning.js`
   - `utils/documentSigning.js`
   - All `views/*signature*.ejs` files
   - `public/js/signature-field-editor.js`

2. **Update Imports**: Adjust paths in the files if your project structure differs

3. **Integrate Routes**: Add to your main app file:
   ```javascript
   const documentSigningRoutes = require('./routes/documentSigning');
   app.use('/document-signing', documentSigningRoutes);
   ```

4. **Initialize Database**: Run the initialization endpoint

5. **Configure Environment**: Set up SMTP and BASE_URL variables

6. **Add Entry Point**: Add a button/link to access the module (like the student profile button)

---

## Support & Maintenance

### File Structure Checklist
- ✅ Database model (`models/signatureDocumentModel.js`)
- ✅ Routes (`routes/documentSigning.js`)
- ✅ Utilities (`utils/documentSigning.js`)
- ✅ Views (6 EJS files)
- ✅ Client JavaScript (`public/js/signature-field-editor.js`)
- ✅ Integration in `app.js`
- ✅ Upload directory created automatically

### Testing Checklist
1. Initialize database tables
2. Create a test document
3. Place signature fields
4. Send for signature
5. Sign as student
6. Sign as staff
7. Verify completion email
8. Download signed document

---

## License & Credits

This document signing module was created as a plug-and-play solution for Node.js LMS applications. It provides DocuSign-like functionality with a focus on simplicity and ease of integration.

**Technologies Used:**
- Node.js + Express
- MySQL (via mysql2)
- EJS templating
- Bootstrap 5
- Canvas API for signatures
- Nodemailer for emails

---

## Quick Start

```bash
# 1. Module is already integrated - just initialize database
curl -X POST http://localhost:3012/document-signing/initialize-db

# 2. Set up environment variables in .env
# 3. Navigate to a student profile
# 4. Click "Document Signing" button
# 5. Upload your first PDF document
# 6. Place signature fields and send!
```

That's it! The module is ready to use. 🎉
