# Document Delete Feature

## Overview
Added the ability for admins to delete uploaded documents from student profiles. This provides better document management and allows removal of incorrect or outdated files.

## Implementation Date
October 13, 2025

---

## Features Implemented

### 1. Delete Button in UI
**File:** `views/student_profile.ejs`

**Changes:**
- Added a delete button (trash icon) next to each uploaded document in the table
- Button only visible to admin users
- Displays document index for backend processing
- Enhanced table column width to accommodate the delete button

**UI Code:**
```html
<td>
  <a class="btn btn-sm btn-outline-primary" href="<%= u.url %>" target="_blank" rel="noopener">Open</a>
  <% if (role === 'admin') { %>
    <button class="btn btn-sm btn-outline-danger" onclick="deleteDocument(<%= student.id %>, <%= index %>, '<%= u.originalName %>')">
      <i class="bi bi-trash"></i>
    </button>
  <% } %>
</td>
```

### 2. Client-Side Delete Function
**File:** `views/student_profile.ejs`

**Features:**
- SweetAlert2 confirmation dialog before deletion
- Shows document filename in confirmation
- Warning about irreversible action
- Success notification after deletion
- Automatic page reload to reflect changes
- Error handling with user-friendly messages

**JavaScript Function:**
```javascript
async function deleteDocument(studentId, documentIndex, fileName) {
  const confirmed = await Swal.fire({
    title: 'Delete Document?',
    html: `Are you sure you want to delete <strong>${fileName}</strong>?<br><span class="text-danger">This action cannot be undone.</span>`,
    icon: 'warning',
    showCancelButton: true,
    confirmButtonColor: '#dc3545',
    cancelButtonColor: '#6c757d',
    confirmButtonText: 'Yes, Delete',
    cancelButtonText: 'Cancel'
  });

  if (confirmed.isConfirmed) {
    try {
      const response = await fetch(`/admin/students/${studentId}/delete-doc/${documentIndex}`, {
        method: 'DELETE',
        headers: { 'Content-Type': 'application/json' }
      });

      if (response.ok) {
        await Swal.fire({
          title: 'Deleted!',
          text: 'Document has been deleted successfully.',
          icon: 'success',
          timer: 2000,
          showConfirmButton: false
        });
        window.location.reload();
      } else {
        throw new Error('Failed to delete document');
      }
    } catch (error) {
      Swal.fire({
        title: 'Error',
        text: error.message || 'Failed to delete document. Please try again.',
        icon: 'error'
      });
    }
  }
}
```

### 3. Backend Delete Route
**File:** `routes/admin.js`

**Route:** `DELETE /admin/students/:id/delete-doc/:index`

**Functionality:**
- Validates student exists and is a student role
- Validates document index is valid
- Removes document from uploads array
- Updates student profile in database
- Optionally deletes physical file from filesystem
- Returns JSON success response

**Implementation:**
```javascript
router.delete('/students/:id/delete-doc/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const docIndex = Number(req.params.index);
    
    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    const profile = student.profile || {};
    const uploads = Array.isArray(profile.uploads) ? profile.uploads : [];
    
    if (docIndex < 0 || docIndex >= uploads.length) {
      return res.status(400).send('Invalid document index');
    }
    
    // Remove the document from the array
    const deletedDoc = uploads.splice(docIndex, 1)[0];
    
    // Update the profile with the new uploads array
    await userModel.updateProfile(studentId, { uploads });
    
    // Optional: Delete the physical file from the filesystem
    if (deletedDoc && deletedDoc.url) {
      const fs = require('fs');
      const path = require('path');
      const filePath = path.join(__dirname, '..', 'public', deletedDoc.url);
      
      fs.unlink(filePath, (err) => {
        if (err) {
          console.warn('Failed to delete physical file:', filePath, err);
        } else {
          console.log('Deleted physical file:', filePath);
        }
      });
    }
    
    return res.status(200).json({ success: true, message: 'Document deleted successfully' });
  } catch (error) {
    console.error('Error deleting document:', error);
    return res.status(500).send('Failed to delete document');
  }
});
```

---

## User Flow

### Admin Deletes a Document

1. **Admin navigates to student profile page**
   - Views the "Uploaded Documents" section
   - Sees a table with all uploaded documents

2. **Admin clicks the trash icon button**
   - Button appears next to each document's "Open" button
   - Red outline indicates destructive action

3. **Confirmation dialog appears**
   - Shows document filename
   - Warns that action cannot be undone
   - Provides "Yes, Delete" and "Cancel" options

4. **Admin confirms deletion**
   - Click "Yes, Delete" to proceed
   - Click "Cancel" to abort

5. **Document is deleted**
   - Success message appears briefly
   - Page automatically reloads
   - Document no longer appears in the list
   - Physical file removed from server

6. **Database updated**
   - Student profile updated with new uploads array
   - Document metadata removed from database

---

## Security & Validation

### Access Control
- ✅ Only admin users can see delete buttons
- ✅ Only admin routes can delete documents
- ✅ Student validation in backend route

### Input Validation
- ✅ Student ID must be valid number
- ✅ Student must exist in database
- ✅ Student role must be 'student'
- ✅ Document index must be within array bounds
- ✅ Document index must be non-negative

### Error Handling
- ✅ 404 error if student not found
- ✅ 400 error if invalid document index
- ✅ 500 error for database/filesystem failures
- ✅ Graceful handling of missing physical files
- ✅ User-friendly error messages

---

## Technical Details

### Database Changes
- Uses existing `updateProfile()` method from userModel
- Updates `profile.uploads` array in `mdtslms_users` table
- No schema changes required

### File System Cleanup
- Physical files deleted from `/public/docs/stxd/` directory
- Deletion failure logged but doesn't fail the request
- Prevents orphaned files accumulating on server

### HTTP Method
- Uses `DELETE` method for RESTful API design
- Returns JSON response for AJAX handling

---

## Benefits

✅ **Better Document Management**
- Remove incorrect uploads
- Delete outdated documents
- Clean up duplicate files

✅ **User-Friendly**
- Clear confirmation dialog
- Shows document filename
- Success/error feedback
- Automatic UI update

✅ **Safe Operation**
- Requires explicit confirmation
- Validates all inputs
- Handles errors gracefully
- Logs all actions

✅ **Clean Implementation**
- RESTful design
- Minimal code changes
- Reuses existing models
- No database migrations needed

---

## Testing Checklist

- [x] Delete button appears only for admin users
- [x] Confirmation dialog displays correctly
- [x] Cancel button aborts deletion
- [x] Successful deletion removes document from UI
- [x] Physical file deleted from filesystem
- [x] Database updated correctly
- [x] Error handling works for invalid indices
- [x] Error handling works for missing students
- [x] Page reloads after successful deletion
- [x] Multiple documents can be deleted sequentially

---

## Future Enhancements (Optional)

### Soft Delete
Instead of permanently deleting, mark as deleted with timestamp:
```javascript
deletedDoc.deletedAt = new Date().toISOString();
deletedDoc.deletedBy = req.session.userId;
uploads[docIndex] = deletedDoc;
```

### Deletion History
Track what was deleted and when:
```javascript
const deletionHistory = profile.deletionHistory || [];
deletionHistory.push({
  fileName: deletedDoc.originalName,
  deletedAt: new Date().toISOString(),
  deletedBy: req.session.userId
});
```

### Bulk Delete
Allow admins to select and delete multiple documents at once.

### Recycle Bin
Move deleted documents to a recycle bin with 30-day retention before permanent deletion.

### Audit Log
Create a separate audit log table to track all document operations (upload, view, delete).

---

## Related Files

- `views/student_profile.ejs` - UI with delete button
- `routes/admin.js` - Backend delete route
- `models/userModel.js` - Profile update method
- `DOCUMENT_UPLOAD_FIX.md` - Original upload implementation

---

## Summary

The document delete feature provides admins with a safe, user-friendly way to remove uploaded documents from student profiles. The implementation follows best practices with proper validation, confirmation dialogs, error handling, and automatic cleanup of physical files. This completes the document management workflow: upload, view, and delete.
