# Career Advisor Interview Scheduling System

## Overview
Comprehensive career advisor interview scheduling system that allows students to book career guidance appointments and administrators to manage, confirm, and track these interviews.

## Implementation Date
November 16, 2025

---

## Features Implemented

### Student-Facing Features

#### 1. Dashboard Button
**File:** `views/student_dashboard.ejs`
- Added "Schedule Career Interview" button with calendar icon
- Prominently placed in the toolbar area for easy access
- Routes to `/student/career-advisor`

#### 2. Interview Scheduling Page
**File:** `views/career_advisor_schedule.ejs`

**Features:**
- Interactive calendar showing available interview dates
- Only shows valid dates (15th and 30th of each month, or nearest weekdays)
- Time slot selection (8 AM - 5 PM in 1-hour increments)
- Pre-filled student information (name, email, phone from profile)
- Required fields:
  - Name
  - Email
  - Phone number
  - Date selection
  - Time selection
  - Area of interest (dropdown)
  - Resume upload (PDF, Word, Pages)
- Optional field:
  - Brief summary of discussion topics

**Available Interest Areas:**
- General Career Guidance
- Resume Review
- Interview Preparation
- Job Search Strategies
- Networking & LinkedIn
- Salary Negotiation
- Career Change/Transition
- Industry Insights
- Other

**Validation:**
- All required fields enforced
- Date must be on approved interview days (weekdays near 15th/30th)
- Date must be within 40 days from today
- Time must be during business hours (8 AM - 5 PM)
- Resume file type validation (PDF, .doc, .docx, .pages)
- File size limit: 10MB

#### 3. Confirmation System
- Students receive immediate confirmation upon submission
- Automated email sent to student with interview details
- All admins receive notification email when student schedules
- Notification emails BCC'd to differentcoders@gmail.com and noreply@mdts-apps.com
- Second email sent when admin confirms with meeting link and advisor name

### Administrative Features

#### 1. Dashboard Integration
**File:** `views/admin_dashboard.ejs`
- Added "Career Advisor Interviews" link under "Other Tools" section
- Includes help tooltip explaining the feature

#### 2. Calendar Integration
**File:** `routes/admin.js` - `buildCalendarItems()` function
- Career advisor interviews automatically appear on admin dashboard calendar
- Displayed with type "career-advisor"
- Shows student name and interest area
- Clickable to navigate to management page
- Color-coded alongside other calendar events

#### 3. Interview Management Page
**File:** `views/admin_career_interviews.ejs`

**Tab Organization:**
- **Upcoming Tab:** Shows all future interviews (sorted by date)
- **Pending Tab:** Interviews awaiting confirmation
- **Confirmed Tab:** Interviews with confirmation sent
- **All Tab:** Complete list of all interviews

**Interview Card Display:**
Each interview shows:
- Student name and status badge
- Date and time prominently displayed
- Contact information (email, phone)
- Interest area
- Optional summary
- Resume download link
- Meeting link (if added)
- Admin notes (if any)

**Actions Available:**
1. **Send Confirmation Email**
   - Add career advisor name
   - Add meeting link (optional)
   - Automatically emails student
   - Updates status to "confirmed"

2. **Add Meeting Link**
   - For already confirmed interviews
   - Sends email to student with link
   - Updates interview record

3. **Mark Complete**
   - Changes status to "completed"
   - Tracks in student interactions

4. **Cancel Interview**
   - Changes status to "cancelled"
   - Visible but dimmed in interface

5. **Add Admin Notes**
   - Internal notes for staff reference
   - Not visible to students

### Backend Components

#### 1. Data Model
**File:** `models/careerAdvisorInterviewModel.js`

**Data Structure:**
```javascript
{
  id: Number (timestamp),
  studentId: Number,
  studentName: String,
  studentEmail: String,
  studentPhone: String,
  date: String (YYYY-MM-DD),
  time: String (12-hour format with AM/PM),
  interestArea: String,
  summary: String (optional),
  resumePath: String (file path),
  status: String ('pending'|'confirmed'|'completed'|'cancelled'),
  confirmationSent: Boolean,
  meetingLink: String (optional),
  advisorName: String (optional),
  notes: String (admin notes),
  createdAt: ISO String,
  updatedAt: ISO String
}
```

**Model Functions:**
- `getAll()` - Get all interviews
- `getById(id)` - Get specific interview
- `getByStudentId(studentId)` - Get all interviews for a student
- `getByDate(date)` - Get interviews on specific date
- `create(data)` - Create new interview
- `update(id, updates)` - Update interview
- `markConfirmationSent(id, advisorName, meetingLink)` - Send confirmation
- `deleteInterview(id)` - Delete interview
- `getUpcoming()` - Get future interviews
- `getForCalendar(startDate, endDate)` - Get interviews for date range
- `isTimeSlotAvailable(date, time)` - Check availability
- `isValidInterviewDate(dateString)` - Validate date
- `getNextAvailableDates(monthsAhead)` - Get valid dates

**Date Validation Logic:**
- Interviews can only be scheduled on weekdays
- Must be on or near the 15th of the month (days 13-16)
- Must be on or near the end of the month (days 28-31)
- If 15th/30th falls on weekend, uses nearest Friday
- Limited to 40 days out from current date
- Returns available dates within the 40-day window

#### 2. Student Routes
**File:** `routes/student.js`

**Routes Added:**
- `GET /student/career-advisor` - Display scheduling page
- `POST /student/career-advisor/schedule` - Submit interview request

**Resume Upload Configuration:**
- Storage: `uploads/` directory
- Filename: `resume-[timestamp]-[random].ext`
- Allowed formats: PDF, Word (.doc, .docx), Pages
- Size limit: 10MB
- Uses multer for file handling

**Integration:**
- Adds interview to student's interactions/lead tracking
- Sends confirmation email to student
- Validates all input fields
- Handles file upload errors gracefully

#### 3. Admin Routes
**File:** `routes/admin.js`

**Routes Added:**
- `GET /admin/career-advisor-interviews` - View all interviews
- `POST /admin/career-advisor-interviews/:id/confirm` - Send confirmation
- `POST /admin/career-advisor-interviews/:id/add-link` - Add meeting link
- `POST /admin/career-advisor-interviews/:id/status` - Update status
- `POST /admin/career-advisor-interviews/:id/notes` - Add notes

**Email Templates:**
1. Initial submission confirmation (to student)
2. Interview confirmation with advisor details (to student)
3. Meeting link notification (to student)
4. Admin notification when student schedules (to all admins, BCC: differentcoders@gmail.com, noreply@mdts-apps.com)

#### 4. Data Storage
**File:** `data/careerAdvisorInterviews.json`
- JSON file storage for interview data
- Automatically created on first use
- Persistent across server restarts

### Student Interactions Tracking

All career advisor interview activity is automatically logged to the student's interactions section:
- Initial interview request
- Confirmation sent
- Status changes
- Completion/cancellation

This provides a complete audit trail visible in:
- Student profile page (for admins)
- Interactions/leads tracking system

---

## User Workflows

### Student Workflow

1. Student clicks "Schedule Career Interview" button on dashboard
2. Views available dates (automatically calculated)
3. Selects desired date from calendar grid
4. Selects time slot (8 AM - 5 PM)
5. Confirms/updates contact information
6. Selects area of interest from dropdown
7. Optionally adds summary of discussion topics
8. Uploads resume (required)
9. Submits form
10. Receives immediate on-screen confirmation
11. Receives email confirmation with interview details
12. Later receives second email with advisor name and meeting link

### Admin Workflow

1. Admin views scheduled interviews via dashboard link
2. Reviews interview details including:
   - Student information
   - Date/time
   - Interest area
   - Resume
   - Summary
3. Downloads and reviews student resume
4. Clicks "Send Confirmation Email"
5. Enters career advisor name
6. Optionally adds meeting link (Zoom, Teams, etc.)
7. Submits confirmation
8. System sends email to student
9. Interview appears on admin calendar
10. Can add meeting link later if not initially available
11. Can add admin notes for internal reference
12. Marks interview as complete after it occurs

---

## Technical Details

### Route Paths

**Student Routes:**
```
GET  /student/career-advisor
POST /student/career-advisor/schedule
```

**Admin Routes:**
```
GET  /admin/career-advisor-interviews
POST /admin/career-advisor-interviews/:id/confirm
POST /admin/career-advisor-interviews/:id/add-link
POST /admin/career-advisor-interviews/:id/status
POST /admin/career-advisor-interviews/:id/notes
```

### File Structure

```
models/
  careerAdvisorInterviewModel.js    (new)

routes/
  student.js                         (modified - added career advisor routes)
  admin.js                           (modified - added career advisor routes)

views/
  student_dashboard.ejs              (modified - added button)
  career_advisor_schedule.ejs        (new - student scheduling page)
  admin_dashboard.ejs                (modified - added link under Other)
  admin_career_interviews.ejs        (new - admin management page)
  partials/
    career_interview_card.ejs        (new - interview display component)

data/
  careerAdvisorInterviews.json       (new - data storage)
```

### Email Notifications

**1. Initial Submission (Student)**
- Subject: "Career Advisor Interview Scheduled"
- Content: Date, time, topic, note about confirmation coming
- Sent immediately upon form submission

**2. Confirmation (Student)**
- Subject: "Career Advisor Interview Confirmed"
- Content: Date, time, advisor name, meeting link (if available), topic
- Sent when admin confirms interview

**3. Meeting Link (Student)**
- Subject: "Meeting Link for Your Career Advisor Interview"
- Content: Meeting link, date, time reminder
- Sent when admin adds link to confirmed interview

### Calendar Integration

Career advisor interviews appear in the admin dashboard calendar with:
- Type identifier: `career-advisor-[id]`
- Title format: "Career Advisor: [Student Name]"
- Description: Interest area and advisor name
- Start/end times: Based on interview time + 45 minutes
- Link: `/admin/career-advisor-interviews`

### Status Flow

```
pending → confirmed → completed
    ↓
cancelled (can be set at any time)
```

### Security Considerations

- Student routes protected by student/admin session middleware
- Admin routes protected by admin-only middleware
- File uploads validated for type and size
- Email addresses validated
- Date/time validation prevents invalid scheduling
- Student can only schedule for themselves (uses session data)

---

## Additional Features Implemented

### Date Selection Intelligence
- Automatically calculates next 6 months of valid dates
- Handles weekends by selecting nearest Friday
- Prevents selection of past dates
- Shows day name, date, month, and year clearly

### User Experience Enhancements
- Visual feedback when date/time selected
- Form validation with helpful error messages
- Pre-filled contact information from student profile
- Automatic phone number formatting
- Resume file type indicator
- Success/error messages with dismissible alerts
- Tab-based organization for admin interface
- Color-coded interview status badges
- Responsive design for mobile/tablet

### Status Badges
- **Pending:** Yellow badge
- **Confirmed:** Green badge
- **Completed:** Gray badge
- **Cancelled:** Red badge

### Resume File Support
- PDF (.pdf)
- Microsoft Word (.doc, .docx)
- Apple Pages (.pages)

---

## Future Enhancement Possibilities

1. **Calendar Sync**
   - Export to Google Calendar
   - Export to Outlook
   - ICS file generation

2. **Notifications**
   - Reminder emails (24 hours before)
   - SMS notifications
   - Push notifications

3. **Analytics**
   - Most popular interest areas
   - Interview completion rates
   - Student satisfaction surveys

4. **Advanced Scheduling**
   - Multiple advisor calendars
   - Availability blocking
   - Recurring availability patterns

5. **Video Integration**
   - Automatic Zoom meeting creation
   - Teams/Meet integration
   - In-platform video calls

6. **Document Management**
   - Multiple document uploads
   - Document versioning
   - Post-interview notes upload

---

## Testing Checklist

### Student Interface
- ✅ Button appears on dashboard
- ✅ Scheduling page loads correctly
- ✅ Available dates display properly
- ✅ Date selection works
- ✅ Time selection works
- ✅ Form validation functions
- ✅ Resume upload works
- ✅ Success message appears
- ✅ Email sent to student

### Admin Interface
- ✅ Link appears under "Other Tools"
- ✅ Interview list page loads
- ✅ Tabs switch correctly
- ✅ Interview cards display all information
- ✅ Confirmation modal works
- ✅ Meeting link modal works
- ✅ Status updates work
- ✅ Notes saving works
- ✅ Emails sent correctly

### Calendar Integration
- ✅ Interviews appear on calendar
- ✅ Date/time correct
- ✅ Clickable links work
- ✅ Event descriptions accurate

### Data Persistence
- ✅ Interviews saved to JSON file
- ✅ Updates persist across page loads
- ✅ File uploads stored correctly

---

## Related Files

**Models:**
- `models/careerAdvisorInterviewModel.js` - Core data model and business logic

**Views:**
- `views/student_dashboard.ejs` - Student button placement
- `views/career_advisor_schedule.ejs` - Student scheduling interface
- `views/admin_dashboard.ejs` - Admin navigation link
- `views/admin_career_interviews.ejs` - Admin management interface
- `views/partials/career_interview_card.ejs` - Interview card component

**Routes:**
- `routes/student.js` - Student-facing routes
- `routes/admin.js` - Admin management routes

**Data:**
- `data/careerAdvisorInterviews.json` - Interview data storage

---

## Summary

The Career Advisor Interview Scheduling System provides a complete end-to-end solution for students to book career guidance appointments and for administrators to manage these interviews efficiently. The system includes:

- **Intelligent date selection** that enforces business rules
- **Comprehensive validation** at every step
- **Automated email notifications** to keep everyone informed
- **Full calendar integration** for administrators
- **Student interaction tracking** for complete audit trails
- **Flexible status management** for interview lifecycle
- **Secure file uploads** for resume handling
- **Responsive design** for all device types
- **Professional email templates** for all notifications

All features are production-ready and fully integrated with the existing LMS architecture.
