# Assistant Teacher Feature Implementation

## Overview
Added support for multiple teachers per class by allowing classes to have an optional assistant teacher. Assistant teachers have the same access and abilities as the main teacher for their assigned classes.

## Changes Made

### 1. Database Schema (`models/classModel.js`)
- **Added `assistantTeacherId` column** to `mdtslms_classes` table
- Created migration function `ensureAssistantTeacherIdColumn()` to automatically add the column if it doesn't exist
- Column type: `INT NULL` (optional foreign key to teacher user ID)

### 2. Class Model Functions (`models/classModel.js`)
- **`createClass()`**: Now accepts `assistantTeacherId` parameter and saves it to the database
- **`updateClass()`**: Now accepts `assistantTeacherId` parameter and updates it in the database
- **`byTeacher()`**: Modified to return classes where the user is either the main teacher OR assistant teacher
  - Query changed to: `WHERE teacherId = ? OR assistantTeacherId = ?`
  - This ensures assistant teachers see their assigned classes in their dashboard
- **`duplicateClass()`**: Updated to copy the `assistantTeacherId` when duplicating a class

### 3. Admin Routes (`routes/admin.js`)
- **POST `/classes`** (Create Class): 
  - Extracts `assistantTeacherId` from form data
  - Passes it to `classModel.createClass()`
- **POST `/classes/:id/edit`** (Update Class):
  - Extracts `assistantTeacherId` from form data
  - Passes it to `classModel.updateClass()`
- **GET `/classes/:id`** (View Class):
  - Loads assistant teacher data using `userModel.findById(klass.assistantTeacherId)`
  - Passes `assistantTeacher` to the view template

### 4. Admin Views

#### `views/create_class.ejs`
Added assistant teacher dropdown after the main teacher field:
```html
<div class="mb-3">
  <label class="form-label">Assistant Teacher <span class="text-muted small">(Optional)</span></label>
  <select name="assistantTeacherId" class="form-select">
    <option value="">None</option>
    <% teachers.forEach(t => { %>
      <option value="<%= t.id %>"><%= t.name %> (@<%= t.username %>)</option>
    <% }) %>
  </select>
</div>
```

#### `views/edit_class.ejs`
Added assistant teacher dropdown with pre-selected value:
```html
<div class="mb-3">
  <label class="form-label">Assistant Teacher <span class="text-muted small">(Optional)</span></label>
  <select name="assistantTeacherId" class="form-select">
    <option value="">None</option>
    <% teachers.forEach(t => { %>
      <option value="<%= t.id %>" <%= Number(klass.assistantTeacherId) === Number(t.id) ? 'selected' : '' %>><%= t.name %> (@<%= t.username %>)</option>
    <% }) %>
  </select>
</div>
```

#### `views/view_class.ejs`
Updated the instructor display to show assistant teacher when present:
```html
<p class="page-meta mb-0">
  Instructor: <a href="..."><%= teacher.name %></a>
  <% if (typeof assistantTeacher !== 'undefined' && assistantTeacher) { %>
    &nbsp;•&nbsp; Assistant: <a href="..."><%= assistantTeacher.name %></a>
  <% } %>
</p>
```

#### `views/class_list.ejs`
Enhanced teacher column to display assistant teacher below main teacher:
```html
<td data-sort-value="...">
  <%= teacherMap[k.teacherId] || 'Unknown' %>
  <% if (k.assistantTeacherId && teacherMap[k.assistantTeacherId]) { %>
    <br><small class="text-muted">Asst: <%= teacherMap[k.assistantTeacherId] %></small>
  <% } %>
</td>
```

## Key Features

### 1. **Automatic Database Migration**
The system automatically adds the `assistantTeacherId` column when any class operation is performed, ensuring backward compatibility.

### 2. **Full Teacher Access**
Assistant teachers have complete access to their assigned classes:
- View and manage students
- Add/edit lectures, assignments, tests, and simulations
- Grade students
- Take attendance
- Access all class resources

### 3. **Optional Field**
The assistant teacher field is completely optional. Classes can function normally with or without an assistant teacher.

### 4. **Admin Control**
Only administrators can assign or change assistant teachers through the create/edit class forms.

### 5. **Teacher Dashboard Access**
Assistant teachers will see their assigned classes in their teacher dashboard alongside any classes where they are the main teacher.

## Usage

### For Administrators:
1. **Create New Class with Assistant Teacher:**
   - Go to Admin → Classes → Create New Class
   - Fill in the required fields
   - Select a teacher from the "Teacher" dropdown
   - Optionally select an assistant teacher from the "Assistant Teacher" dropdown
   - Complete the form and submit

2. **Add Assistant Teacher to Existing Class:**
   - Go to Admin → Classes → Select a class
   - Click "Edit"
   - Select a teacher from the "Assistant Teacher" dropdown
   - Save changes

3. **Remove Assistant Teacher:**
   - Edit the class
   - Set the "Assistant Teacher" dropdown to "None"
   - Save changes

### For Teachers (including Assistant Teachers):
- No changes needed in workflow
- Assistant teachers will automatically see their assigned classes in their dashboard
- All teaching features work identically for both main and assistant teachers

## Technical Notes

- **Data Type**: `assistantTeacherId` is stored as `INT NULL` in the database
- **Null Values**: When no assistant teacher is assigned, the field is set to `NULL`
- **Backward Compatibility**: Existing classes without an assistant teacher will continue to work normally
- **Performance**: Minimal impact - only adds one JOIN operation when querying classes by teacher
- **Validation**: No validation prevents assigning the same person as both main and assistant teacher (this could be added if needed)

## Future Enhancements (Optional)
- Prevent assigning the same teacher as both main and assistant
- Add role differentiation (e.g., assistant can view but not delete)
- Add notification to assistant teachers when assigned to a class
- Add class history to show when assistant teachers were added/removed
