const userModel = require('../models/userModel');
const classModel = require('../models/classModel');

/**
 * Auto-transition students to alumni status
 * Run this script daily (via cron) to check for students who should be moved to alumni
 * 
 * Criteria: Students with status='approved' who have no active classes and 
 * their last class ended more than 7 days ago
 */
async function transitionToAlumni() {
  console.log('Starting auto-transition to alumni process...');
  
  try {
    // Get all approved students
    const allUsers = await userModel.getAll();
    const approvedStudents = allUsers.filter(u => u.role === 'student' && u.status === 'approved');
    
    console.log(`Found ${approvedStudents.length} approved students to check`);
    
    // Get all classes
    const allClasses = await classModel.getAllClasses();
    
    const now = new Date();
    const oneWeekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
    
    let transitionedCount = 0;
    
    for (const student of approvedStudents) {
      // Find all classes this student is enrolled in
      const studentClasses = allClasses.filter(klass => 
        (klass.studentIds || []).includes(student.id)
      );
      
      if (studentClasses.length === 0) {
        // Student not enrolled in any classes - skip for now
        // (They might be waiting to be enrolled)
        continue;
      }
      
      // Check if student has any active classes
      const hasActiveClass = studentClasses.some(klass => {
        if (!klass.endDate) return true; // No end date = still active
        const endDate = new Date(klass.endDate);
        return endDate > now; // Class ends in the future = active
      });
      
      if (hasActiveClass) {
        // Student has active classes, don't transition
        continue;
      }
      
      // Find the most recent class end date
      let mostRecentEndDate = null;
      studentClasses.forEach(klass => {
        if (klass.endDate) {
          const endDate = new Date(klass.endDate);
          if (!mostRecentEndDate || endDate > mostRecentEndDate) {
            mostRecentEndDate = endDate;
          }
        }
      });
      
      // If we have an end date and it's more than 7 days ago, transition to alumni
      if (mostRecentEndDate && mostRecentEndDate < oneWeekAgo) {
        console.log(`Transitioning student ${student.id} (${student.email}) to alumni - last class ended ${mostRecentEndDate.toDateString()}`);
        
        await userModel.setStatus(student.id, 'alumni');
        
        // Store the timestamp when they became alumni
        const profile = student.profile || {};
        profile.becameAlumniAt = new Date().toISOString();
        profile.autoTransitionedToAlumni = true;
        await userModel.updateProfile(student.id, profile);
        
        transitionedCount++;
      }
    }
    
    console.log(`Auto-transition complete. Transitioned ${transitionedCount} students to alumni.`);
    return transitionedCount;
    
  } catch (error) {
    console.error('Error in auto-transition to alumni:', error);
    throw error;
  }
}

// If run directly (not imported as module)
if (require.main === module) {
  transitionToAlumni()
    .then(count => {
      console.log(`Done! Transitioned ${count} students.`);
      process.exit(0);
    })
    .catch(error => {
      console.error('Failed:', error);
      process.exit(1);
    });
}

module.exports = transitionToAlumni;
