const express = require('express');
const router = express.Router();
const classModel = require('../models/classModel');
const userModel = require('../models/userModel');
const db = require('../models/db');
const { generateQuestions } = require('../utils/questionGenerator');
const announcementModel = require('../models/announcementModel');
const discussionModel = require('../models/discussionModel');
const messageModel = require('../models/messageModel');
const emailTemplates = require('../utils/emailTemplates');
const testModel = require('../models/testModel');
const resourceLibrary = require('../utils/resourceLibrary');

const fs = require('fs');
const path = require('path');
let marked;
try {
  marked = require('marked');
} catch (err) {
  console.warn('[teacher manual] marked package not found, falling back to plain text manual rendering');
  marked = null;
}
const nodemailer = require('nodemailer');
const crypto = require('crypto');
const PDFDocument = require('pdfkit');

const CONFIG_PATH = path.join(__dirname, '..', 'data', 'signature-docs.json');
let signatureDocsConfig = {};

function loadSignatureDocsConfig() {
  try {
    signatureDocsConfig = JSON.parse(fs.readFileSync(CONFIG_PATH, 'utf8'));
  } catch (err) {
    console.error('Failed to load signature-docs.json:', err.message);
    signatureDocsConfig = {};
  }
}
loadSignatureDocsConfig();


const transporter = nodemailer.createTransport({
  host: 'mdts-apps.com',
  port: 465,
  secure: true,
  auth: {
    user: 'noreply@mdts-apps.com',
    pass: 'c@r,5ysPI@&s'
  }
});
const multer = require('multer');
const upload = multer();

router.get('/manual', (req, res) => {
  if (!req.session || !['teacher', 'admin'].includes(req.session.role)) {
    return res.status(403).send('Forbidden');
  }
  const mdPath = path.join(__dirname, '..', 'docs', 'teacher-manual.md');
  try {
    const markdown = fs.readFileSync(mdPath, 'utf-8');
    const { marked } = require('marked');
    marked.setOptions({
      headerIds: true,
      mangle: false
    });
    const htmlContent = marked.parse(markdown);
    const fullHtml = `
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Teacher Manual - MDTS LMS</title>
  <link rel="stylesheet" href="/styles.css">
  <style>
    body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif; line-height: 1.6; max-width: 900px; margin: 40px auto; padding: 0 20px; }
    h1 { color: #2c3e50; border-bottom: 3px solid #3498db; padding-bottom: 10px; }
    h2 { color: #34495e; margin-top: 30px; border-bottom: 2px solid #ecf0f1; padding-bottom: 8px; }
    h3 { color: #555; margin-top: 20px; }
    img { max-width: 100%; height: auto; border: 1px solid #ddd; border-radius: 4px; margin: 15px 0; }
    code { background: #f4f4f4; padding: 2px 6px; border-radius: 3px; font-family: 'Courier New', monospace; }
    pre { background: #f4f4f4; padding: 15px; border-radius: 5px; overflow-x: auto; }
    ul, ol { margin: 10px 0; padding-left: 30px; }
    li { margin: 5px 0; }
    blockquote { border-left: 4px solid #3498db; padding-left: 15px; color: #666; margin: 15px 0; }
    a { color: #3498db; text-decoration: none; }
    a:hover { text-decoration: underline; }
    .back-link { margin-bottom: 20px; }
    .back-link a { display: inline-block; padding: 8px 15px; background: #3498db; color: white; border-radius: 4px; }
    .back-link a:hover { background: #2980b9; text-decoration: none; }
  </style>
</head>
<body>
  <div class="back-link">
    <a href="/teacher">← Back to Dashboard</a>
  </div>
  ${htmlContent}
</body>
</html>
    `;
    res.send(fullHtml);
  } catch (err) {
    console.error('Failed to load teacher manual', err);
    res.status(404).send('User guide not found');
  }
});

// Storage for test media uploads
const mediaStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, path.join(__dirname, '../uploads'));
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + '-' + file.originalname);
  }
});
const mediaUpload = multer({ storage: mediaStorage });

// Storage for syllabus uploads
const syllabusStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, path.join(__dirname, '../uploads/syllabi'));
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, 'syllabus-' + uniqueSuffix + path.extname(file.originalname));
  }
});
const syllabusUpload = multer({ 
  storage: syllabusStorage,
  limits: { fileSize: 10 * 1024 * 1024 }, // 10MB
  fileFilter: (req, file, cb) => {
    const allowedTypes = /pdf|doc|docx/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);
    if (extname && mimetype) {
      return cb(null, true);
    }
    cb(new Error('Only PDF, DOC, and DOCX files are allowed'));
  }
});

router.use((req, res, next) => {
  if (!req.session || !['teacher', 'admin'].includes(req.session.role)) return res.status(403).send('Forbidden');
  next();
});

// Parse JSON bodies for AJAX contact
router.use(express.json());

// Download CSV template for test uploads
router.get('/tests/template.csv', (_req, res) => {
  const header = [
    'Question','Answer','Explanation','Picture',
    'OptionA','OptionB','OptionC','OptionD','OptionE','OptionF','OptionG',
    'Test','Content Type','Title','Item Type','Path'
  ].join(',') + '\n';
  res.setHeader('Content-Type', 'text/csv; charset=utf-8');
  res.setHeader('Content-Disposition', 'attachment; filename="test_template.csv"');
  res.send(header);
});

router.get('/profile', async (req, res) => {
  const teacher = await userModel.findById(req.session.user.id);
  res.render('teacher_profile', { teacher, role: 'teacher', saved: req.query.saved });
});

router.post('/profile', mediaUpload.single('photo'), async (req, res) => {
  const id = req.session.user.id;
  if (req.file) {
    await userModel.updateProfile(id, {
      photo: {
        url: `/uploads/${req.file.filename}`,
        originalName: req.file.originalname
      }
    });
  }
    // Optional drawn signature
  if (req.body && req.body.signatureDataUrl && req.body.signatureDataUrl.startsWith('data:image/')) {
    await userModel.updateProfile(id, { signature: { url: req.body.signatureDataUrl } });
  }
  res.redirect('/teacher/profile?saved=1');
});

router.post('/profile/links', mediaUpload.single('image'), async (req, res) => {
  const { url } = req.body;
  if (url) {
    const link = { url };
    if (req.file) {
      link.image = { url: `/uploads/${req.file.filename}`, originalName: req.file.originalname };
    }
    try { await userModel.addLinks(req.session.user.id, [link]); }
    catch (e) { console.error('add link', e); }
  }
    // Optional drawn signature
  if (req.body && req.body.signatureDataUrl && req.body.signatureDataUrl.startsWith('data:image/')) {
    await userModel.updateProfile(id, { signature: { url: req.body.signatureDataUrl } });
  }
  res.redirect('/teacher/profile?saved=1');
});


// Dashboard with weekly schedule
router.get('/', async (req, res) => {
  const classes = await classModel.byTeacher(req.session.user.id);
  const users = await userModel.getAll();
  const students = users.filter(u => u.role === 'student');
  const pendingStudents = students.filter(u => u.status === 'pending' || !u.status);
  const approvedStudents = students.filter(u => u.status === 'approved');
  const classSizes = classes.map(c => ({ name: c.name, size: (c.studentIds || []).length }));
  const weekly = { Monday:[], Tuesday:[], Wednesday:[], Thursday:[], Friday:[], Saturday:[], Sunday:[] };
  classes.forEach(k => (k.schedule||[]).forEach(s => {
    if (weekly[s.day]) weekly[s.day].push({ className: k.name, start: s.start, end: s.end });
  }));
  res.render('teacher_dashboard', {
    teacher: req.session.user,
    classes,
    weekly,
    pendingCount: pendingStudents.length,
    approvedCount: approvedStudents.length,
    classSizes
  });
});

router.get('/announcements', async (req, res) => {
  const classes = await classModel.byTeacher(req.session.user.id);
  const announcements = await announcementModel.forTeacher(req.session.user.id);
  res.render('teacher_announcements', { teacher: req.session.user, classes, announcements });
});

router.get('/simplify', (req, res) => {
  res.render('topic_helper', { user: { role: 'teacher' }, result: null, topic: '' });
});

router.post('/simplify', async (req, res) => {
  const { topic } = req.body;
  const apiKey = process.env.OPENAI_API_KEY;
  let result = '';
  if (!apiKey) result = 'Missing OpenAI API key';
  if (!topic) result = 'Topic is required';
  if (!result) {
    try {
      const response = await fetch('https://api.openai.com/v1/chat/completions', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${apiKey}`
        },
        body: JSON.stringify({
          model: 'gpt-3.5-turbo',
          messages: [{ role: 'user', content: `Explain the following in simple terms for a 5th grade student: ${topic}` }]
        })
      });
      const data = await response.json();
      result = data.choices?.[0]?.message?.content || 'No summary generated';
    } catch (e) {
      console.error('OpenAI error', e);
      result = 'Error generating summary';
    }
  }
  res.render('topic_helper', { user: { role: 'teacher' }, result, topic });
});


router.get('/mailbox', async (req, res) => {
  const messages = await messageModel.getMailbox(req.session.user.id);
  const allUsers = await userModel.getAll();
  const userMap = new Map(allUsers.map(u => [u.id, u]));
  const formatted = messages.map(m => ({
    ...m,
    fromName: userMap.get(m.senderId)?.name || `User ${m.senderId}`,
    toName: userMap.get(m.recipientId)?.name || `User ${m.recipientId}`
  }));
  const students = allUsers.filter(u => u.role === 'student');
  res.render('mailbox', { messages: formatted, users: students, user: req.session.user });
});

router.post('/mailbox', async (req, res) => {
  const recipientId = Number(req.body.to);
  const { subject, body } = req.body;
  if (recipientId && body) {
    await messageModel.sendMessage(req.session.user.id, recipientId, subject || '', body);
  }
  res.redirect('/teacher/mailbox');
});

// Contact a single student (AJAX)
router.post('/students/:id/contact', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    if (!student || !student.email) return res.status(404).json({ error: 'Student not found' });
    const { subject, message } = req.body || {};
    if (!subject || !message) return res.status(400).json({ error: 'Missing subject or message' });
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: student.email,
      subject,
      html: message,
      text: String(message).replace(/<[^>]*>/g, '')
    });
    const now = new Date();
    try { await userModel.setLastContacted(id, now); } catch(_) {}
    return res.json({ ok: true, lastContacted: now.toISOString() });
  } catch (e) {
    console.error('Teacher student contact error', e);
    return res.status(500).json({ error: 'Failed to send' });
  }
});

// Send welcome email to student (teacher)
router.post('/students/:id/send-welcome', async (req, res) => {
  try {
    const id = Number(req.params.id);
    const student = await userModel.findById(id);
    if (!student || !student.email) return res.status(404).json({ error: 'Student not found' });
    
    const { classId, className } = req.body || {};
    
    // Use the studentWelcome email template
    const { subject, html, text } = emailTemplates.render('studentWelcome', {
      name: student.name || 'Student',
      className: className || 'your class',
      classId: classId || ''
    });
    
    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: student.email,
      subject,
      html,
      text
    });
    
    const now = new Date();
    
    // Update last contacted
    try { await userModel.setLastContacted(id, now); } catch(_) {}
    
    // Log interaction in lead contacts
    try {
      const leadModel = require('../models/leadModel');
      const lead = await leadModel.upsertByEmail({ 
        name: student.name || '', 
        email: student.email, 
        phone: student.profile?.phones?.cell || '', 
        interestPercent: 0, 
        source: 'teacher-welcome' 
      });
      if (lead?.id) {
        await leadModel.addContact(lead.id, { 
          direction: 'outbound', 
          method: 'email', 
          note: `Welcome email sent for ${className || 'class'}` 
        });
      }
    } catch (e) { 
      console.error('Lead log failed', e); 
    }
    
    return res.json({ ok: true, sentAt: now.toISOString() });
  } catch (e) {
    console.error('Teacher send welcome email error', e);
    return res.status(500).json({ error: 'Failed to send welcome email' });
  }
});

function dataUrlToBuffer(url) {
  const match = /^data:.+;base64,(.+)$/.exec(url || '');
  return match ? Buffer.from(match[1], 'base64') : null;
}

function buildCompletionPdf({ studentName, courseName, dateStr, instructorSig, directorSig, instructorName, directorName, schoolName, certificateId }) {
  return new Promise((resolve) => {
    const doc = new PDFDocument({ 
      size: 'LETTER', 
      layout: 'landscape', 
      margin: 0,
      bufferPages: false
    });
    const buffers = [];
    doc.on('data', buffers.push.bind(buffers));
    doc.on('end', () => resolve(Buffer.concat(buffers)));

    const { width, height } = doc.page;
    const centerX = width / 2;

    // Background gradient effect (multiple rectangles with opacity)
    doc.rect(0, 0, width, height).fill('#1e3c72');
    doc.rect(0, 0, width, height).fillOpacity(0.5).fill('#2a5298');
    doc.fillOpacity(1);

    // Outer border
    doc.lineWidth(3).strokeColor('#ffffff').opacity(0.3)
       .rect(40, 40, width - 80, height - 80).stroke();
    
    // Inner border
    doc.lineWidth(1).strokeColor('#ffffff').opacity(0.2)
       .rect(50, 50, width - 100, height - 100).stroke();
    doc.opacity(1);

    // Corner decorations
    const cornerSize = 60;
    const cornerOffset = 30;
    doc.lineWidth(3).strokeColor('#ffffff').opacity(0.4);
    // Top-left
    doc.moveTo(cornerOffset, cornerOffset + cornerSize).lineTo(cornerOffset, cornerOffset).lineTo(cornerOffset + cornerSize, cornerOffset).stroke();
    // Top-right
    doc.moveTo(width - cornerOffset - cornerSize, cornerOffset).lineTo(width - cornerOffset, cornerOffset).lineTo(width - cornerOffset, cornerOffset + cornerSize).stroke();
    // Bottom-left
    doc.moveTo(cornerOffset, height - cornerOffset - cornerSize).lineTo(cornerOffset, height - cornerOffset).lineTo(cornerOffset + cornerSize, height - cornerOffset).stroke();
    // Bottom-right
    doc.moveTo(width - cornerOffset - cornerSize, height - cornerOffset).lineTo(width - cornerOffset, height - cornerOffset).lineTo(width - cornerOffset, height - cornerOffset - cornerSize).stroke();
    doc.opacity(1);

    // School name
    doc.fontSize(20).fillColor('#ffffff').font('Helvetica-Bold')
       .text(schoolName || 'MD Technical School', 0, 80, { align: 'center', width: width });
    doc.fontSize(11).fillColor('#ffffff').font('Helvetica')
       .text('EXCELLENCE IN TECHNICAL EDUCATION', 0, 108, { align: 'center', width: width });

    // Certificate title
    doc.fontSize(48).fillColor('#ffd700').font('Times-Bold')
       .text('CERTIFICATE', 0, 145, { align: 'center', width: width });
    doc.fontSize(16).fillColor('#ffffff').font('Helvetica')
       .text('of Achievement', 0, 200, { align: 'center', width: width });

    // Decorative line before recipient
    doc.moveTo(width * 0.2, 235).lineTo(width * 0.8, 235)
       .strokeColor('#ffffff').opacity(0.3).lineWidth(2).stroke();
    doc.opacity(1);

    // Recipient section
    doc.fontSize(14).fillColor('#ffffff').font('Helvetica')
       .text('This is proudly presented to', 0, 250, { align: 'center', width: width });
    
    doc.fontSize(36).fillColor('#ffd700').font('Times-Bold')
       .text(studentName, 0, 280, { align: 'center', width: width });

    // Decorative line after recipient
    doc.moveTo(width * 0.2, 330).lineTo(width * 0.8, 330)
       .strokeColor('#ffffff').opacity(0.3).lineWidth(2).stroke();
    doc.opacity(1);

    // Achievement text
    doc.fontSize(14).fillColor('#ffffff').font('Helvetica')
       .text('for successfully completing the requirements and demonstrating', 0, 350, { align: 'center', width: width })
       .text('exceptional proficiency in', 0, 370, { align: 'center', width: width });
    
    doc.fontSize(20).fillColor('#ffd700').font('Helvetica-Bold')
       .text(courseName, 0, 400, { align: 'center', width: width });

    // Completion date
    const yPos = 435;
    doc.fontSize(14).fillColor('#ffffff').font('Helvetica')
       .text(`Completed on ${dateStr}`, 0, yPos, { align: 'center', width: width });

    // Signatures section
    const sigY = height - 140;
    const leftSigX = width * 0.25;
    const rightSigX = width * 0.75;

    // Instructor signature
    if (instructorSig) {
      const buf = dataUrlToBuffer(instructorSig);
      if (buf) {
        try {
          doc.image(buf, leftSigX - 75, sigY - 60, { width: 150, height: 60, fit: [150, 60] });
        } catch (e) {
          console.error('Error adding instructor signature:', e);
        }
      }
    }
    doc.moveTo(leftSigX - 75, sigY).lineTo(leftSigX + 75, sigY)
       .strokeColor('#ffffff').opacity(0.4).lineWidth(2).stroke();
    doc.opacity(1);
    if (instructorName) {
      doc.fontSize(14).fillColor('#ffffff').font('Helvetica-Bold')
         .text(instructorName, leftSigX - 100, sigY + 8, { align: 'center', width: 200 });
    }
    doc.fontSize(11).fillColor('#ffffff').font('Helvetica')
       .text('INSTRUCTOR', leftSigX - 100, sigY + 26, { align: 'center', width: 200 });

    // Director signature
    if (directorSig) {
      const buf = dataUrlToBuffer(directorSig);
      if (buf) {
        try {
          doc.image(buf, rightSigX - 75, sigY - 60, { width: 150, height: 60, fit: [150, 60] });
        } catch (e) {
          console.error('Error adding director signature:', e);
        }
      }
    }
    doc.moveTo(rightSigX - 75, sigY).lineTo(rightSigX + 75, sigY)
       .strokeColor('#ffffff').opacity(0.4).lineWidth(2).stroke();
    doc.opacity(1);
    if (directorName) {
      doc.fontSize(14).fillColor('#ffffff').font('Helvetica-Bold')
         .text(directorName, rightSigX - 100, sigY + 8, { align: 'center', width: 200 });
    }
    doc.fontSize(11).fillColor('#ffffff').font('Helvetica')
       .text('PROGRAM DIRECTOR', rightSigX - 100, sigY + 26, { align: 'center', width: 200 });

    // Footer
    doc.fontSize(10).fillColor('#ffffff').font('Helvetica').opacity(0.7)
       .text(`Issued by ${schoolName || 'MD Technical School'}`, 0, height - 50, { align: 'center', width: width });
    if (certificateId) {
      doc.fontSize(9).fillColor('#ffffff').font('Courier').opacity(0.5)
         .text(`Certificate ID: ${certificateId}`, 0, height - 35, { align: 'center', width: width });
    }

    doc.end();
  });
}

// Preview certificate before sending
router.get('/classes/:classId/students/:studentId/certificate/preview', async (req, res) => {
  try {
    const classId = Number(req.params.classId);
    const studentId = Number(req.params.studentId);
    
    const student = await userModel.findById(studentId);
    if (!student) return res.status(404).send('Student not found');
    
    const klass = await classModel.findClassById(classId);
    if (!klass) return res.status(404).send('Class not found');
    
    const teacher = await userModel.findById(klass.teacherId);
    const dateStr = new Date().toLocaleDateString('en-US', { 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    });
    
    const directorSig = req.app.locals.branding?.directorSignature;
    const directorName = req.app.locals.branding?.directorName || 'Program Director';
    const instructorSig = teacher?.profile?.signature?.url || directorSig;
    const instructorName = teacher?.name || teacher?.username || 'Instructor';
    const schoolName = req.app.locals.branding?.schoolName || 'MD Technical School';
    const certificateId = `CERT-${Date.now()}-${studentId}-${classId}`;
    
    res.render('certificate_professional', {
      branding: req.app.locals.branding,
      schoolName,
      studentName: student.name || `${student.firstName || ''} ${student.lastName || ''}`.trim(),
      courseName: klass.name,
      testTitle: '',
      score: '',
      dateStr,
      instructorSig,
      instructorName,
      directorSig,
      directorName,
      credentialAwarded: 'Certificate of Completion',
      certificateId
    });
  } catch (e) {
    console.error('Certificate preview error:', e);
    res.status(500).send('Error generating certificate preview');
  }
});

// Send course completion certificate to student
router.post('/classes/:classId/students/:studentId/certificate', async (req, res) => {
  try {
    const classId = Number(req.params.classId);
    const studentId = Number(req.params.studentId);
    const student = await userModel.findById(studentId);
    if (!student || !student.email) return res.status(404).json({ error: 'Student not found' });

    const klass = await classModel.findClassById(classId);
    const teacher = klass ? await userModel.findById(klass.teacherId) : null;
    const dateStr = new Date().toLocaleDateString('en-US', { 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    });
    const directorSig = req.app.locals.branding?.directorSignature;
    const directorName = req.app.locals.branding?.directorName || 'Program Director';
    const instructorSig = teacher?.profile?.signature?.url || directorSig;
    const instructorName = teacher?.name || teacher?.username || 'Instructor';
    const schoolName = req.app.locals.branding?.schoolName || 'MD Technical School';
    const certificateId = `CERT-${Date.now()}-${studentId}-${classId}`;

    const pdfBuffer = await buildCompletionPdf({
      studentName: student.name || `${student.firstName || ''} ${student.lastName || ''}`.trim(),
      courseName: klass ? klass.name : 'Course',
      dateStr,
      instructorSig,
      directorSig,
      instructorName,
      directorName,
      schoolName,
      certificateId
    });

    const admins = await userModel.getByRole('admin');
    const cc = admins.map(a => a.email).filter(Boolean);
    const link = `${req.protocol}://${req.get('host')}/student/certificates/${classId}/completion`;

    await transporter.sendMail({
      from: 'noreply@mdts-apps.com',
      to: student.email,
      cc,
      subject: '🎓 Your Certificate of Completion is Ready!',
      html: `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
          <div style="background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%); padding: 30px; text-align: center; border-radius: 10px 10px 0 0;">
            <h1 style="color: #ffd700; margin: 0; font-size: 28px;">🎓 Congratulations!</h1>
          </div>
          <div style="background: #ffffff; padding: 30px; border-radius: 0 0 10px 10px; box-shadow: 0 4px 6px rgba(0,0,0,0.1);">
            <p style="font-size: 16px; color: #333;">Dear ${student.name || 'Student'},</p>
            <p style="font-size: 16px; color: #333; line-height: 1.6;">
              We are thrilled to inform you that you have successfully completed <strong>${klass ? klass.name : 'your course'}</strong>!
            </p>
            <p style="font-size: 16px; color: #333; line-height: 1.6;">
              Your professional certificate is attached to this email and also available for viewing online.
            </p>
            <div style="text-align: center; margin: 30px 0;">
              <a href="${link}" style="background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%); color: white; padding: 15px 40px; text-decoration: none; border-radius: 5px; font-weight: bold; display: inline-block;">
                View Certificate Online
              </a>
            </div>
            <p style="font-size: 14px; color: #666; line-height: 1.6;">
              This achievement represents your dedication and hard work. Keep this certificate for your professional records.
            </p>
            <hr style="border: none; border-top: 1px solid #eee; margin: 30px 0;">
            <p style="font-size: 12px; color: #999; text-align: center;">
              ${schoolName}<br>
              Excellence in Technical Education
            </p>
          </div>
        </div>
      `,
      text: `Congratulations ${student.name || ''}! Your certificate of completion is attached or available at ${link}`,
      attachments: [{ filename: 'certificate.pdf', content: pdfBuffer }]
    });

    try {
      await userModel.addCertificate(studentId, {
        title: 'Course Completion',
        classId,
        testId: 0,
        url: `/student/certificates/${classId}/completion`
      });
    } catch (_) {}

    return res.json({ ok: true });
  } catch (e) {
    console.error('Send completion certificate error', e);
    return res.status(500).json({ error: 'Failed to send' });
  }
});

router.post('/announcements', async (req, res) => {
  const { message, classId } = req.body;
  if (message && classId) {
    await announcementModel.create({ authorId: req.session.user.id, audience: 'class', classId: Number(classId), message });
  }
  res.redirect('/teacher/announcements');
});

router.post('/announcements/:id/delete', async (req, res) => {
  const id = Number(req.params.id);
  if (!Number.isNaN(id)) {
    await announcementModel.remove(id);
  }
  res.redirect('/teacher/announcements');
});

// Class view + roster + tests
router.get('/classes/:id', async (req, res) => {
  const klass = await classModel.findClassById(Number(req.params.id));
  if (!klass) return res.status(404).send('Not found');
  const users = await userModel.getAll();
  const students = users
    .filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id))
    .sort((a, b) => {
      // Sort alphabetically by name
      const nameA = (a.name || a.profile?.firstName || '').toLowerCase();
      const nameB = (b.name || b.profile?.firstName || '').toLowerCase();
      return nameA.localeCompare(nameB);
    });
  
  // Fetch welcome email status for each student
  const leadModel = require('../models/leadModel');
  const studentWelcomeStatus = {};
  for (const student of students) {
    if (student.email) {
      try {
        const lead = await leadModel.getByEmail(student.email);
        if (lead) {
          const contacts = await leadModel.getContacts(lead.id);
          const welcomeContact = contacts.find(c => 
            c.method === 'email' && 
            c.note && 
            c.note.toLowerCase().includes('welcome')
          );
          if (welcomeContact) {
            studentWelcomeStatus[student.id] = {
              sent: true,
              date: welcomeContact.createdAt
            };
          }
        }
      } catch (e) {
        console.error(`Failed to fetch welcome status for student ${student.id}`, e);
      }
    }
  }
  
 const today = new Date().toISOString().slice(0,10);
  const attendanceToday = (klass.attendance || []).find(a => a.date === today) || { present: [] };
  const discussions = await discussionModel.getByClass(klass.id);
  // Load available simulations from sims folder
  const fs = require('fs');
  const path = require('path');
  const simsDir = path.join(__dirname, '..', 'sims');
  let availableSims = [];
  try {
    const entries = fs.readdirSync(simsDir, { withFileTypes: true });
    availableSims = entries
      .filter(d => d.isDirectory())
      .map(d => ({ name: d.name, indexPath: path.join(simsDir, d.name, 'index.html') }))
      .filter(e => fs.existsSync(e.indexPath))
      .map(e => ({ title: e.name, url: `/sims/${e.name}/index.html` }));
  } catch (_) {}
  
  // Load tests from database
  const allTests = await testModel.getAllTests();
  const sortedTests = allTests.sort((a, b) => {
    const titleA = a.title || '';
    const titleB = b.title || '';
    return titleA.localeCompare(titleB);
  });
  
  const lib = resourceLibrary.loadLibrary();
  res.render('teacher_view_class', { 
    klass, 
    students, 
    today, 
    attendanceToday, 
    discussions, 
    availableSims, 
    availableLectures: lib.lectures, 
    availableAssignments: lib.assignments, 
    availableTests: sortedTests,
    studentWelcomeStatus
  });
});

router.post('/classes/:id/rename', async (req, res) => {
  const id = Number(req.params.id);
  const { name } = req.body;
  if (name && name.trim()) {
    await classModel.renameClass(id, name.trim());
  }
  res.redirect(`/teacher/classes/${id}`);
});

// add discussion message
router.post('/classes/:id/discussion', async (req, res) => {
  const classId = Number(req.params.id);
  const { message } = req.body;
  if (message && message.trim()) {
    await discussionModel.addMessage(classId, req.session.user.id, message.trim());
    const klass = await classModel.findClassById(classId);
    const teacher = await userModel.findById(klass.teacherId);
    if (teacher && teacher.email) {
      try {
        const { subject, html, text } = emailTemplates.render('discussionNotification', {
          klassName: klass.name,
          message: message.trim()
        });
        await transporter.sendMail({
          to: teacher.email,
          from: process.env.SMTP_USER,
          subject,
          html,
          text
        });
      } catch (e) {
        console.error('Email send failed', e);
      }
    }
  }
  res.redirect(`/teacher/classes/${classId}#discussion`);
});
router.post('/classes/:id/checklist', async (req, res) => {
  const classId = Number(req.params.id);
  const items = Array.isArray(req.body.item) ? req.body.item : (req.body.item ? [req.body.item] : []);
  const done = Array.isArray(req.body.done) ? req.body.done.map(Number) : (req.body.done ? [Number(req.body.done)] : []);
  const checklist = items.map((text, idx) => ({ text, done: done.includes(idx) }));
  await classModel.updateChecklist(classId, checklist);
  res.redirect(`/teacher/classes/${classId}`);
});

// Upload syllabus for a class
router.post('/classes/:id/syllabus', syllabusUpload.single('syllabus'), async (req, res) => {
  try {
    const classId = Number(req.params.id);
    
    if (!req.file) {
      return res.status(400).send('No file uploaded');
    }
    
    // Ensure uploads/syllabi directory exists
    const fs = require('fs');
    const syllabusDir = path.join(__dirname, '../uploads/syllabi');
    if (!fs.existsSync(syllabusDir)) {
      fs.mkdirSync(syllabusDir, { recursive: true });
    }
    
    const syllabusPath = '/uploads/syllabi/' + req.file.filename;
    const syllabusFileName = req.file.originalname;
    
    await classModel.updateSyllabus(classId, syllabusPath, syllabusFileName);
    
    res.redirect(`/teacher/classes/${classId}`);
  } catch (error) {
    console.error('Syllabus upload error:', error);
    res.status(500).send('Failed to upload syllabus');
  }
});

// Delete syllabus from a class
router.delete('/classes/:id/syllabus', async (req, res) => {
  try {
    const classId = Number(req.params.id);
    
    // Get the class to find the file path
    const klass = await classModel.findClassById(classId);
    
    if (klass.syllabusPath) {
      // Delete the physical file
      const fs = require('fs');
      const filePath = path.join(__dirname, '..', klass.syllabusPath);
      
      if (fs.existsSync(filePath)) {
        fs.unlinkSync(filePath);
      }
    }
    
    // Remove from database
    await classModel.removeSyllabus(classId);
    
    res.sendStatus(200);
  } catch (error) {
    console.error('Syllabus delete error:', error);
    res.status(500).send('Failed to delete syllabus');
  }
});


// new test creation form
router.get('/classes/:id/tests/new', async (req, res) => {
  const klass = await classModel.findClassById(Number(req.params.id));
  if (!klass) return res.status(404).send('Not found');
  res.render('create_test', { klass });
});

// generate and save a test
router.post('/classes/:id/tests', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, lecture, questionCount, optionCount, timeLimit, official } = req.body;
  const questions = generateQuestions(
    lecture || '',
    Number(questionCount) || 0,
    Number(optionCount) || 4,
    title || ''
  );
  await testModel.replaceTestQuestions(title, questions);
  await classModel.addTest(classId, { 
    title, 
    timeLimit: Number(timeLimit) || 90,
    official: official === 'on' || official === true || official === 'true' // checkbox value
  });
  // Test is now automatically available in library from database
  const lines = questions.map(q => {
    const optionCells = [];
    for (let i = 0; i < 7; i++) optionCells.push(q.options[i] || '');
    return [
      q.question,
      q.options[q.answer] || '',
      q.explanation || '',
      q.picture || '',
      ...optionCells,
      title,
      q.contentType,
      q.title,
      q.itemType,
      q.path
    ].join('\t');
  }).join('\n');
  const outDir = path.join(__dirname, '..', 'data', 'tests');
  if (!fs.existsSync(outDir)) fs.mkdirSync(outDir, { recursive: true });
  fs.writeFileSync(path.join(outDir, `${title.replace(/\s+/g, '_')}.tsv`), lines, 'utf8');
  res.redirect(`/teacher/classes/${classId}`);
});

router.post('/classes/:id/lectures', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, date, dripDay } = req.body;
  const finalUrl = url && url.trim();
  if (title && finalUrl) {
    await classModel.addLecture(classId, {
      title: title.trim(),
      url: finalUrl,
      date,
      dripDay: dripDay ? Number(dripDay) : null
    });
    try { resourceLibrary.addLecture({ title: title.trim(), url: finalUrl }); } catch (_) {}
  }
  res.redirect(`/teacher/classes/${classId}#lectures`);
});

router.post('/classes/:id/lectures/:lectureId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const lectureId = Number(req.params.lectureId);
  await classModel.removeLecture(classId, lectureId);
  res.redirect(`/teacher/classes/${classId}#lectures`);
});

router.post('/classes/:id/lectures/:lectureId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const lectureId = Number(req.params.lectureId);
  const { title, date, dripDay } = req.body;
  await classModel.updateLecture(classId, lectureId, { 
    title: title && title.trim(), 
    date,
    dripDay: dripDay ? Number(dripDay) : null
  });
  res.redirect(`/teacher/classes/${classId}#lectures`);
});

router.post('/classes/:id/assignments', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, date, dripDay } = req.body;
  if (title && url) {
    await classModel.addAssignment(classId, { 
      title: title.trim(), 
      url: url.trim(), 
      date,
      dripDay: dripDay ? Number(dripDay) : null
    });
    try { resourceLibrary.addAssignment({ title: title.trim(), url: url.trim() }); } catch (_) {}
  }
  res.redirect(`/teacher/classes/${classId}#assignments`);
});

router.post('/classes/:id/assignments/:assignmentId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const assignmentId = Number(req.params.assignmentId);
  await classModel.removeAssignment(classId, assignmentId);
  res.redirect(`/teacher/classes/${classId}#assignments`);
});

router.post('/classes/:id/assignments/:assignmentId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const assignmentId = Number(req.params.assignmentId);
  const { title, date, dripDay } = req.body;
  await classModel.updateAssignment(classId, assignmentId, { 
    title: title && title.trim(), 
    date,
    dripDay: dripDay ? Number(dripDay) : null
  });
  res.redirect(`/teacher/classes/${classId}#assignments`);
});


router.post('/classes/:id/simulations', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, date, dripDay } = req.body;
  if (title && url) {
    await classModel.addSimulation(classId, { 
      title: title.trim(), 
      url: url.trim(), 
      date,
      dripDay: dripDay ? Number(dripDay) : null
    });
  }
  res.redirect(`/teacher/classes/${classId}#simulations`);
});

router.post('/classes/:id/simulations/:simId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const simId = Number(req.params.simId);
  await classModel.removeSimulation(classId, simId);
  res.redirect(`/teacher/classes/${classId}#simulations`);
});

router.post('/classes/:id/simulations/:simId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const simId = Number(req.params.simId);
  const { title, date, dripDay } = req.body;
  await classModel.updateSimulation(classId, simId, { 
    title: title && title.trim(), 
    date,
    dripDay: dripDay ? Number(dripDay) : null
  });
  res.redirect(`/teacher/classes/${classId}#simulations`);
});

router.post('/classes/:id/tests/library', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, timeLimit, dueDate, official } = req.body;
  if (title) {
    await classModel.addTest(classId, { 
      title: title.trim(), 
      timeLimit: Number(timeLimit) || 90, 
      dueDate,
      official: official === 'on' || official === true || official === 'true' // Default to true for library tests
    });
  }
  res.redirect(`/teacher/classes/${classId}#tests`);
});

  router.post('/classes/:id/tests/upload', upload.single('csv'), async (req, res) => {
    const classId = Number(req.params.id);
    const title = (req.body.title || 'Uploaded Test').trim();
    const timeLimit = Number(req.body.timeLimit) || 90;
    const official = req.body.official === 'on' || req.body.official === true || req.body.official === 'true';

    const csv = req.file && req.file.buffer.toString('utf-8');
    if (csv) {
      const lines = csv.split(/\r?\n/).filter(l => l.trim());
      const [, ...rows] = lines; // skip header
      const questions = rows.map(line => {
        const cols = line.split(',');
        return {
          question: cols[0],
          answer: cols[1],
          explanation: cols[2],
          picture: cols[3],
          options: cols.slice(4, 11).filter(Boolean),
          test: cols[11],
          contentType: cols[12],
          title: cols[13],
          itemType: cols[14],
          path: cols[15]
        };
      });
      await testModel.replaceTestQuestions(title, questions);
      await classModel.addTest(classId, { title, timeLimit, official });
      // Test is now automatically available in library from database
    }
    res.redirect(`/teacher/classes/${classId}#tests`);
  });

// Upload media assets for tests
router.post('/classes/:id/tests/media', mediaUpload.single('media'), (req, res) => {
  const classId = Number(req.params.id);
  if (!req.file) return res.status(400).send('No file uploaded');
  res.redirect(`/teacher/classes/${classId}#tests`);
});

router.post('/classes/:id/tests/:testId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  await classModel.removeTest(classId, testId);
  res.redirect(`/teacher/classes/${classId}#tests`);
});

router.post('/classes/:id/tests/:testId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  const { title, dueDate, official, dripDay } = req.body;
  await classModel.updateTest(classId, testId, { 
    title: title && title.trim(), 
    dueDate,
    official: official === 'on' || official === true || official === 'true',
    dripDay: dripDay ? Number(dripDay) : null
  });
  res.redirect(`/teacher/classes/${classId}#tests`);
});

// Links management routes
router.post('/classes/:id/links', async (req, res) => {
  try {
    const classId = Number(req.params.id);
    const { title, url, description } = req.body;
    
    console.log('Teacher adding link to class:', classId, { title, url, description });
    
    if (!title || !url) {
      console.error('Missing title or URL for link');
      return res.status(400).send('Title and URL are required');
    }
    
    const link = await classModel.addLink(classId, {
      title: title.trim(),
      url: url.trim(),
      description: description ? description.trim() : ''
    });
    
    console.log('Link added successfully:', link);
    res.redirect(`/teacher/classes/${classId}#links`);
  } catch (error) {
    console.error('Error adding link:', error);
    console.error('Error stack:', error.stack);
    res.status(500).send('Error adding link: ' + error.message);
  }
});

router.post('/classes/:id/links/:linkId/delete', async (req, res) => {
  const classId = Number(req.params.id);
  const linkId = Number(req.params.linkId);
  await classModel.removeLink(classId, linkId);
  res.redirect(`/teacher/classes/${classId}#links`);
});

router.post('/classes/:id/links/:linkId/update', async (req, res) => {
  const classId = Number(req.params.id);
  const linkId = Number(req.params.linkId);
  const { title, url, description } = req.body;
  await classModel.updateLink(classId, linkId, { 
    title: title && title.trim(), 
    url: url && url.trim(),
    description: description ? description.trim() : ''
  });
  res.redirect(`/teacher/classes/${classId}#links`);
});


router.post('/classes/:id/tests/generate', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, prompt, timeLimit } = req.body;
  const questionCount = Math.max(1, Math.min(200, Number(req.body.questionCount) || 10));
  const apiKey = process.env.OPENAI_API_KEY;
  if (!apiKey) return res.status(500).send('Missing OpenAI API key');

  try {
    // 1) Call OpenAI – ask for JSON **only**
    const response = await fetch('https://api.openai.com/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        Authorization: `Bearer ${apiKey}`
      },
      body: JSON.stringify({
        model: 'gpt-3.5-turbo', // swap to a current model in your env if needed
        messages: [
          {
            role: 'system',
            content:
              'You are a data generator. Respond with ONLY a JSON array, no prose, no code fences.'
          },
          {
            role: 'user',
            content:
              `Create a JSON array of exactly ${questionCount} questions for: ${prompt}.
Each array item must be an object with the EXACT keys:
"Question","Answer","Explanation","Picture",
"OptionA","OptionB","OptionC","OptionD","OptionE","OptionF","OptionG",
"Test","Content Type","Title","Item Type","Path".
Do not include any extra keys. Do not include backticks or code fences.`
          }
        ],
        temperature: 0.2
      })
    });

    const data = await response.json();
    if (!response.ok) {
      console.error('OpenAI error payload:', data);
      return res.status(502).send('AI generation failed');
    }

    let raw = data?.choices?.[0]?.message?.content || '';
    // 2) Strip code fences if the model still included them
    raw = raw.trim();
    raw = raw.replace(/^```(?:json)?\s*/i, '').replace(/```$/i, '').trim();

    // 3) Parse and validate
    let items;
    try {
      items = JSON.parse(raw);
    } catch (e) {
      console.error('JSON parse error. Raw content:', raw);
      items = [];
    }
    if (!Array.isArray(items)) {
      console.error('Model did not return an array. Content:', raw);
      items = [];
    }
    // Enforce max question count
    items = items.slice(0, questionCount);

    const testTitle = title || 'AI Generated Test';

    // 4) Ensure answer is present among options, then insert
    const sql = `
      INSERT INTO mdtsapps_myclass.LMSTest5
      (Question, Answer, Explanation, Picture,
       OptionA, OptionB, OptionC, OptionD, OptionE, OptionF, OptionG,
       Test, \`Content Type\`, Title, \`Item Type\`, Path)
      VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
    `;

    // If you're using mysql2/promise:
    // const [stmt] = await db.prepare(sql);  // if you use prepare()
    // We'll just execute per-row here:
    let inserted = 0;
    for (const obj of items) {
      // 5) Defensive access (normalize keys in case of casing drift)
      const get = (k) => obj?.[k] ?? '';
      // Ensure options contain the answer
      const ans = String(get('Answer') || '').trim();
      const keys = ['OptionA','OptionB','OptionC','OptionD','OptionE','OptionF','OptionG'];
      let opts = keys.map(k => String(obj?.[k] || '').trim());
      const hasAns = ans && opts.some(o => o.toLowerCase() === ans.toLowerCase());
      if (ans && !hasAns) {
        const emptyIdx = opts.findIndex(o => !o);
        if (emptyIdx >= 0) opts[emptyIdx] = ans; else opts[opts.length - 1] = ans;
        // Write back so get() below reflects corrected options
        keys.forEach((k,i) => { obj[k] = opts[i]; });
      }

      const values = [
        get('Question'),
        get('Answer'),
        get('Explanation'),
        get('Picture'),
        get('OptionA'),
        get('OptionB'),
        get('OptionC'),
        get('OptionD'),
        get('OptionE'),
        get('OptionF'),
        get('OptionG'),
        get('Test') || testTitle,
        get('Content Type'),
        get('Title'),
        get('Item Type'),
        get('Path')
      ];

      try {
        // mysql2/promise:
        // await db.execute(sql, values);
        await db.query(sql, values); // works if your helper wires to execute internally
        inserted++;
      } catch (dbErr) {
        console.error('DB insert error for values:', values, dbErr);
      }
    }

    // 6) Record the test on the class regardless of per-row failures
    try {
      await classModel.addTest(classId, {
        title: testTitle,
        timeLimit: Number(timeLimit) || 90
      });
      // Test is now automatically available in library from database
    } catch (e) {
      console.error('classModel.addTest failed:', e);
    }

    console.log(`Inserted ${inserted} question rows into LMSTest5.`);
    res.redirect(`/teacher/classes/${classId}#tests`);
  } catch (e) {
    console.error('Route error', e);
    res.status(500).send('Unexpected error');
  }
});

// Add external test link
router.post('/classes/:id/tests/external', async (req, res) => {
  const classId = Number(req.params.id);
  const { title, url, dueDate, dripDay, official } = req.body;
  
  if (title && url) {
    await classModel.addTest(classId, {
      title: title.trim(),
      externalUrl: url.trim(),
      dueDate: dueDate || null,
      dripDay: dripDay ? Number(dripDay) : null,
      official: official === 'on' || official === true || official === 'true',
      isExternal: true
    });
  }
  res.redirect(`/teacher/classes/${classId}#tests`);
});

// Grade submission (upsert)
router.post('/classes/:id/grades', async (req, res) => {
  const classId = Number(req.params.id);
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Not found');
  // Test grades (if any were submitted)
  for (const key of Object.keys(req.body)) {
    const m = key.match(/^grade_(\d+)_(\d+)$/);
    if (!m) continue;
    const studentId = Number(m[1]);
    const testId = Number(m[2]);
    const val = req.body[key];
    if (val === '') continue;
    let score = Number(val);
    if (Number.isNaN(score)) continue;
    if (score < 0) score = 0;
    if (score > 100) score = 100;
    await classModel.upsertGrade(classId, testId, studentId, score);
  }

  // Assignment grades
  for (const studentId of klass.studentIds || []) {
    for (const a of klass.assignments || []) {
      const key = `asg_${studentId}_${a.id}`;
      const val = req.body[key];
      if (val === undefined || val === '') continue;
      let score = Number(val);
      if (Number.isNaN(score)) continue;
      if (score < 0) score = 0;
      if (score > 100) score = 100;
      await classModel.upsertAssignmentGrade(classId, a.id, Number(studentId), score);
    }
    for (const l of klass.simulations || []) {
      const key = `lab_${studentId}_${l.id}`;
      const passed = !!req.body[key];
      await classModel.upsertLabStatus(classId, l.id, Number(studentId), passed);
    }
  }

  res.redirect(`/teacher/classes/${classId}#grades`);
});

// View student profile
router.get('/students/:id', async (req, res) => {
  const student = await userModel.findById(Number(req.params.id));
  if (!student) return res.status(404).send('Not found');
  let lead = null, leadContacts = [];
  try {
    if (student.email) {
      const leadModel = require('../models/leadModel');
      lead = await leadModel.getByEmail(student.email);
      if (lead) leadContacts = await leadModel.getContacts(lead.id);
    }
  } catch (e) { console.error('Lead fetch failed', e); }
  
  // Get all classes where this student is enrolled
  const allClasses = await classModel.getAllClasses();
  const studentClasses = allClasses.filter(c => (c.studentIds || []).includes(student.id));
  
  // Get course financial information for enrolled classes
  const courseModel = require('../models/courseModel');
  const allCourses = await courseModel.getAllCourses();
  
  // Create multiple lookup indexes for better matching
  const courseLookup = {};
  const courseByLowerName = {};
  allCourses.forEach(course => {
    const lowerName = course.program_name.toLowerCase().trim();
    courseLookup[course.program_name] = course;
    courseByLowerName[lowerName] = course;
    // Also index by words in the name for partial matching
    const words = lowerName.split(/[\s\/\-]+/);
    words.forEach(word => {
      if (word.length > 3 && !courseByLowerName[word]) {
        courseByLowerName[word] = course;
      }
    });
  });
  
  // Build tuition breakdown by matching class names to courses
  const tuitionBreakdown = [];
  let totalTuition = 0;
  
  studentClasses.forEach(klass => {
    let matchingCourse = null;
    
    // Try 1: Exact match by full name
    matchingCourse = courseLookup[klass.name];
    
    // Try 2: Case-insensitive match by full name
    if (!matchingCourse) {
      matchingCourse = courseByLowerName[klass.name.toLowerCase().trim()];
    }
    
    // Try 3: Match by shortName (badge like "HEL 101")
    if (!matchingCourse && klass.shortName) {
      matchingCourse = courseByLowerName[klass.shortName.toLowerCase().trim()];
    }
    
    // Try 4: Fuzzy match - check if course name contains class name or vice versa
    if (!matchingCourse) {
      const className = klass.name.toLowerCase().trim();
      const classShortName = (klass.shortName || '').toLowerCase().trim();
      
      for (const course of allCourses) {
        const courseName = course.program_name.toLowerCase().trim();
        
        // Check if course name contains class name (e.g., "Risk Management Framework" contains "Risk Management")
        if (courseName.includes(className) || className.includes(courseName)) {
          matchingCourse = course;
          break;
        }
        
        // Check shortName match
        if (classShortName && (courseName.includes(classShortName) || classShortName.includes(courseName))) {
          matchingCourse = course;
          break;
        }
        
        // Check if significant words match
        const classWords = className.split(/[\s\/\-]+/).filter(w => w.length > 3);
        const courseWords = courseName.split(/[\s\/\-]+/).filter(w => w.length > 3);
        const matchingWords = classWords.filter(cw => courseWords.includes(cw));
        
        if (matchingWords.length >= 2 || (matchingWords.length === 1 && classWords.length === 1)) {
          matchingCourse = course;
          break;
        }
      }
    }
    
    if (matchingCourse) {
      const cost = parseFloat(matchingCourse.program_cost) || 0;
      totalTuition += cost;
      tuitionBreakdown.push({
        courseName: matchingCourse.program_name,
        cost: cost,
        hours: matchingCourse.program_hours || 0,
        className: klass.name,
        classShortName: klass.shortName || '',
        classId: klass.id,
        isManual: false
      });
    }
  });
  
  // Add manual tuition courses
  const manualTuitionCourses = student.profile?.manualTuitionCourses || [];
  manualTuitionCourses.forEach((manual, index) => {
    const cost = parseFloat(manual.cost) || 0;
    totalTuition += cost;
    tuitionBreakdown.push({
      courseName: manual.courseName,
      cost: cost,
      hours: manual.clockHours || 0,
      className: '',
      classShortName: '',
      classId: null,
      isManual: true,
      manualIndex: index
    });
  });
  
  const registrationFee = 75.00;
  const totalCost = totalTuition + registrationFee;
  
  res.render('student_profile', { 
    student, 
    role: 'teacher', 
    reset: req.query.reset, 
    signatureDocsConfig, 
    lead, 
    leadContacts,
    studentClasses,
    tuitionBreakdown,
    registrationFee,
    totalTuition,
    totalCost
  });
});

router.post('/students/:id/reset-password', async (req, res) => {
  const id = Number(req.params.id);
  const user = await userModel.findById(id);
  if (!user) return res.status(404).send('Not found');
  const newPassword = crypto.randomBytes(4).toString('hex');
  await userModel.updatePassword(user.username, newPassword);
  if (user.email) {
    try {
      const { subject, html, text } = emailTemplates.render('passwordReset', {
        name: user.name || 'User',
        newPassword
      });
      await transporter.sendMail({
        from: 'no-reply@mdts-apps.com',
        to: user.email,
        subject,
        html,
        text
      });
    } catch (e) {
      console.error('Error sending reset email', e);
    }
  }
  res.redirect(`/teacher/students/${id}?reset=1`);});

router.get('/', async (req, res) => {
  const classes = await classModel.byTeacher(req.session.user.id);
  const weekly = { Monday:[], Tuesday:[], Wednesday:[], Thursday:[], Friday:[], Saturday:[], Sunday:[] };

  const events = [];
  const now = new Date();
  const threeWeeksFromNow = new Date();
  threeWeeksFromNow.setDate(now.getDate() + 21);

classes.forEach(klass => {
  (klass.schedule || []).forEach(s => {
    const dayIndex = ['Sunday','Monday','Tuesday','Wednesday','Thursday','Friday','Saturday'].indexOf(s.day);
    let date = new Date(now);
    while (date <= threeWeeksFromNow) {
      if (date.getDay() === dayIndex) {
        const [sh, sm] = String(s.start || '0:0').split(':').map(Number);
        events.push({
          title: klass.name + ' (Class)',
          start: new Date(date.getFullYear(), date.getMonth(), date.getDate(), sh, sm || 0),
          url: `/teacher/classes/${klass.id}` // NEW: link to teacher's class view
        });
      }
      date.setDate(date.getDate() + 1);
    }
  });

  (klass.tests || []).forEach(test => {
    if (test.dueDate) {
      const due = new Date(test.dueDate);
      if (due >= now && due <= threeWeeksFromNow) {
        events.push({
          title: klass.name + ' - ' + test.title + ' (Due)',
          start: due,
          url: `/teacher/classes/${klass.id}` // Could link to grade view or test edit
        });
      }
    }
  });
});


  res.render('teacher_dashboard', { teacher: req.session.user, classes, weekly, events });
});

// Detailed attendance submission (new format with time in/out, hours, absent/tardy)
router.post('/classes/:id/attendance/detailed', async (req, res) => {
  const classId = Number(req.params.id);
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Class not found');
  
  const { date, scheduledHours, timeIn, timeOut } = req.body;
  
  // Collect student entries from form
  const entries = [];
  let idx = 0;
  
  console.log('RAW FORM DATA:', JSON.stringify(req.body, null, 2));
  
  while (req.body[`studentId_${idx}`] !== undefined) {
    let studentIdRaw = req.body[`studentId_${idx}`];
    // Handle array case (if form has duplicate fields)
    if (Array.isArray(studentIdRaw)) {
      studentIdRaw = studentIdRaw[0];
    }
    const studentId = Number(studentIdRaw);
    
    // Skip if studentId is not a valid number
    if (isNaN(studentId) || !studentId) {
      idx++;
      continue;
    }
    
    let hoursRaw = req.body[`hoursCompleted_${idx}`];
    if (Array.isArray(hoursRaw)) hoursRaw = hoursRaw[0];
    const hoursCompleted = parseFloat(hoursRaw) || 0;
    
    const present = req.body[`present_${idx}`] === 'on';
    const tardy = req.body[`tardy_${idx}`] === 'on';
    
    let reasonRaw = req.body[`reason_${idx}`];
    if (Array.isArray(reasonRaw)) reasonRaw = reasonRaw[0];
    const reason = reasonRaw || '';
    
    console.log(`Student ${idx} (ID ${studentId}):`, { present, tardy, hours: hoursCompleted });
    
    entries.push({
      studentId,
      hoursCompleted: present ? hoursCompleted : 0,
      present,
      tardy,
      reason
    });
    idx++;
  }
  
  // Build attendance data
  const currentUser = req.user || req.session?.user || {};
  const attendanceData = {
    date: date || new Date().toISOString().slice(0, 10),
    scheduledHours: parseFloat(scheduledHours) || 0,
    timeIn: timeIn || '',
    timeOut: timeOut || '',
    entries,
    instructorSignature: null,
    signedAt: null,
    takenBy: currentUser.name || currentUser.email || 'Instructor',
    takenById: currentUser.id || null
  };
  
  // Record detailed attendance
  await classModel.recordDetailedAttendance(classId, attendanceData);
  
  // Update each student's profile with attendance record
  for (const entry of entries) {
    if (!entry.absent && entry.studentId && !isNaN(entry.studentId)) {
      const student = await userModel.findById(entry.studentId);
      if (student) {
        const profile = student.profile || {};
        profile.attendance = profile.attendance || [];
        
        // Check if attendance for this date and class already exists
        const existing = profile.attendance.find(a => a.date === attendanceData.date && a.classId === classId);
        if (!existing) {
          profile.attendance.push({
            date: attendanceData.date,
            classId: classId,
            className: klass.name,
            studentName: student.name,
            hoursCompleted: entry.hoursCompleted,
            tardy: entry.tardy
          });
          await userModel.updateProfile(entry.studentId, { attendance: profile.attendance });
        }
      }
    }
  }
  
  res.redirect(`/teacher/classes/${classId}`);
});

// Individual student attendance record view
router.get('/classes/:classId/students/:studentId/attendance-record', async (req, res) => {
  try {
    const classId = Number(req.params.classId);
    const studentId = Number(req.params.studentId);
    
    console.log('Attendance record request:', { classId, studentId });
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      console.log('Class not found:', classId);
      return res.status(404).send('Class not found');
    }
    
    const student = await userModel.findById(studentId);
    if (!student) {
      console.log('Student not found:', studentId);
      return res.status(404).send('Student not found');
    }
    
    // Get attendance record using the new function
    const attendanceRecord = await classModel.getStudentAttendanceRecord(classId, studentId);
    console.log('Attendance record retrieved:', attendanceRecord ? 'success' : 'null');
    
    // Get teacher info
    const teacher = await userModel.findById(klass.teacherId);
    
    res.render('student_attendance_record', {
      klass,
      student,
      teacher,
      attendanceRecord: attendanceRecord || { records: [], summary: { totalScheduledHours: 0, totalCompletedHours: 0, percentageCompleted: 0, meetsRequirement: false } },
      user: req.session.user
    });
  } catch (error) {
    console.error('Error loading attendance record:', error);
    res.status(500).send('Error loading attendance record: ' + error.message);
  }
});

// Delete a specific attendance entry for a student
router.delete('/classes/:classId/students/:studentId/attendance/:date', async (req, res) => {
  try {
    const classId = Number(req.params.classId);
    const studentId = Number(req.params.studentId);
    const date = req.params.date;
    
    console.log('Delete attendance request:', { classId, studentId, date });
    
    const result = await classModel.deleteStudentAttendanceEntry(classId, studentId, date);
    
    if (result.success) {
      res.status(200).json({ success: true, message: 'Attendance record deleted successfully' });
    } else {
      res.status(400).send(result.error || 'Failed to delete attendance record');
    }
  } catch (error) {
    console.error('Error deleting attendance record:', error);
    res.status(500).send('Error deleting attendance record: ' + error.message);
  }
});


// Attendance submission (legacy format - kept for backward compatibility)
router.post('/classes/:id/attendance', async (req, res) => {
  const classId = Number(req.params.id);
  const date = req.body.date || new Date().toISOString().slice(0,10);
  const presentIds = Object.keys(req.body)
    .filter(k => k.startsWith('present_'))
    .map(k => Number(k.replace('present_', '')));
  
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Class not found');
  
  // Record attendance in class model
  await classModel.recordAttendance(classId, date, presentIds);
  
  // Update each student's profile with attendance record
  for (const studentId of presentIds) {
    const student = await userModel.findById(studentId);
    if (student) {
      const profile = student.profile || {};
      profile.attendance = profile.attendance || [];
      
      // Check if attendance for this date and class already exists
      const existing = profile.attendance.find(a => a.date === date && a.classId === classId);
      if (!existing) {
        profile.attendance.push({
          date: date,
          classId: classId,
          className: klass.name,
          studentName: student.name
        });
        await userModel.updateProfile(studentId, { attendance: profile.attendance });
      }
    }
  }
  
  res.redirect(`/teacher/classes/${classId}`);
});

// Attendance report for a class
router.get('/classes/:id/attendance', async (req, res) => {
  const klass = await classModel.findClassById(Number(req.params.id));
  if (!klass) return res.status(404).send('Not found');
  const users = await userModel.getAll();
  const students = users
    .filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id))
    .sort((a, b) => {
      // Sort alphabetically by name
      const nameA = (a.name || a.profile?.firstName || '').toLowerCase();
      const nameB = (b.name || b.profile?.firstName || '').toLowerCase();
      return nameA.localeCompare(nameB);
    })
    .map(s => {
      // Calculate attendance stats for each student
      const attendance = klass.attendance || [];
      const presentDates = attendance
        .filter(a => {
          // Check new format: entries array
          const entry = (a.entries || []).find(e => Number(e.studentId) === s.id);
          if (entry && entry.present === true) {
            return true;
          }
          // Check legacy format: present array
          return (a.present || []).includes(s.id);
        })
        .map(a => a.date)
        .sort(); // Sort dates chronologically
      
      return {
        ...s,
        studentId: s.profile?.studentId || s.id,
        totalPresent: presentDates.length,
        presentDates: presentDates
      };
    });
  res.render('attendance_report', { klass, students, user: req.session.user });
});

// Overall Attendance Report - Day by Day (teacher)
router.get('/classes/:id/attendance/overall', async (req, res) => {
  const klass = await classModel.findClassById(Number(req.params.id));
  if (!klass || klass.teacherId !== req.session.user.id) {
    return res.status(404).send('Class not found or access denied');
  }
  
  const users = await userModel.getAll();
  const students = users
    .filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id))
    .sort((a, b) => {
      const nameA = (a.name || a.profile?.firstName || '').toLowerCase();
      const nameB = (b.name || b.profile?.firstName || '').toLowerCase();
      return nameA.localeCompare(nameB);
    });
  
  // Get all attendance records sorted by date
  const attendanceRecords = (klass.attendance || []).sort((a, b) => a.date.localeCompare(b.date));
  
  res.render('overall_attendance_report', { klass, students, attendanceRecords, user: req.session.user });
});

// Individual student grade report
router.get('/classes/:classId/students/:studentId/grade-report', async (req, res) => {
  const classId = Number(req.params.classId);
  const studentId = Number(req.params.studentId);
  
  const klass = await classModel.findClassById(classId);
  if (!klass || klass.teacherId !== req.session.user.id) {
    return res.status(404).send('Class not found or access denied');
  }
  
  const users = await userModel.getAll();
  const student = users.find(u => u.id === studentId);
  
  if (!student || student.role !== 'student' || !(klass.studentIds || []).includes(studentId)) {
    return res.status(404).send('Student not found or not enrolled in this class');
  }
  
  res.render('grade_report', { klass, student });
});

// Print gradebook for entire class
router.get('/classes/:id/gradebook/print', async (req, res) => {
  const classId = Number(req.params.id);
  const klass = await classModel.findClassById(classId);
  
  if (!klass || klass.teacherId !== req.session.user.id) {
    return res.status(404).send('Class not found or access denied');
  }
  
  const users = await userModel.getAll();
  const students = users.filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id));
  
  res.render('gradebook_print', { klass, students });
});

router.get('/reports/class/:id', async (req, res) => {
  const id = Number(req.params.id);
  const klass = await classModel.findClassById(id);
  if (!klass || klass.teacherId !== req.session.user.id) return res.status(404).send('Not found');
  const users = await userModel.getAll();
  const students = users.filter(u => u.role === 'student' && (klass.studentIds || []).includes(u.id));
  res.render('class_report', { klass, students, scope: 'teacher' });
});

// Reports
router.get('/reports', async (req, res) => {
  const classes = await classModel.byTeacher(req.session.user.id);
  const users = await userModel.getAll();
  const studentMap = Object.fromEntries(users.filter(u => u.role === 'student').map(u => [u.id, u]));
  const report = classes.map(k => {
    const studentsArr = (k.studentIds || [])
      .map(id => studentMap[id])
      .filter(Boolean)
      .map(u => ({ id: u.id, name: u.name }));
    return {
      classId: k.id,
      className: k.name,
      studentsList: studentsArr,
      studentCount: studentsArr.length,
      tests: (k.tests || []).length,
      grades: (k.grades || []).length
    };
  });
  res.render('reports', { report, scope: 'teacher' });
});

// Preview test routes
router.get('/classes/:id/tests/:testId/preview', async (req, res) => {
  // Access control handled by middleware above
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  console.log('Teacher preview test', { classId, testId, userId: req.session.user?.id, userRole: req.session.role });
  const klass = await classModel.findClassById(classId);
  if (!klass) {
    console.log('Class not found', classId);
    return res.status(404).send('Not found');
  }
  const test = (klass.tests || []).find(t => t.id === testId);
  if (!test) {
    console.log('Test not found', { classId, testId });
    return res.status(404).send('Test not found');
  }
  test.questions = await testModel.getQuestionsByTest(test.title);
  res.render('take_test', {
    klass,
    test,
    attempts: 0,
    user: req.session.user,
    action: `/teacher/classes/${classId}/tests/${testId}/preview`
  });
});

router.post('/classes/:id/tests/:testId/preview', async (req, res) => {
  // Access control handled by middleware above
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  console.log('Teacher submit preview', { classId, testId, userId: req.session.user?.id });
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Not found');
  const test = (klass.tests || []).find(t => t.id === testId);
  if (!test) return res.status(404).send('Test not found');
  test.questions = await testModel.getQuestionsByTest(test.title);
  let score = 0;
  test.questions.forEach((q, i) => {
    const chosen = Number(req.body[`q_${i}`]);
    const correct = Number(q.answer);
    if (!Number.isNaN(chosen) && chosen === correct) score++;
  });
  const pct = Math.round((score / test.questions.length) * 100);
  console.log('Preview score', { classId, testId, userId: req.session.user?.id, score: pct });
  res.render('test_result', { klass, test, score: pct, student: req.session.user, user: req.session.user });
});


// Upload media for a question (teacher preview)
router.post('/classes/:id/tests/:testId/questions/:qIndex/media', mediaUpload.single('media'), async (req, res) => {
  // Access control handled by middleware above
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  const qIndex = Number(req.params.qIndex);
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send('Not found');
  const test = (klass.tests || []).find(t => t.id === testId);
  if (!test) return res.status(404).send('Test not found');
  if (!req.file) return res.redirect(`/teacher/classes/${classId}/tests/${testId}/preview`);
  const url = `/uploads/${req.file.filename}`;
  const isVideo = (req.file.mimetype || '').startsWith('video/');
  try {
    // Resolve question text by index to avoid attribute escaping issues
    let questionText = '';
    try {
      const questions = await testModel.getQuestionsByTest(test.title);
      if (Array.isArray(questions) && questions[qIndex]) questionText = questions[qIndex].question || '';
    } catch (_) {}
    await testModel.updateQuestionMediaByText(test.title, questionText, url, isVideo);
  } catch (e) {
    console.error('update media failed', e);
  }
  res.redirect(`/teacher/classes/${classId}/tests/${testId}/preview`);
});
module.exports = router;
