const express = require("express");
const router = express.Router();
const announcementModel = require("../models/announcementModel");

const classModel = require("../models/classModel");
const { generateQuestions } = require("../utils/questionGenerator");
const testModel = require("../models/testModel");

const discussionModel = require("../models/discussionModel");
const userModel = require("../models/userModel");
const nodemailer = require("nodemailer");
const messageModel = require("../models/messageModel");
const emailTemplates = require("../utils/emailTemplates");
const signatureDocumentModel = require("../models/signatureDocumentModel");
const careerAdvisorInterviewModel = require("../models/careerAdvisorInterviewModel");
const careerAdvisorSettings = require("../models/careerAdvisorSettings");
const leadModel = require("../models/leadModel");
const notificationSettings = require("../utils/notificationSettings");
const multer = require("multer");
const path = require("path");

const fs = require("fs");
const CONFIG_PATH = path.join(__dirname, "..", "data", "signature-docs.json");
let marked;
try {
  marked = require("marked");
} catch (err) {
  console.warn(
    "[student manual] marked package not found, using plain text manual rendering"
  );
  marked = null;
}

let signatureDocsConfig = {};
function loadSignatureDocsConfig() {
  try {
    signatureDocsConfig = JSON.parse(fs.readFileSync(CONFIG_PATH, "utf8"));
  } catch (err) {
    console.error("Failed to load signature-docs.json:", err.message);
    signatureDocsConfig = {};
  }
}
loadSignatureDocsConfig();

const transporter = nodemailer.createTransport({
  host: "mdts-apps.com",
  port: 465,
  secure: true,
  auth: {
    user: "noreply@mdts-apps.com",
    pass: "c@r,5ysPI@&s",
  },
});

router.get("/manual", (req, res) => {
  if (!req.session || !["student", "admin"].includes(req.session.role)) {
    return res.status(403).send("Forbidden");
  }
  const manualPath = path.join(__dirname, "..", "docs", "student-manual.md");
  let manualHtml = "";
  let updatedAt = null;
  try {
    const md = fs.readFileSync(manualPath, "utf8");
    const render = (src, opts) => {
      if (!marked) {
        const escape = (str) =>
          str
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;");
        return `<pre style="white-space:pre-wrap;">${escape(src)}</pre>`;
      }
      if (typeof marked.parse === "function") return marked.parse(src, opts);
      return marked(src, opts);
    };
    manualHtml = render(md, { mangle: false, headerIds: true });
    updatedAt = fs.statSync(manualPath).mtime;
  } catch (err) {
    console.error("Failed to load student manual", err);
  }
  res.render("student_manual", {
    user: req.session.user,
    manualHtml,
    manualUpdatedAt: updatedAt,
  });
});

router.use((req, res, next) => {
  if (!req.session || !["student", "admin"].includes(req.session.role))
    return res.status(403).send("Forbidden");
  next();
});

// Uploader for assignment submissions
const submissionStorage = multer.diskStorage({
  destination: (req, file, cb) => cb(null, path.join(__dirname, "../uploads")),
  filename: (req, file, cb) => cb(null, Date.now() + "-" + file.originalname),
});
const submissionUpload = multer({
  storage: submissionStorage,
  limits: { fileSize: 25 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const ext = path.extname(file.originalname).toLowerCase();
    const okExt = [".png", ".jpg", ".jpeg", ".csv"];
    const okMime = [
      "image/png",
      "image/jpeg",
      "text/csv",
      "application/vnd.ms-excel",
    ];
    if (okExt.includes(ext) || okMime.includes(file.mimetype)) cb(null, true);
    else cb(new Error("Invalid file type"));
  },
});
router.get("/profile", async (req, res) => {
  const student = await userModel.findById(req.session.user.id);
  if (!student) return res.status(404).send("Not found");
  const updated = String(req.query.updated || "") === "1";
  let lead = null,
    leadContacts = [];
  try {
    if (student.email) {
      const leadModel = require("../models/leadModel");
      lead = await leadModel.getByEmail(student.email);
      if (lead) leadContacts = await leadModel.getContacts(lead.id);
    }
  } catch (e) {
    console.error("Lead fetch failed", e);
  }

  // Get all classes where this student is enrolled
  const classModel = require("../models/classModel");
  const allClasses = await classModel.getAllClasses();
  const studentClasses = allClasses.filter((c) =>
    (c.studentIds || []).includes(student.id)
  );

  // Get course financial information for enrolled classes
  const courseModel = require('../models/courseModel');
  const allCourses = await courseModel.getAllCourses();
  
  // Create multiple lookup indexes for better matching
  const courseLookup = {};
  const courseByLowerName = {};
  allCourses.forEach(course => {
    const lowerName = course.program_name.toLowerCase().trim();
    courseLookup[course.program_name] = course;
    courseByLowerName[lowerName] = course;
    // Also index by words in the name for partial matching
    const words = lowerName.split(/[\s\/\-]+/);
    words.forEach(word => {
      if (word.length > 3 && !courseByLowerName[word]) {
        courseByLowerName[word] = course;
      }
    });
  });
  
  // Build tuition breakdown by matching class names to courses
  const tuitionBreakdown = [];
  let totalTuition = 0;
  
  studentClasses.forEach(klass => {
    let matchingCourse = null;
    
    // Try 1: Exact match by full name
    matchingCourse = courseLookup[klass.name];
    
    // Try 2: Case-insensitive match by full name
    if (!matchingCourse) {
      matchingCourse = courseByLowerName[klass.name.toLowerCase().trim()];
    }
    
    // Try 3: Match by shortName (badge like "HEL 101")
    if (!matchingCourse && klass.shortName) {
      matchingCourse = courseByLowerName[klass.shortName.toLowerCase().trim()];
    }
    
    // Try 4: Fuzzy match - check if course name contains class name or vice versa
    if (!matchingCourse) {
      const className = klass.name.toLowerCase().trim();
      const classShortName = (klass.shortName || '').toLowerCase().trim();
      
      for (const course of allCourses) {
        const courseName = course.program_name.toLowerCase().trim();
        
        // Check if course name contains class name (e.g., "Risk Management Framework" contains "Risk Management")
        if (courseName.includes(className) || className.includes(courseName)) {
          matchingCourse = course;
          break;
        }
        
        // Check shortName match
        if (classShortName && (courseName.includes(classShortName) || classShortName.includes(courseName))) {
          matchingCourse = course;
          break;
        }
        
        // Check if significant words match
        const classWords = className.split(/[\s\/\-]+/).filter(w => w.length > 3);
        const courseWords = courseName.split(/[\s\/\-]+/).filter(w => w.length > 3);
        const matchingWords = classWords.filter(cw => courseWords.includes(cw));
        
        if (matchingWords.length >= 2 || (matchingWords.length === 1 && classWords.length === 1)) {
          matchingCourse = course;
          break;
        }
      }
    }
    
    if (matchingCourse) {
      const cost = parseFloat(matchingCourse.program_cost) || 0;
      totalTuition += cost;
      tuitionBreakdown.push({
        courseName: matchingCourse.program_name,
        cost: cost,
        hours: matchingCourse.program_hours || 0,
        className: klass.name,
        classShortName: klass.shortName || '',
        classId: klass.id,
        isManual: false
      });
    }
  });
  
  // Add manual tuition courses
  const manualTuitionCourses = student.profile?.manualTuitionCourses || [];
  manualTuitionCourses.forEach((manual, index) => {
    const cost = parseFloat(manual.cost) || 0;
    totalTuition += cost;
    tuitionBreakdown.push({
      courseName: manual.courseName,
      cost: cost,
      hours: manual.clockHours || 0,
      className: '',
      classShortName: '',
      classId: null,
      isManual: true,
      manualIndex: index
    });
  });
  
  const registrationFee = 75.00;
  const totalCost = totalTuition + registrationFee;

  res.render("student_profile", {
    student,
    role: "student",
    signatureDocsConfig,
    updated,
    lead,
    leadContacts,
    studentClasses,
    tuitionBreakdown,
    registrationFee,
    totalTuition,
    totalCost,
  });
});

// Configure multer for ID and transcript uploads with flexible field support
const documentStorage = multer.diskStorage({
  destination: (req, file, cb) => cb(null, path.join(__dirname, "../uploads")),
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
    cb(
      null,
      file.fieldname + "-" + uniqueSuffix + path.extname(file.originalname)
    );
  },
});

const documentUpload = multer({
  storage: documentStorage,
  limits: { fileSize: 10 * 1024 * 1024 }, // 10MB limit
  fileFilter: (req, file, cb) => {
    const allowed = [".pdf", ".jpg", ".jpeg", ".png", ".doc", ".docx"];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else
      cb(
        new Error(
          `Invalid file type: ${ext}. Only PDF, JPG, PNG, DOC, and DOCX files are allowed.`
        )
      );
  },
}).any(); // Accept any field names to avoid "Unexpected field" errors

// Handle document uploads (ID and Transcript)
router.post("/profile/upload-documents", (req, res) => {
  documentUpload(req, res, async (err) => {
    const student = await userModel.findById(req.session.user.id);
    if (!student) return res.status(404).send("Student not found");

    if (err) {
      console.error("Document upload error:", err);
      return res.status(400).render("student_profile", {
        student,
        role: "student",
        signatureDocsConfig,
        updated: false,
        error: err.message,
        studentClasses: [],
      });
    }

    try {
      const documents = {};

      // With .any(), files are in req.files array, not grouped by fieldname
      if (req.files && Array.isArray(req.files)) {
        req.files.forEach((file) => {
          if (file.fieldname === "idDocument") {
            documents.idDocument = `/uploads/${file.filename}`;
          } else if (file.fieldname === "transcriptDocument") {
            documents.transcriptDocument = `/uploads/${file.filename}`;
          }
        });
      }

      // Update the student profile with the document paths
      if (Object.keys(documents).length > 0) {
        await userModel.updateDocuments(student.id, documents);
      }

      res.redirect("/student/profile?updated=1");
    } catch (e) {
      console.error("Failed to update documents:", e);
      res.status(500).render("student_profile", {
        student,
        role: "student",
        signatureDocsConfig,
        updated: false,
        error: "Failed to save documents. Please try again.",
        studentClasses: [],
      });
    }
  });
});

router.post("/profile/complete", (req, res) => {
  upload(req, res, async (err) => {
    const student = await userModel.findById(req.session.user.id);
    if (!student) return res.status(404).send("Not found");
    if (err)
      return res
        .status(400)
        .render("student_profile", {
          student,
          role: "student",
          step2Error: err.message,
          signatureDocsConfig,
          studentClasses: [],
        });
    const {
      ssn: rawSSN,
      emergencyName,
      emergencyRelation,
      emergencyPhone,
      agree,
      grievanceAck,
    } = req.body;
    const ssn = rawSSN ? String(rawSSN).replace(/\D/g, "") : undefined;
    if (!agree) {
      return res
        .status(400)
        .render("student_profile", {
          student,
          role: "student",
          step2Error: "You must agree to the registration agreement.",
          signatureDocsConfig,
          studentClasses: [],
        });
    }
    try {
      const docs = (student.profile && student.profile.documents) || [];
      const reg = docs.find((d) => d.type === "registration-agreement");
      if (reg) reg.agreed = true;
      await userModel.updateProfile(student.id, {
        ssn,
        emergencyContact: {
          name: emergencyName,
          relation: emergencyRelation,
          phone: emergencyPhone,
        },
        grievanceAcknowledged: !!grievanceAck,
        documents: docs,
      });
      if (req.files) {
        const collected = [];
        if (Array.isArray(req.files.gedDoc))
          collected.push(...req.files.gedDoc);
        if (Array.isArray(req.files.govIdDoc))
          collected.push(...req.files.govIdDoc);
        if (collected.length) {
          const uploads = collected.map((f) => ({
            originalName: f.originalname,
            mimeType: f.mimetype,
            size: f.size,
            url: `/docs/stxd/${f.filename}`,
          }));
          await userModel.addUploads(student.id, uploads);
        }
      }
      res.redirect("/student/profile?updated=1");
    } catch (e) {
      console.error("student complete", e);
      res
        .status(500)
        .render("student_profile", {
          student,
          role: "student",
          step2Error: "Failed to update profile.",
          studentClasses: [],
        });
    }
  });
});

const upload = multer({
  storage: multer.diskStorage({
    destination: (req, file, cb) =>
      cb(null, path.join(__dirname, "../docs/stxd")),
    filename: (req, file, cb) => cb(null, Date.now() + "-" + file.originalname),
  }),
  limits: { fileSize: 15 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = [".pdf", ".jpg", ".jpeg", ".png", ".doc", ".docx"];
    if (allowed.includes(path.extname(file.originalname).toLowerCase()))
      cb(null, true);
    else cb(new Error("Invalid file type"));
  },
}).fields([
  { name: "gedDoc", maxCount: 1 },
  { name: "govIdDoc", maxCount: 1 },
]);

// Uploader for student-signed copies returned to admin
const signedReturnUpload = multer({
  storage: multer.diskStorage({
    destination: (req, file, cb) =>
      cb(null, path.join(__dirname, "../docs/stxd")),
    filename: (req, file, cb) => cb(null, Date.now() + "-" + file.originalname),
  }),
  limits: { fileSize: 20 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = [".pdf", ".jpg", ".jpeg", ".png", ".doc", ".docx"];
    if (allowed.includes(path.extname(file.originalname).toLowerCase()))
      cb(null, true);
    else cb(new Error("Invalid file type"));
  },
}).single("signedFile");

router.post("/sign-doc", async (req, res) => {
  const { docType, signatureDataUrl } = req.body;
  if (docType && signatureDataUrl) {
    try {
      const user = await userModel.signDocument(
        req.session.user.id,
        docType,
        signatureDataUrl
      );
      const docs = (user.profile && user.profile.documents) || [];
      const doc = docs.find((d) => d.type === docType);
      const pending = docs.filter(
        (d) => !d.requiredRole && !d.signatureDataUrl
      );
      if (!pending.length) {
        try {
          await userModel.markApplicationComplete(req.session.user.id);
        } catch (err) {
          console.error("complete mark", err);
        }
      }
      if (
        req.xhr ||
        (req.headers.accept && req.headers.accept.includes("application/json"))
      ) {
        return res.json({ success: true, signedAt: doc?.signedAt });
      }
    } catch (e) {
      console.error("student sign", e);
      if (
        req.xhr ||
        (req.headers.accept && req.headers.accept.includes("application/json"))
      ) {
        return res.status(500).json({ success: false });
      }
    }
  }
  res.redirect("/student/profile");
});

// Student uploads signed copy for a specific to-sign document
router.post("/documents/to-sign/:docId", (req, res) => {
  signedReturnUpload(req, res, async (err) => {
    try {
      const studentId = req.session.user.id;
      const docId = String(req.params.docId || "");
      const student = await userModel.findById(studentId);
      if (!student) return res.status(404).send("Not found");
      if (err) return res.status(400).send(err.message || "Upload failed");
      if (!req.file) return res.status(400).send("No file uploaded");
      const profile = student.profile || {};
      const toSignDocs = Array.isArray(profile.toSignDocs)
        ? profile.toSignDocs
        : [];
      const idx = toSignDocs.findIndex((d) => String(d.id) === docId);
      if (idx === -1) return res.status(404).send("Document not found");
      const meta = {
        originalName: req.file.originalname,
        mimeType: req.file.mimetype,
        size: req.file.size,
        url: `/docs/stxd/${req.file.filename}`,
      };
      toSignDocs[idx].studentFile = meta;
      toSignDocs[idx].completedAt = new Date().toISOString();
      // Also add to general signedDocs list for admin visibility
      const signedDocs = Array.isArray(profile.signedDocs)
        ? profile.signedDocs
        : [];
      signedDocs.push({ ...meta, relatedTo: docId });
      await userModel.updateProfile(studentId, { toSignDocs, signedDocs });

      // Notify office inbox or all admins by email
      try {
        const office =
          process.env.OFFICE_INBOX && String(process.env.OFFICE_INBOX).trim();
        if (office) {
          await transporter.sendMail({
            from: "noreply@mdts-apps.com",
            to: office,
            subject: "Student uploaded signed document",
            text: `${
              student.name || student.username
            } uploaded a signed document: ${toSignDocs[idx].originalName}`,
            html: `<p><strong>${
              student.name || student.username
            }</strong> uploaded a signed document: ${
              toSignDocs[idx].originalName
            }</p>`,
          });
        } else {
          const admins = await userModel.getByRole("admin");
          const tos = admins.map((a) => a.email).filter(Boolean);
          if (tos.length) {
            await transporter.sendMail({
              from: "noreply@mdts-apps.com",
              to: tos.join(","),
              subject: "Student uploaded signed document",
              text: `${
                student.name || student.username
              } uploaded a signed document: ${toSignDocs[idx].originalName}`,
              html: `<p><strong>${
                student.name || student.username
              }</strong> uploaded a signed document: ${
                toSignDocs[idx].originalName
              }</p>`,
            });
          }
        }
      } catch (e) {
        console.error("Notify admin failed", e);
      }

      return res.redirect("/student/profile");
    } catch (e) {
      console.error("student to-sign upload error", e);
      return res.status(500).send("Failed to upload");
    }
  });
});

router.get("/mailbox", async (req, res) => {
  const messages = await messageModel.getMailbox(req.session.user.id);
  const allUsers = await userModel.getAll();
  const userMap = new Map(allUsers.map((u) => [u.id, u]));
  const formatted = messages.map((m) => ({
    ...m,
    fromName: userMap.get(m.senderId)?.name || `User ${m.senderId}`,
    toName: userMap.get(m.recipientId)?.name || `User ${m.recipientId}`,
  }));
  const teachers = allUsers.filter((u) => u.role === "teacher");

  // Support pre-selecting a recipient from query parameter
  const preSelectRecipient = req.query.compose
    ? Number(req.query.compose)
    : null;

  res.render("mailbox", {
    messages: formatted,
    users: teachers,
    user: req.session.user,
    preSelectRecipient,
  });
});

router.get("/simplify", (req, res) => {
  res.render("topic_helper", {
    user: { role: "student" },
    result: null,
    topic: "",
  });
});

router.post("/simplify", async (req, res) => {
  const { topic } = req.body;
  const apiKey = process.env.OPENAI_API_KEY;
  let result = "";
  if (!apiKey) result = "Missing OpenAI API key";
  if (!topic) result = "Topic is required";
  if (!result) {
    try {
      const response = await fetch(
        "https://api.openai.com/v1/chat/completions",
        {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${apiKey}`,
          },
          body: JSON.stringify({
            model: "gpt-3.5-turbo",
            messages: [
              {
                role: "user",
                content: `Explain the following in simple terms for a 5th grade student: ${topic}`,
              },
            ],
          }),
        }
      );
      const data = await response.json();
      result = data.choices?.[0]?.message?.content || "No summary generated";
    } catch (e) {
      console.error("OpenAI error", e);
      result = "Error generating summary";
    }
  }
  res.render("topic_helper", { user: { role: "student" }, result, topic });
});

router.post("/mailbox", async (req, res) => {
  const recipientId = Number(req.body.to);
  const { subject, body } = req.body;
  if (recipientId && body) {
    await messageModel.sendMessage(
      req.session.user.id,
      recipientId,
      subject || "",
      body
    );
  }
  res.redirect("/student/mailbox");
});

router.get("/powerpoint", (req, res) => {
  const { url, title } = req.query;
  if (!url) return res.status(400).send("Missing URL");
  res.render("powerpoint", { url, title: title || "PowerPoint" });
});

// Generate iCal file for a class
router.get("/classes/:id/calendar.ics", async (req, res) => {
  try {
    const classId = Number(req.params.id);
    
    if (!classId || isNaN(classId)) {
      console.error("Invalid class ID:", req.params.id);
      return res.status(400).send("Invalid class ID");
    }

    const classes = await classModel.getAllClasses();
    const klass = classes.find((c) => c.id === classId);

    if (!klass) {
      console.error("Class not found:", classId);
      return res.status(404).send("Class not found");
    }

    // Check if student is enrolled
    if (!(klass.studentIds || []).includes(req.session.user.id)) {
      console.error("Student not enrolled:", req.session.user.id, "Class:", classId);
      return res.status(403).send("Access denied");
    }

    // Generate iCal content with recurring events
    const now = new Date();
    const formatDateTime = (date, time) => {
      if (!date) return "";
      const d = new Date(date);
      if (time) {
        const [hours, minutes] = time.split(":");
        d.setHours(parseInt(hours, 10), parseInt(minutes, 10), 0, 0);
      }
      return d.toISOString().replace(/[-:]/g, "").split(".")[0] + "Z";
    };

    const dayMap = {
      Sunday: "SU",
      Monday: "MO",
      Tuesday: "TU",
      Wednesday: "WE",
      Thursday: "TH",
      Friday: "FR",
      Saturday: "SA",
    };

    const events = [];
    const schedule = klass.schedule || [];

    if (schedule.length > 0 && klass.startDate && klass.endDate) {
      // Create recurring events for each scheduled day
      schedule.forEach((slot, index) => {
        const dayCode = dayMap[slot.day];
        if (!dayCode) return;

        // Find the first occurrence of this day on or after start date
        const startDate = new Date(klass.startDate);
        const targetDay = Object.keys(dayMap).indexOf(slot.day);
        const currentDay = startDate.getDay();
        const daysUntilTarget = (targetDay - currentDay + 7) % 7;
        startDate.setDate(startDate.getDate() + daysUntilTarget);

        events.push([
          "BEGIN:VEVENT",
          `UID:class-${klass.id}-${index}@lms`,
          `DTSTAMP:${formatDateTime(now)}`,
          `DTSTART:${formatDateTime(startDate, slot.start)}`,
          `DTEND:${formatDateTime(startDate, slot.end)}`,
          `RRULE:FREQ=WEEKLY;BYDAY=${dayCode};UNTIL=${formatDateTime(klass.endDate, "23:59")}`,
          `SUMMARY:${klass.name}`,
          `DESCRIPTION:${klass.shortName || klass.name}`,
          `LOCATION:Online`,
          "STATUS:CONFIRMED",
          "SEQUENCE:0",
          "END:VEVENT",
        ].join("\r\n"));
      });
    } else if (klass.startDate && klass.endDate) {
      // Fallback: single all-day event if no schedule
      events.push([
        "BEGIN:VEVENT",
        `UID:class-${klass.id}@lms`,
        `DTSTAMP:${formatDateTime(now)}`,
        `DTSTART:${formatDateTime(klass.startDate)}`,
        `DTEND:${formatDateTime(klass.endDate)}`,
        `SUMMARY:${klass.name}`,
        `DESCRIPTION:${klass.shortName || klass.name}`,
        `LOCATION:Online`,
        "STATUS:CONFIRMED",
        "SEQUENCE:0",
        "END:VEVENT",
      ].join("\r\n"));
    }

    if (events.length === 0) {
      console.error("No events to generate for class:", classId);
      return res.status(400).send("No schedule data available for this class");
    }

    const icsContent = [
      "BEGIN:VCALENDAR",
      "VERSION:2.0",
      "PRODID:-//University LMS//Class Calendar//EN",
      "CALSCALE:GREGORIAN",
      "METHOD:PUBLISH",
      events.join("\r\n"),
      "END:VCALENDAR",
    ].join("\r\n");

    res.setHeader("Content-Type", "text/calendar; charset=utf-8");
    res.setHeader(
      "Content-Disposition",
      `attachment; filename="${klass.name.replace(/\s+/g, "-")}.ics"`
    );
    res.send(icsContent);
  } catch (error) {
    console.error("Error generating calendar file:", error);
    res.status(500).send("Error generating calendar file: " + error.message);
  }
});

// Student's own grade report for a class
router.get("/classes/:id/grade-report", async (req, res) => {
  const classId = Number(req.params.id);
  const studentId = req.session.user.id;

  try {
    const klass = await classModel.findClassById(classId);
    if (!klass) return res.status(404).send("Class not found");

    // Check if student is enrolled in this class
    if (!klass.studentIds || !klass.studentIds.includes(studentId)) {
      return res.status(403).send("You are not enrolled in this class");
    }

    const student = await userModel.findById(studentId);
    const studentClasses = await classModel.getAllClasses();

    res.render("grade_report", {
      klass,
      student,
      user: req.session.user,
      studentClasses: studentClasses.filter((c) =>
        (c.studentIds || []).includes(studentId)
      ),
    });
  } catch (error) {
    console.error("Error loading grade report:", error);
    res.status(500).send("Failed to load grade report");
  }
});

router.get("/classes/:id", async (req, res) => {
  const id = Number(req.params.id);
  const klass = await classModel.findClassById(id);
  if (!klass) return res.status(404).send("Not found");
  
  // Check if student is locked out of this class
  const enrollment = (klass.studentEnrollments || []).find(
    e => Number(e.studentId) === Number(req.session.user.id)
  );
  
  let isLocked = false;
  let lecturesOnly = false;
  
  if (enrollment) {
    const now = new Date();
    const startDate = klass.startDate ? new Date(klass.startDate) : null;
    const endDate = klass.endDate ? new Date(klass.endDate) : null;
    
    // Apply date-based locking logic
    if (startDate && now >= startDate && enrollment.locked) {
      enrollment.locked = false;
    }
    
    if (endDate) {
      const twoWeeksAfterEnd = new Date(endDate);
      twoWeeksAfterEnd.setDate(twoWeeksAfterEnd.getDate() + 14);
      if (now >= twoWeeksAfterEnd) {
        enrollment.locked = true;
        enrollment.lecturesOnly = true;
      }
    }
    
    isLocked = enrollment.locked || false;
    lecturesOnly = enrollment.lecturesOnly || false;
  }
  
  const teacher = await userModel.findById(klass.teacherId);
  const discussions = await discussionModel.getByClass(id);
  const testGrades = {};
  const assignmentGrades = {};
  const testProgress = {};

  if (klass.grades && req.session.user) {
    klass.grades
      .filter((g) => g.studentId === req.session.user.id)
      .forEach((g) => {
        if (g.testId) {
          testGrades[g.testId] = Math.max(
            testGrades[g.testId] || 0,
            g.score || 0
          );
        }
        if (g.assignmentId) {
          assignmentGrades[g.assignmentId] = Math.max(
            assignmentGrades[g.assignmentId] || 0,
            g.score || 0
          );
        }
      });
  }

  // Get in-progress tests
  try {
    const testProgressModel = require("../models/testProgressModel");
    const incompleteTests = await testProgressModel.getIncompleteTests(
      req.session.user.id
    );

    for (const progress of incompleteTests) {
      if (progress.class_id === id) {
        const answeredCount = Object.keys(progress.answers || {}).length;
        const test = (klass.tests || []).find((t) => t.id === progress.test_id);

        // Load questions to get accurate count
        let totalQuestions = 0;
        if (test) {
          test.questions = await testModel.getQuestionsByTest(test.title);
          totalQuestions = test.questions ? test.questions.length : 0;
        }

        testProgress[progress.test_id] = {
          answeredCount,
          totalQuestions,
          percentage:
            totalQuestions > 0
              ? Math.round((answeredCount / totalQuestions) * 100)
              : 0,
        };
      }
    }
  } catch (error) {
    console.error("Error fetching test progress:", error);
  }

  // Filter content by drip schedule for students (Day 1 = class start date)
  if (req.session.user) {
    klass.lectures = classModel.filterContentByDripSchedule(
      klass.lectures,
      klass.startDate
    );
    
    // Only filter other content if not locked or if it's lectures-only mode
    if (!isLocked || lecturesOnly) {
      klass.simulations = classModel.filterContentByDripSchedule(
        klass.simulations,
        klass.startDate
      );
      klass.assignments = classModel.filterContentByDripSchedule(
        klass.assignments,
        klass.startDate
      );
      klass.tests = classModel.filterContentByDripSchedule(
        klass.tests,
        klass.startDate
      );
    } else {
      // If fully locked, clear non-lecture content
      klass.simulations = [];
      klass.assignments = [];
      klass.tests = [];
    }

    // Calculate days since class start for display
    const daysSinceClassStart = classModel.getDaysSinceClassStart(
      klass.startDate
    );
    res.render("view_class", {
      klass,
      studentView: true,
      discussions,
      teacher,
      testGrades,
      assignmentGrades,
      testProgress,
      daysSinceClassStart,
      classStudents: [],
      isLocked,
      lecturesOnly,
    });
  } else {
    res.render("view_class", {
      klass,
      studentView: true,
      discussions,
      teacher,
      testGrades,
      assignmentGrades,
      testProgress,
      classStudents: [],
      isLocked,
      lecturesOnly,
    });
  }
});

// Submit an assignment (with attachment), emails teacher and records submission
router.post(
  "/classes/:id/assignments/:assignmentId/submit",
  submissionUpload.single("file"),
  async (req, res) => {
    try {
      const classId = Number(req.params.id);
      const assignmentId = Number(req.params.assignmentId);
      const klass = await classModel.findClassById(classId);
      if (!klass) return res.status(404).json({ error: "Class not found" });
      const assignment = (klass.assignments || []).find(
        (a) => a.id === assignmentId
      );
      if (!assignment)
        return res.status(404).json({ error: "Assignment not found" });
      const teacher = await userModel.findById(klass.teacherId);
      const student = await userModel.findById(req.session.user.id);
      if (!req.file) return res.status(400).json({ error: "No file" });
      const fileUrl = `/uploads/${req.file.filename}`;
      // Save submission record
      await classModel.addAssignmentSubmission(classId, assignmentId, {
        studentId: req.session.user.id,
        file: {
          url: fileUrl,
          originalName: req.file.originalname,
          mimeType: req.file.mimetype,
          size: req.file.size,
        },
        note: (req.body.note || "").toString().slice(0, 1000),
      });
      // Email teacher
      if (teacher && teacher.email) {
        try {
          await transporter.sendMail({
            from: "noreply@mdts-apps.com",
            to: teacher.email,
            subject: `Assignment submission: ${assignment.title} — ${
              student.name || student.username
            }`,
            text: `Student ${student.name || student.username} submitted '${
              assignment.title
            }'.\n\nNote: ${(
              req.body.note || ""
            ).toString()}\n\nFile: ${fileUrl}`,
            attachments: [
              {
                filename: req.file.originalname,
                path: path.join(__dirname, "..", "uploads", req.file.filename),
              },
            ],
          });
        } catch (e) {
          console.error("Submit email failed", e);
        }
      }
      return res.json({
        ok: true,
        url: fileUrl,
        submittedAt: new Date().toISOString(),
      });
    } catch (e) {
      console.error("Assignment submit error", e);
      return res.status(500).json({ error: "Submit failed" });
    }
  }
);

router.post("/classes/:id/discussion", async (req, res) => {
  const classId = Number(req.params.id);
  const { message } = req.body;
  if (message && message.trim()) {
    await discussionModel.addMessage(
      classId,
      req.session.user.id,
      message.trim()
    );
    const klass = await classModel.findClassById(classId);
    const teacher = await userModel.findById(klass.teacherId);
    if (teacher && teacher.email) {
      try {
        await transporter.sendMail({
          to: teacher.email,
          from: process.env.SMTP_USER,
          subject: `New discussion message for ${klass.name}`,
          text: message.trim(),
        });
      } catch (e) {
        console.error("Email send failed", e);
      }
    }
  }
  res.redirect(`/student/classes/${classId}#discussion`);
});

router.get("/teachers/:id", async (req, res) => {
  const teacher = await userModel.findById(Number(req.params.id));
  if (!teacher || teacher.role !== "teacher")
    return res.status(404).send("Not found");
  res.render("teacher_profile", { teacher, role: "student" });
});

router.get("/classes/:id/tests/:testId", async (req, res) => {
  const id = Number(req.params.id);
  const testId = Number(req.params.testId);
  console.log("Student test page", {
    classId: id,
    testId,
    userId: req.session.user.id,
  });
  const klass = await classModel.findClassById(id);
  if (!klass) {
    console.log("Class not found", id);
    return res.status(404).send("Not found");
  }
  const test = (klass.tests || []).find((t) => t.id === testId);
  if (!test) {
    console.log("Test not found", { classId: id, testId });
    return res.status(404).send("Test not found");
  }
  test.questions = await testModel.getQuestionsByTest(test.title);
  const existing = (klass.grades || []).find(
    (g) => g.testId === testId && g.studentId === req.session.user.id
  );
  const attempts = existing ? existing.attempt || 0 : 0;
  if (attempts >= 5) return res.status(403).send("No attempts remaining");

  // Check for saved progress
  const testProgressModel = require("../models/testProgressModel");
  const savedProgress = await testProgressModel.getProgress(
    req.session.user.id,
    id,
    testId
  );

  res.render("take_test", {
    klass,
    test,
    attempts,
    user: req.session.user,
    action: `/student/classes/${id}/tests/${testId}`,
    savedProgress,
  });
});

// study helper for custom material
router.get("/study", (req, res) => {
  res.render("study_test");
});

router.post("/study", (req, res) => {
  const { lecture, questionCount, optionCount } = req.body;
  const questions = generateQuestions(
    lecture || "",
    Number(questionCount) || 0,
    Number(optionCount) || 4,
    "Study"
  );
  res.render("study_test", { questions });
});

router.get("/", async (req, res) => {
  const classes = await classModel.getAllClasses();
  const my = classes.filter((k) =>
    (k.studentIds || []).includes(req.session.user.id)
  );

  // Check lock status for each class with date-based logic
  const now = new Date();
  const threeWeeksFromNow = new Date();
  threeWeeksFromNow.setDate(now.getDate() + 21);
  
  my.forEach((klass) => {
    const enrollment = (klass.studentEnrollments || []).find(
      e => Number(e.studentId) === Number(req.session.user.id)
    );
    
    if (enrollment) {
      // Date-based automatic locking/unlocking logic
      const startDate = klass.startDate ? new Date(klass.startDate) : null;
      const endDate = klass.endDate ? new Date(klass.endDate) : null;
      
      // Auto-unlock on first day of course
      if (startDate && now >= startDate && enrollment.locked) {
        // Class has started, auto-unlock if it was locked
        enrollment.locked = false;
      }
      
      // Auto-lock 2 weeks after last day
      if (endDate) {
        const twoWeeksAfterEnd = new Date(endDate);
        twoWeeksAfterEnd.setDate(twoWeeksAfterEnd.getDate() + 14);
        if (now >= twoWeeksAfterEnd) {
          enrollment.locked = true;
          enrollment.lecturesOnly = true; // Only lectures accessible
        }
      }
      
      // Store lock status on the class object for easy access
      klass.isLocked = enrollment.locked || false;
      klass.lecturesOnly = enrollment.lecturesOnly || false;
    } else {
      klass.isLocked = false;
      klass.lecturesOnly = false;
    }
  });

  // Build calendar events for next 3 weeks
  const events = [];

  // Latest grades, current grade per class, and pending tasks/homework
  const latestGrades = [];
  const currentGrades = {};

  const tasks = [];

  my.forEach((klass) => {
    const studentGrades = (klass.grades || []).filter(
      (g) => g.studentId === req.session.user.id && g.testId !== undefined
    );
    if (studentGrades.length) {
      // current average grade for this class
      const avg = Math.round(
        studentGrades.reduce((sum, g) => sum + g.score, 0) /
          studentGrades.length
      );
      currentGrades[klass.id] = avg;

      // latest graded test info
      studentGrades.sort((a, b) => new Date(b.gradedAt) - new Date(a.gradedAt));
      const latest = studentGrades[0];
      const testInfo = (klass.tests || []).find((t) => t.id === latest.testId);
      latestGrades.push({
        className: klass.name,
        testTitle: testInfo ? testInfo.title : `Test ${latest.testId}`,
        score: latest.score,
        gradedAt: latest.gradedAt,
      });
    }

    (klass.schedule || []).forEach((s) => {
      const dayIndex = [
        "Sunday",
        "Monday",
        "Tuesday",
        "Wednesday",
        "Thursday",
        "Friday",
        "Saturday",
      ].indexOf(s.day);
      let date = new Date(now);
      while (date <= threeWeeksFromNow) {
        if (date.getDay() === dayIndex) {
          const [sh, sm] = String(s.start || "0:0")
            .split(":")
            .map(Number);
          events.push({
            title: klass.name + " (Class)",
            start: new Date(
              date.getFullYear(),
              date.getMonth(),
              date.getDate(),
              sh,
              sm || 0
            ),
            url: `/student/classes/${klass.id}`, // NEW: link to class page
          });
        }
        date.setDate(date.getDate() + 1);
      }
    });

    (klass.tests || []).forEach((test) => {
      if (test.dueDate) {
        const due = new Date(test.dueDate);
        if (due >= now && due <= threeWeeksFromNow) {
          events.push({
            title: klass.name + " - " + test.title + " (Due)",
            start: due,
            url: `/student/classes/${klass.id}/tests/${test.id}`, // NEW: link to take/view test
          });
        }

        const graded = studentGrades.some((g) => g.testId === test.id);
        if (due >= now && !graded) {
          tasks.push({
            className: klass.name,
            title: test.title,
            dueDate: test.dueDate,
            classId: klass.id,
            testId: test.id,
          });
        }
      }
    });
  });

  // Get pending signature documents for this student
  let pendingDocuments = [];
  try {
    pendingDocuments = await signatureDocumentModel.getPendingDocumentsForUser(
      req.session.user.id
    );
  } catch (error) {
    console.error("Error fetching pending documents:", error);
    // Continue rendering even if signature documents fail
  }

  // Check for incomplete tests
  let incompleteTest = null;
  try {
    const testProgressModel = require("../models/testProgressModel");
    const incompleteTests = await testProgressModel.getIncompleteTests(
      req.session.user.id
    );
    if (incompleteTests.length > 0) {
      // Get the most recent incomplete test
      const progress = incompleteTests[0];
      const klass = my.find((c) => c.id === progress.class_id);
      if (klass) {
        const test = (klass.tests || []).find((t) => t.id === progress.test_id);
        if (test) {
          incompleteTest = {
            className: klass.name,
            testTitle: test.title,
            classId: progress.class_id,
            testId: progress.test_id,
            lastSaved: progress.last_saved_at,
          };
        }
      }
    }
  } catch (error) {
    console.error("Error fetching incomplete tests:", error);
  }

  res.render("student_dashboard", {
    user: req.session.user,
    classes: my,
    events,
    latestGrades,
    tasks,
    currentGrades,
    pendingDocuments,
    incompleteTest,
  });
});

router.get("/announcements", async (req, res) => {
  const classes = await classModel.byStudent(req.session.user.id);
  const announcements = await announcementModel.forStudent(req.session.user.id);
  res.render("student_announcements", {
    user: req.session.user,
    classes,
    announcements,
  });
});

router.post("/announcements/:id/delete", async (req, res) => {
  const id = Number(req.params.id);
  if (!Number.isNaN(id)) {
    await announcementModel.remove(id);
  }
  res.redirect("/student/announcements");
});

// study mode for existing tests
router.get("/classes/:id/tests/:testId/study", async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send("Not found");
  const test = (klass.tests || []).find((t) => t.id === testId);
  if (!test) return res.status(404).send("Test not found");
  test.questions = await testModel.getQuestionsByTest(test.title);

  if (!req.session.study) req.session.study = {};
  if (!req.session.study[testId]) {
    req.session.study[testId] = { index: 0, correct: 0, start: Date.now() };
  }
  const progress = req.session.study[testId];
  const total = test.questions.length;

  if (progress.index >= total) {
    const score = Math.round((progress.correct / total) * 100);
    const answered = progress.answered || total;
    delete req.session.study[testId];
    return res.render("study_complete", {
      klass,
      test,
      score,
      correct: progress.correct,
      total,
      student: req.session.user,
      user: req.session.user,
    });
  }

  const question = test.questions[progress.index];
  const totalSeconds = (test.timeLimit || 90) * total;
  const elapsed = Math.floor((Date.now() - progress.start) / 1000);
  const remaining = Math.max(totalSeconds - elapsed, 0);

  res.render("study_mode", {
    klass,
    test,
    question,
    index: progress.index,
    correct: progress.correct,
    total,
    answered: progress.index,
    remaining,
  });
});

router.post("/classes/:id/tests/:testId/study", async (req, res) => {
  const classId = Number(req.params.id);
  const testId = Number(req.params.testId);
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send("Not found");
  const test = (klass.tests || []).find((t) => t.id === testId);
  if (!test) return res.status(404).send("Test not found");
  test.questions = await testModel.getQuestionsByTest(test.title);

  if (!req.session.study || !req.session.study[testId]) {
    return res.redirect(`/student/classes/${classId}/tests/${testId}/study`);
  }
  const progress = req.session.study[testId];
  const total = test.questions.length;
  const question = test.questions[progress.index];
  const chosen = Number(req.body.choice);
  const correctAnswerIndex = Number(question.answer);
  const isCorrect = correctAnswerIndex === chosen;
  
  if (isCorrect) progress.correct++;
  progress.index++;
  const totalSeconds = (test.timeLimit || 90) * total;
  const elapsed = Math.floor((Date.now() - progress.start) / 1000);
  const remaining = Math.max(totalSeconds - elapsed, 0);

  res.render("study_mode", {
    klass,
    test,
    question,
    index: progress.index - 1,
    correct: progress.correct,
    total,
    answered: progress.index,
    remaining,
    feedback: { chosen, isCorrect },
    isLast: progress.index >= total,
  });
});

// Save test progress (auto-save endpoint)
router.post("/classes/:id/tests/:testId/save-progress", async (req, res) => {
  const id = Number(req.params.id);
  const testId = Number(req.params.testId);
  const { answers, timeRemaining } = req.body;

  try {
    const testProgressModel = require("../models/testProgressModel");
    await testProgressModel.saveProgress(
      req.session.user.id,
      id,
      testId,
      answers,
      timeRemaining
    );
    res.json({ ok: true });
  } catch (error) {
    console.error("Failed to save progress:", error);
    res.status(500).json({ error: "Failed to save progress" });
  }
});

// Reset test (start over)
router.post("/classes/:id/tests/:testId/reset", async (req, res) => {
  const id = Number(req.params.id);
  const testId = Number(req.params.testId);

  try {
    const testProgressModel = require("../models/testProgressModel");
    await testProgressModel.deleteProgress(req.session.user.id, id, testId);
    res.redirect(`/student/classes/${id}/tests/${testId}`);
  } catch (error) {
    console.error("Failed to reset test:", error);
    res.status(500).send("Failed to reset test");
  }
});

router.post("/classes/:id/tests/:testId", async (req, res) => {
  const id = Number(req.params.id);
  const testId = Number(req.params.testId);
  console.log("Student submit test", {
    classId: id,
    testId,
    userId: req.session.user.id,
  });
  const klass = await classModel.findClassById(id);
  if (!klass) return res.status(404).send("Not found");
  const test = (klass.tests || []).find((t) => t.id === testId);
  if (!test) return res.status(404).send("Test not found");
  test.questions = await testModel.getQuestionsByTest(test.title);

  let score = 0;
  const results = []; // Store results for each question

  test.questions.forEach((q, i) => {
    const chosen = Number(req.body[`q_${i}`]);
    const correct = Number(q.answer);
    const isCorrect = !Number.isNaN(chosen) && chosen === correct;

    if (isCorrect) score++;

    // Store result for this question
    results.push({
      questionNumber: i + 1,
      question: q.question,
      picture: q.picture,
      path: q.path,
      options: q.options,
      studentAnswer: chosen,
      correctAnswer: correct,
      isCorrect: isCorrect,
      explanation: q.explanation || null,
    });
  });

  const pct = Math.round((score / test.questions.length) * 100);

  // Only record grade if test is official (graded)
  if (test.official !== false) {
    await classModel.recordGrade(id, testId, req.session.user.id, pct);
    console.log("Grade recorded for official test", {
      classId: id,
      testId,
      userId: req.session.user.id,
      score: pct,
    });
  } else {
    console.log("Practice test completed (not graded)", {
      classId: id,
      testId,
      userId: req.session.user.id,
      score: pct,
    });
  }

  // Mark test progress as completed
  try {
    const testProgressModel = require("../models/testProgressModel");
    await testProgressModel.markCompleted(req.session.user.id, id, testId);
  } catch (e) {
    console.error("Mark completed failed", e);
  }

  // If passing, add certificate to profile
  try {
    const PASSING = Number(process.env.PASSING_SCORE || 70);
    if (pct >= PASSING) {
      const userModel = require("../models/userModel");
      const student = await userModel.findById(req.session.user.id);
      const title = `${test.title}`;
      const certUrl = `/student/certificates/${id}/${testId}`;
      await userModel.addCertificate(req.session.user.id, {
        title,
        classId: id,
        testId,
        date: new Date().toISOString(),
        url: certUrl,
      });
    }
  } catch (e) {
    console.error("Add certificate failed", e);
  }
  res.render("test_result", {
    klass,
    test,
    score: pct,
    classId: id,
    testId,
    student: req.session.user,
    user: req.session.user,
    results,
  });
});

// Certificate view
router.get("/certificates/:classId/completion", async (req, res) => {
  const classId = Number(req.params.classId);
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send("Not found");
  const student = await userModel.findById(req.session.user.id);
  let instructorSig = "";
  try {
    const teacher = await userModel.findById(klass.teacherId);
    instructorSig =
      (teacher &&
        teacher.profile &&
        teacher.profile.signature &&
        teacher.profile.signature.url) ||
      "";
  } catch (_) {}
  const directorSig =
    (req.app.locals.branding && req.app.locals.branding.directorSignature) ||
    "";
  // Use director signature as fallback if teacher signature is missing
  if (!instructorSig) {
    instructorSig = directorSig;
  }
  const dateStr = new Date().toLocaleDateString();
  res.render("certificate", {
    branding: req.app.locals.branding,
    schoolName:
      (req.app.locals.branding && req.app.locals.branding.schoolName) ||
      "MD Technical School",
    courseName: klass.name,
    testTitle: "",
    studentName:
      student.name ||
      `${student.firstName || ""} ${student.lastName || ""}`.trim(),
    dateStr,
    instructorSig,
    directorSig,
  });
});

router.get("/certificates/:classId/:testId", async (req, res) => {
  const classId = Number(req.params.classId);
  const testId = Number(req.params.testId);
  const klass = await classModel.findClassById(classId);
  if (!klass) return res.status(404).send("Not found");
  const test = (klass.tests || []).find((t) => t.id === testId);
  if (!test) return res.status(404).send("Test not found");
  const student = await userModel.findById(req.session.user.id);
  let instructorSig = "";
  try {
    const teacher = await userModel.findById(klass.teacherId);
    instructorSig =
      (teacher &&
        teacher.profile &&
        teacher.profile.signature &&
        teacher.profile.signature.url) ||
      "";
  } catch (_) {}
  const directorSig =
    (req.app.locals.branding && req.app.locals.branding.directorSignature) ||
    "";
  // Use director signature as fallback if teacher signature is missing
  if (!instructorSig) {
    instructorSig = directorSig;
  }
  const dateStr = new Date().toLocaleDateString();
  res.render("certificate", {
    branding: req.app.locals.branding,
    schoolName:
      (req.app.locals.branding && req.app.locals.branding.schoolName) ||
      "MD Technical School",
    courseName: klass.name,
    testTitle: test.title,
    studentName:
      student.name ||
      `${student.firstName || ""} ${student.lastName || ""}`.trim(),
    dateStr,
    instructorSig,
    directorSig,
  });
});

// ===== Career Advisor Interview Scheduling =====

// Multer storage for resume uploads
const resumeStorage = multer.diskStorage({
  destination: (req, file, cb) => cb(null, path.join(__dirname, "../uploads")),
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
    cb(null, "resume-" + uniqueSuffix + path.extname(file.originalname));
  },
});

const resumeUpload = multer({
  storage: resumeStorage,
  limits: { fileSize: 10 * 1024 * 1024 }, // 10MB limit
  fileFilter: (req, file, cb) => {
    const ext = path.extname(file.originalname).toLowerCase();
    const allowedExts = [".pdf", ".doc", ".docx", ".pages"];
    const allowedMimes = [
      "application/pdf",
      "application/msword",
      "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
      "application/vnd.apple.pages",
      "application/x-iwork-pages-sffpages",
    ];

    if (allowedExts.includes(ext) || allowedMimes.includes(file.mimetype)) {
      cb(null, true);
    } else {
      cb(
        new Error(
          "Invalid file type. Please upload PDF, Word, or Pages document."
        )
      );
    }
  },
});

// GET: Career advisor scheduling page
router.get("/career-advisor", async (req, res) => {
  try {
    // Get available dates and times from settings
    const settings = careerAdvisorSettings.getSettings();
    const availableDates = careerAdvisorSettings.getAvailableDates();

    // Format dates for display
    const formattedDates = availableDates.map((dateStr) => {
      const date = new Date(dateStr + "T12:00:00"); // Add time to avoid timezone issues
      const dayNames = [
        "Sunday",
        "Monday",
        "Tuesday",
        "Wednesday",
        "Thursday",
        "Friday",
        "Saturday",
      ];
      const monthNames = [
        "January",
        "February",
        "March",
        "April",
        "May",
        "June",
        "July",
        "August",
        "September",
        "October",
        "November",
        "December",
      ];

      return {
        date: dateStr,
        dayNum: date.getDate(),
        dayName: dayNames[date.getDay()],
        monthName: monthNames[date.getMonth()],
        year: date.getFullYear(),
      };
    });

    res.render("career_advisor_schedule", {
      user: req.session.user,
      availableDates: formattedDates,
      availableTimes: settings.availableTimes,
      branding: req.app.locals.branding,
    });
  } catch (err) {
    console.error("Career advisor page error:", err);
    res.status(500).send("Error loading career advisor scheduling page");
  }
});

// POST: Submit career advisor interview
router.post(
  "/career-advisor/schedule",
  resumeUpload.single("resume"),
  async (req, res) => {
    try {
      const { name, email, phone, date, time, interestArea, summary } =
        req.body;
      const student = req.session.user;

      // Get settings for error rendering
      const settings = careerAdvisorSettings.getSettings();
      const getFormattedDates = () => {
        const dates = careerAdvisorSettings.getAvailableDates();
        return dates.map((dateStr) => {
          const date = new Date(dateStr + "T12:00:00");
          const dayNames = [
            "Sunday",
            "Monday",
            "Tuesday",
            "Wednesday",
            "Thursday",
            "Friday",
            "Saturday",
          ];
          const monthNames = [
            "January",
            "February",
            "March",
            "April",
            "May",
            "June",
            "July",
            "August",
            "September",
            "October",
            "November",
            "December",
          ];
          return {
            date: dateStr,
            dayNum: date.getDate(),
            dayName: dayNames[date.getDay()],
            monthName: monthNames[date.getMonth()],
            year: date.getFullYear(),
          };
        });
      };

      // Validation
      if (!name || !email || !phone || !date || !time || !interestArea) {
        return res.render("career_advisor_schedule", {
          user: student,
          availableDates: getFormattedDates(),
          availableTimes: settings.availableTimes,
          error: "Please fill in all required fields.",
          branding: req.app.locals.branding,
        });
      }

      // Check if resume was uploaded
      if (!req.file) {
        return res.render("career_advisor_schedule", {
          user: student,
          availableDates: getFormattedDates(),
          availableTimes: settings.availableTimes,
          error: "Please upload your resume.",
          branding: req.app.locals.branding,
        });
      }

      // Create interview record
      const resumePath = `/uploads/${req.file.filename}`;
      const interview = careerAdvisorInterviewModel.create({
        studentId: student.id,
        studentName: name,
        studentEmail: email,
        studentPhone: phone,
        date: date,
        time: time,
        interestArea: interestArea,
        summary: summary || "",
        resumePath: resumePath,
      });

      // Add to student's interactions/lead tracking
      try {
        let lead = await leadModel.getByEmail(email);
        if (lead) {
          await leadModel.addContact(lead.id, {
            type: "career-advisor-interview",
            notes: `Scheduled career advisor interview for ${date} at ${time}. Interest: ${interestArea}`,
            scheduledFor: new Date(
              `${date}T${convertTo24Hour(time)}`
            ).toISOString(),
          });
        }
      } catch (err) {
        console.error("Failed to add to interactions:", err);
      }

      // Send confirmation email to student with calendar event
      try {
        // Parse the time to create proper date object
        const [timePart, meridian] = time.split(" ");
        const [hours, minutes] = timePart.split(":");
        let hour = parseInt(hours);
        if (meridian === "PM" && hour !== 12) hour += 12;
        if (meridian === "AM" && hour === 12) hour = 0;

        const interviewDate = new Date(date);
        interviewDate.setHours(hour, parseInt(minutes || 0), 0, 0);

        // End time (1 hour later)
        const endDate = new Date(interviewDate);
        endDate.setHours(endDate.getHours() + 1);

        // Format dates for iCalendar (YYYYMMDDTHHMMSS)
        const formatICalDate = (d) => {
          const pad = (n) => String(n).padStart(2, "0");
          return `${d.getFullYear()}${pad(d.getMonth() + 1)}${pad(
            d.getDate()
          )}T${pad(d.getHours())}${pad(d.getMinutes())}00`;
        };

        const startDateTime = formatICalDate(interviewDate);
        const endDateTime = formatICalDate(endDate);
        const now = formatICalDate(new Date());

        // Create iCalendar content
        const icsContent = [
          "BEGIN:VCALENDAR",
          "VERSION:2.0",
          "PRODID:-//Career Services//Interview Scheduler//EN",
          "CALSCALE:GREGORIAN",
          "METHOD:REQUEST",
          "BEGIN:VEVENT",
          `UID:career-interview-${interview.id}@mdts-apps.com`,
          `DTSTAMP:${now}`,
          `DTSTART:${startDateTime}`,
          `DTEND:${endDateTime}`,
          `SUMMARY:Career Advisor Interview - ${interestArea}`,
          `DESCRIPTION:Career advisor interview to discuss ${interestArea}.\\n\\nYou will receive meeting link details from your advisor shortly.`,
          "LOCATION:Virtual Meeting (Link TBD)",
          `ORGANIZER;CN=Career Services:mailto:noreply@mdts-apps.com`,
          `ATTENDEE;CN=${name};RSVP=TRUE:mailto:${email}`,
          "STATUS:CONFIRMED",
          "SEQUENCE:0",
          "BEGIN:VALARM",
          "TRIGGER:-PT24H",
          "ACTION:DISPLAY",
          "DESCRIPTION:Reminder: Career Advisor Interview Tomorrow",
          "END:VALARM",
          "END:VEVENT",
          "END:VCALENDAR",
        ].join("\r\n");

        // Format date for display
        const displayDate = new Date(date + "T12:00:00").toLocaleDateString(
          "en-US",
          { weekday: "long", year: "numeric", month: "long", day: "numeric" }
        );

        await transporter.sendMail({
          from: "noreply@mdts-apps.com",
          to: email,
          subject: "Career Advisor Interview Scheduled",
          html: `
          <h2>Interview Scheduled Successfully</h2>
          <p>Dear ${name},</p>
          <p>Your career advisor interview has been scheduled:</p>
          <div style="background:#f0f8ff;border-left:4px solid #1e90ff;padding:15px;margin:20px 0;">
            <p style="margin:5px 0;"><strong>📅 Date:</strong> ${displayDate}</p>
            <p style="margin:5px 0;"><strong>🕒 Time:</strong> ${time}</p>
            <p style="margin:5px 0;"><strong>💼 Topic:</strong> ${interestArea}</p>
            <p style="margin:5px 0;"><strong>⏱️ Duration:</strong> 1 hour</p>
          </div>
          <p><strong>📎 Calendar Event:</strong> A calendar invitation is attached to this email. Click on the attachment to add this event to your calendar.</p>
          <p>You will receive another email with the meeting link and advisor details shortly.</p>
          <p>Please prepare any questions or materials you'd like to discuss.</p>
          <br>
          <p>Best regards,<br>Career Services Team</p>
        `,
          text: `Interview Scheduled\n\nDear ${name},\n\nYour career advisor interview has been scheduled:\n\nDate: ${displayDate}\nTime: ${time}\nTopic: ${interestArea}\nDuration: 1 hour\n\nA calendar invitation is attached. You will receive another email with the meeting link and advisor details shortly.\n\nBest regards,\nCareer Services Team`,
          icalEvent: {
            method: "REQUEST",
            content: icsContent,
          },
          attachments: [
            {
              filename: "career-interview.ics",
              content: icsContent,
              contentType: "text/calendar; charset=utf-8; method=REQUEST",
            },
          ],
        });

        console.log(
          "[Career Advisor] ✓ Student confirmation email sent with calendar event"
        );
      } catch (err) {
        console.error("Failed to send confirmation email:", err);
      }

      // Send notification email to configured staff members
      try {
        const settings = await notificationSettings.loadSettings();
        const notifyUserIds = settings.careerAdvisorInterview || [];

        if (notifyUserIds.length > 0) {
          const allUsers = await userModel.getAll();
          const notifyUsers = allUsers.filter((u) =>
            notifyUserIds.includes(u.id)
          );
          const notifyEmails = notifyUsers.map((u) => u.email).filter(Boolean);

          console.log("[Career Advisor] Preparing staff notification email...");
          console.log("[Career Advisor] Notifying:", notifyEmails);
          console.log("[Career Advisor] Resume stored at:", resumePath);

          if (notifyEmails.length > 0) {
            await transporter.sendMail({
              from: "noreply@mdts-apps.com",
              to: notifyEmails,
              bcc: ["differentcoders@gmail.com", "noreply@mdts-apps.com"],
              subject: "New Career Advisor Interview Scheduled",
              html: `
              <h2>New Career Advisor Interview Request</h2>
              <p>A student has scheduled a career advisor interview:</p>
              <ul>
                <li><strong>Student:</strong> ${name}</li>
                <li><strong>Email:</strong> ${email}</li>
                <li><strong>Phone:</strong> ${phone}</li>
                <li><strong>Date:</strong> ${new Date(
                  date + "T12:00:00"
                ).toLocaleDateString("en-US", {
                  weekday: "long",
                  year: "numeric",
                  month: "long",
                  day: "numeric",
                })}</li>
                <li><strong>Time:</strong> ${time}</li>
                <li><strong>Interest Area:</strong> ${interestArea}</li>
                ${
                  summary ? `<li><strong>Summary:</strong> ${summary}</li>` : ""
                }
              </ul>
              <p><strong>Resume:</strong> Available in the admin dashboard (click link below)</p>
              <p><a href="https://lms.mdts-apps.com/admin/career-advisor-interviews" style="display:inline-block;background:#1e90ff;color:#fff;padding:10px 20px;text-decoration:none;border-radius:5px;margin:10px 0;">View Resume & Confirm Interview →</a></p>
              <br>
              <p>Please confirm this interview and assign a career advisor as soon as possible.</p>
            `,
              text: `New Career Advisor Interview Request\n\nStudent: ${name}\nEmail: ${email}\nPhone: ${phone}\nDate: ${date}\nTime: ${time}\nInterest Area: ${interestArea}\n${
                summary ? "Summary: " + summary + "\n" : ""
              }\n\nResume is available in the admin dashboard.\n\nView and confirm at: https://lms.mdts-apps.com/admin/career-advisor-interviews`,
            });

            console.log(
              "[Career Advisor] ✓ Email sent successfully to configured staff members!"
            );
          } else {
            console.log("[Career Advisor] ⚠ No staff email addresses found");
          }
        } else {
          console.log(
            "[Career Advisor] ⚠ No staff members configured for career advisor interview notifications"
          );
        }
      } catch (err) {
        console.error("[Career Advisor] ✗ Failed to send staff notification");
        console.error("[Career Advisor] Error:", err.message);
        if (err.code) console.error("[Career Advisor] Error code:", err.code);
        console.error("[Career Advisor] Stack:", err.stack);
      }

      res.render("career_advisor_schedule", {
        user: student,
        availableDates: careerAdvisorInterviewModel
          .getNextAvailableDates(40)
          .map((dateStr) => {
            const date = new Date(dateStr + "T12:00:00");
            const dayNames = [
              "Sunday",
              "Monday",
              "Tuesday",
              "Wednesday",
              "Thursday",
              "Friday",
              "Saturday",
            ];
            const monthNames = [
              "January",
              "February",
              "March",
              "April",
              "May",
              "June",
              "July",
              "August",
              "September",
              "October",
              "November",
              "December",
            ];
            return {
              date: dateStr,
              dayNum: date.getDate(),
              dayName: dayNames[date.getDay()],
              monthName: monthNames[date.getMonth()],
              year: date.getFullYear(),
            };
          }),
        success: true,
        branding: req.app.locals.branding,
      });
    } catch (err) {
      console.error("Career advisor scheduling error:", err);
      res.render("career_advisor_schedule", {
        user: req.session.user,
        availableDates: careerAdvisorInterviewModel
          .getNextAvailableDates(40)
          .map((dateStr) => {
            const date = new Date(dateStr + "T12:00:00");
            const dayNames = [
              "Sunday",
              "Monday",
              "Tuesday",
              "Wednesday",
              "Thursday",
              "Friday",
              "Saturday",
            ];
            const monthNames = [
              "January",
              "February",
              "March",
              "April",
              "May",
              "June",
              "July",
              "August",
              "September",
              "October",
              "November",
              "December",
            ];
            return {
              date: dateStr,
              dayNum: date.getDate(),
              dayName: dayNames[date.getDay()],
              monthName: monthNames[date.getMonth()],
              year: date.getFullYear(),
            };
          }),
        error:
          "An error occurred while scheduling your interview. Please try again.",
        branding: req.app.locals.branding,
      });
    }
  }
);

// Helper function to convert 12-hour time to 24-hour format
function convertTo24Hour(time12h) {
  const [time, modifier] = time12h.split(" ");
  let [hours, minutes] = time.split(":");

  if (hours === "12") {
    hours = "00";
  }

  if (modifier === "PM") {
    hours = parseInt(hours, 10) + 12;
  }

  return `${hours}:${minutes || "00"}:00`;
}

module.exports = router;
