const express = require('express');
const router = express.Router();
const nodemailer = require('nodemailer');
const interviewModel = require('../models/interviewModel');
const userModel = require('../models/userModel');
const leadModel = require('../models/leadModel');

// Email transporter
const transporter = nodemailer.createTransport({
  host: process.env.SMTP_HOST || 'mail.mdts-apps.com',
  port: Number(process.env.SMTP_PORT || 465),
  secure: true,
  auth: { user: process.env.SMTP_USER || 'no-reply@mdts-apps.com', pass: process.env.SMTP_PASS || 'TheMadden04!' }
});

function to24h(str) {
  // expects HH:MM (24h)
  const [h, m] = String(str).split(':').map(n => Number(n));
  return { h, m };
}

function validateSlot(dateStr, timeStr) {
  const d = new Date(dateStr + 'T00:00:00');
  if (isNaN(d.getTime())) return { ok: false, message: 'Invalid date' };
  const { h, m } = to24h(timeStr || '');
  if (!Number.isInteger(h) || !Number.isInteger(m)) return { ok: false, message: 'Invalid time' };
  const day = d.getDay(); // 0 Sun, 1 Mon, ... 6 Sat
  if (day === 0) return { ok: false, message: 'Sundays unavailable' };
  if (h < 8 || h > 20 || (h === 20 && m > 0)) return { ok: false, message: 'Outside hours (8:00–20:00)' };
  const scheduled = new Date(d);
  scheduled.setHours(h, m, 0, 0);
  if (scheduled.getTime() < Date.now() + 5 * 60 * 1000) return { ok: false, message: 'Must be in the future' };
  return { ok: true, scheduled };
}

function buildICS({ start, end, summary, description, uid, organizer, location }) {
  const dt = (date) => new Date(date).toISOString().replace(/[-:]/g, '').replace(/\.\d{3}Z$/, 'Z');
  const lines = [
    'BEGIN:VCALENDAR',
    'VERSION:2.0',
    'PRODID:-//MDTS LMS//Interviews//EN',
    'CALSCALE:GREGORIAN',
    'METHOD:REQUEST',
    'BEGIN:VEVENT',
    `UID:${uid}`,
    `DTSTAMP:${dt(new Date())}`,
    `DTSTART:${dt(start)}`,
    `DTEND:${dt(end)}`,
    `SUMMARY:${summary}`,
    `DESCRIPTION:${(description||'').replace(/\n/g,'\\n')}`,
    organizer ? `ORGANIZER:MAILTO:${organizer}` : '',
    location ? `LOCATION:${location}` : '',
    'END:VEVENT',
    'END:VCALENDAR'
  ].filter(Boolean);
  return lines.join('\r\n');
}

// Student schedules an interview
router.post('/schedule-interview', async (req, res) => {
  try {
    const { studentName, studentEmail, studentPhone, date, time, notes } = req.body;
    const v = validateSlot(date, time);
    if (!v.ok) {
      return res.status(400).render('pending', { user: { name: studentName }, error: v.message, selfPay: false, financialAid: false });
    }
    const start = v.scheduled;
    const end = new Date(start.getTime() + 30 * 60000);
    const interview = await interviewModel.create({ studentName, studentEmail, studentPhone, scheduledAt: start, durationMinutes: 30, notes: notes || '' });

    // Notify admins with calendar invite
    const admins = await userModel.getByRole('admin');
    const toEmails = admins.filter(a => a && a.email && a.active !== false).map(a => a.email);
    if (toEmails.length) {
      const ics = buildICS({
        start,
        end,
        summary: `Student Interview: ${studentName}`,
        description: `30-minute phone call with ${studentName} (${studentEmail}${studentPhone ? ', ' + studentPhone : ''}).`,
        uid: `mdts-interview-${interview.id}@mdts-apps.com`,
        organizer: 'no-reply@mdts-apps.com',
        location: 'Phone Call'
      });
      await transporter.sendMail({
        from: 'no-reply@mdts-apps.com',
        to: toEmails.join(','),
        subject: `New Interview Scheduled: ${studentName} – ${start.toLocaleString()}`,
        html: `<p>A student scheduled a 30-minute phone interview.</p>
               <ul>
                 <li><strong>Name:</strong> ${studentName}</li>
                 <li><strong>Email:</strong> ${studentEmail}</li>
                 ${studentPhone ? `<li><strong>Phone:</strong> ${studentPhone}</li>` : ''}
                 <li><strong>When:</strong> ${start.toLocaleString()} – ${end.toLocaleTimeString()}</li>
                 ${notes ? `<li><strong>Notes:</strong> ${(notes||'').replace(/</g,'&lt;')}</li>` : ''}
               </ul>
               <p>Open Admin › Interviews to assign an admin.</p>`,
        alternatives: [{ contentType: 'text/calendar; method=REQUEST', content: ics, headers: { 'Content-Class': 'urn:content-classes:calendarmessage' } }]
      });
    }

    // Record in lead contacts as a note
    try {
      const lead = await leadModel.upsertByEmail({ name: studentName || '', email: (studentEmail||'').trim().toLowerCase(), phone: studentPhone || '', interestPercent: 0, source: 'interview' });
      if (lead?.id) {
        const n = `Scheduled interview with Student ${studentName} on ${start.toLocaleString()} (30 min).${notes ? `\nNotes: ${notes}` : ''}`;
        await leadModel.addContact(lead.id, { direction: 'inbound', method: 'interview', note: n });
      }
    } catch (e) { console.error('lead note (interview) failed', e); }

    return res.render('pending', { user: { name: studentName, email: studentEmail }, scheduled: { at: start.toISOString() }, selfPay: false, financialAid: false });
  } catch (e) {
    console.error('Schedule interview error', e);
    return res.status(500).render('pending', { user: {}, error: 'Failed to schedule. Please try again.', selfPay: false, financialAid: false });
  }
});

// Admin schedules a follow-up interview for an existing interview
router.post('/admin/interviews/:id/follow-up', async (req, res) => {
  if (!req.session || req.session.role !== 'admin') return res.status(403).json({ error: 'forbidden' });
  try {
    const baseId = Number(req.params.id);
    const base = await interviewModel.findById(baseId);
    if (!base) return res.status(404).json({ error: 'not found' });
    const { date, time, notes } = req.body || {};
    const v = validateSlot(date, time);
    if (!v.ok) return res.status(400).json({ error: v.message });
    const start = v.scheduled;
    const end = new Date(start.getTime() + 30 * 60000);
    const interview = await interviewModel.create({
      studentName: base.studentName,
      studentEmail: base.studentEmail,
      studentPhone: base.studentPhone,
      scheduledAt: start,
      durationMinutes: 30,
      notes: notes ? `Follow-up: ${notes}` : 'Follow-up'
    });

    // Notify admins with calendar invite
    const admins = await userModel.getByRole('admin');
    const toEmails = admins.filter(a => a && a.email && a.active !== false).map(a => a.email);
    if (toEmails.length) {
      const ics = buildICS({
        start,
        end,
        summary: `Follow-up Call: ${base.studentName}`,
        description: `30-minute follow-up phone call with ${base.studentName} (${base.studentEmail}${base.studentPhone ? ', ' + base.studentPhone : ''}).${notes ? `\nNotes: ${notes}` : ''}`,
        uid: `mdts-interview-follow-${interview.id}@mdts-apps.com`,
        organizer: 'no-reply@mdts-apps.com',
        location: 'Phone Call'
      });
      await transporter.sendMail({
        from: 'no-reply@mdts-apps.com',
        to: toEmails.join(','),
        subject: `Follow-up Interview Scheduled: ${base.studentName} – ${start.toLocaleString()}`,
        html: `<p>A follow-up 30-minute phone call was scheduled.</p>
               <ul>
                 <li><strong>Name:</strong> ${base.studentName}</li>
                 <li><strong>Email:</strong> ${base.studentEmail}</li>
                 ${base.studentPhone ? `<li><strong>Phone:</strong> ${base.studentPhone}</li>` : ''}
                 <li><strong>When:</strong> ${start.toLocaleString()} – ${end.toLocaleTimeString()}</li>
                 ${notes ? `<li><strong>Notes:</strong> ${(notes||'').replace(/</g,'&lt;')}</li>` : ''}
               </ul>
               <p>Open Admin › Interviews to assign an admin.</p>`,
        alternatives: [{ contentType: 'text/calendar; method=REQUEST', content: ics, headers: { 'Content-Class': 'urn:content-classes:calendarmessage' } }]
      });
    }

    // Record in lead contacts as a note
    try {
      const lead = await leadModel.upsertByEmail({ name: base.studentName || '', email: (base.studentEmail||'').trim().toLowerCase(), phone: base.studentPhone || '', interestPercent: 0, source: 'interview-followup' });
      if (lead?.id) {
        const n = `Scheduled follow-up interview with Student ${base.studentName} on ${start.toLocaleString()} (30 min).${notes ? `\nNotes: ${notes}` : ''}`;
        await leadModel.addContact(lead.id, { direction: 'outbound', method: 'interview-followup', note: n });
      }
    } catch (e) { console.error('lead note (follow-up) failed', e); }

    return res.json({ ok: true, id: interview.id, at: start.toISOString() });
  } catch (e) {
    console.error('Admin follow-up schedule error', e);
    return res.status(500).json({ error: 'failed' });
  }
});

// Admin: interviews calendar page
router.get('/admin/interviews', async (req, res) => {
  if (!req.session || req.session.role !== 'admin') return res.redirect('/login');
  const admins = await userModel.getByRole('admin');
  res.render('admin_interviews', { user: req.session.user, admins });
});

// Admin: list interviews in a range
router.get('/admin/interviews/data', async (req, res) => {
  if (!req.session || req.session.role !== 'admin') return res.status(403).json({ error: 'forbidden' });
  try {
    const start = req.query.start ? new Date(req.query.start) : new Date();
    const end = req.query.end ? new Date(req.query.end) : new Date(start.getTime() + 7 * 86400000);
    const list = await interviewModel.listBetween(start, end);
    res.json({ items: list });
  } catch (e) {
    console.error(e);
    res.status(500).json({ error: 'failed' });
  }
});

// Admin: assign interview to an admin
router.post('/admin/interviews/:id/assign', async (req, res) => {
  if (!req.session || req.session.role !== 'admin') return res.status(403).json({ error: 'forbidden' });
  try {
    const id = Number(req.params.id);
    const adminId = req.body.adminId ? Number(req.body.adminId) : null;
    const updated = await interviewModel.assignAdmin(id, adminId);
    res.json({ ok: true, item: updated });
  } catch (e) {
    console.error(e);
    res.status(500).json({ error: 'failed' });
  }
});

// Admin: add note to interview (stores as lead contact)
router.post('/admin/interviews/:id/note', async (req, res) => {
  if (!req.session || req.session.role !== 'admin') return res.status(403).json({ error: 'forbidden' });
  try {
    const id = Number(req.params.id);
    const { note, method } = req.body || {};
    if (!note || !String(note).trim()) return res.status(400).json({ error: 'empty' });
    const item = await interviewModel.findById(id);
    if (!item) return res.status(404).json({ error: 'not found' });
    const lead = await leadModel.upsertByEmail({ name: item.studentName || '', email: (item.studentEmail||'').trim().toLowerCase(), phone: item.studentPhone || '', interestPercent: 0, source: 'interview-note' });
    if (lead?.id) {
      await leadModel.addContact(lead.id, { direction: 'outbound', method: method || 'interview-note', note: String(note) });
      return res.json({ ok: true });
    }
    return res.status(500).json({ error: 'lead failed' });
  } catch (e) {
    console.error('admin interview note error', e);
    return res.status(500).json({ error: 'failed' });
  }
});

router.delete('/admin/interviews/:id', async (req, res) => {
  if (!req.session || req.session.role !== 'admin') return res.status(403).json({ error: 'forbidden' });
  try {
    const id = Number(req.params.id);
    if (!Number.isInteger(id)) return res.status(400).json({ error: 'invalid id' });
    const interview = await interviewModel.findById(id);
    if (!interview) return res.status(404).json({ error: 'not found' });
    await interviewModel.remove(id);
    res.json({ ok: true });
  } catch (e) {
    console.error('admin interview delete error', e);
    res.status(500).json({ error: 'failed' });
  }
});

module.exports = router;
// Also expose a simple page to schedule directly
router.get('/schedule-interview', (req, res) => {
  const user = (req.session && req.session.user) ? req.session.user : {};
  res.render('schedule_interview', { user, message: req.query.msg || null, error: null });
});
