/**
 * Mobile App API Routes
 * 
 * All endpoints return JSON responses for mobile app consumption.
 * Authentication is handled via session or Bearer token.
 */

const express = require('express');
const router = express.Router();

// Database
const db = require('../models/db');

// Models
const userModel = require('../models/userModel');
const classModel = require('../models/classModel');
const testModel = require('../models/testModel');
const discussionModel = require('../models/discussionModel');
const messageModel = require('../models/messageModel');
const announcementModel = require('../models/announcementModel');
const signatureDocumentModel = require('../models/signatureDocumentModel');
const courseModel = require('../models/courseModel');
const todoModel = require('../models/todoModel');
const testProgressModel = require('../models/testProgressModel');
const leadModel = require('../models/leadModel');

// ============================================================================
// AUTHENTICATION
// ============================================================================

/**
 * POST /api/auth/login
 * Authenticate user and return session token
 * 
 * Expected Response:
 * {
 *   success: true,
 *   user: { id, email, name, firstName, lastName, role, avatar },
 *   sessionId: string
 * }
 */
router.post('/auth/login', async (req, res) => {
  try {
    const { username, password } = req.body;
    
    if (!username || !password) {
      return res.status(400).json({ 
        success: false, 
        error: 'Username and password are required' 
      });
    }
    
    const user = await userModel.findByUsername(username);
    if (!user) {
      return res.status(401).json({ 
        success: false, 
        error: 'Invalid credentials' 
      });
    }
    
    if (!userModel.verifyPassword(user, password)) {
      return res.status(401).json({ 
        success: false, 
        error: 'Invalid credentials' 
      });
    }
    
    // Check if account is active
    if (user.active === false) {
      return res.status(403).json({ 
        success: false, 
        error: 'Account deactivated - Please contact administrator' 
      });
    }
    
    // Set session
    req.session.user = { id: user.id, email: user.email, name: user.name };
    req.session.role = user.role;
    
    res.json({
      success: true,
      user: {
        id: user.id,
        email: user.email,
        name: user.name,
        firstName: user.firstName,
        lastName: user.lastName,
        role: user.role,
        avatar: user.profile?.avatar || null
      },
      sessionId: req.sessionID
    });
  } catch (error) {
    console.error('API login error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * POST /api/auth/logout
 * Logout and destroy session
 * 
 * Expected Response:
 * {
 *   success: true,
 *   message: 'Logged out successfully'
 * }
 */
router.post('/auth/logout', (req, res) => {
  req.session.destroy((err) => {
    if (err) {
      return res.status(500).json({ success: false, error: 'Failed to logout' });
    }
    res.json({ success: true, message: 'Logged out successfully' });
  });
});

/**
 * GET /api/auth/check
 * Check if user is authenticated
 * 
 * Expected Response:
 * {
 *   authenticated: boolean,
 *   user: { id, email, name },
 *   role: string
 * }
 */
router.get('/auth/check', (req, res) => {
  if (req.session && req.session.user) {
    res.json({
      authenticated: true,
      user: req.session.user,
      role: req.session.role
    });
  } else {
    res.json({ authenticated: false });
  }
});

// ============================================================================
// AUTH MIDDLEWARE - All routes below require authentication
// ============================================================================

const requireAuth = (req, res, next) => {
  if (!req.session || !req.session.user) {
    return res.status(401).json({ 
      success: false, 
      error: 'Authentication required' 
    });
  }
  next();
};

const requireStudent = (req, res, next) => {
  if (!req.session || !['student', 'admin'].includes(req.session.role)) {
    return res.status(403).json({ 
      success: false, 
      error: 'Student access required' 
    });
  }
  next();
};

// Apply auth middleware to all routes below
router.use(requireAuth);

// ============================================================================
// DASHBOARD / HOME
// ============================================================================

/**
 * GET /api/dashboard
 * Get dashboard data for the current user
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     user: { id, name, firstName, lastName, email, avatar },
 *     stats: { enrolledClasses, overallProgress, upcomingItems, unreadMessages },
 *     upcomingItems: [{ type, title, dueDate, classId, className, id }],
 *     announcements: [{ id, message, createdAt }],
 *     enrolledClasses: [{ id, name, shortName, teacherId }]
 *   }
 * }
 */
router.get('/dashboard', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const user = await userModel.findById(userId);
    
    if (!user) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }
    
    // Get enrolled classes
    const allClasses = await classModel.getAllClasses();
    const enrolledClasses = allClasses.filter(c => 
      (c.studentIds || []).includes(userId)
    );
    
    // Get upcoming assignments and tests
    const upcomingItems = [];
    const now = new Date();
    
    enrolledClasses.forEach(klass => {
      // Add upcoming tests
      (klass.tests || []).forEach(test => {
        if (test.dueDate && new Date(test.dueDate) > now) {
          upcomingItems.push({
            type: 'test',
            title: test.title,
            dueDate: test.dueDate,
            classId: klass.id,
            className: klass.name,
            id: test.id
          });
        }
      });
      
      // Add upcoming assignments
      (klass.assignments || []).forEach(assignment => {
        if (assignment.dueDate && new Date(assignment.dueDate) > now) {
          upcomingItems.push({
            type: 'assignment',
            title: assignment.title,
            dueDate: assignment.dueDate,
            classId: klass.id,
            className: klass.name,
            id: assignment.id
          });
        }
      });
    });
    
    // Sort upcoming items by due date
    upcomingItems.sort((a, b) => new Date(a.dueDate) - new Date(b.dueDate));
    
    // Get announcements
    const announcements = await announcementModel.forStudent(userId);
    const activeAnnouncements = announcements.slice(0, 5);
    
    // Get unread messages count
    const messages = await messageModel.getMailbox(userId);
    const unreadCount = messages.filter(m => m.recipientId === userId && !m.readAt).length;
    
    // Calculate overall progress
    let totalProgress = 0;
    let classCount = enrolledClasses.length;
    
    enrolledClasses.forEach(klass => {
      const lectures = klass.lectures || [];
      const simulations = klass.simulations || [];
      const tests = klass.tests || [];
      
      let completed = 0;
      let total = lectures.length + simulations.length + tests.length;
      
      // Count completed lectures
      lectures.forEach(l => {
        const completions = l.completions || [];
        if (completions.some(c => c.studentId === userId && c.completed)) completed++;
      });
      
      // Count completed simulations
      simulations.forEach(l => {
        const completions = l.completions || [];
        if (completions.some(c => c.studentId === userId && c.completed)) completed++;
      });
      
      // Count completed tests
      const grades = klass.grades || [];
      tests.forEach(t => {
        if (grades.some(g => g.testId === t.id && g.studentId === userId)) completed++;
      });
      
      if (total > 0) {
        totalProgress += (completed / total) * 100;
      }
    });
    
    const overallProgress = classCount > 0 ? Math.round(totalProgress / classCount) : 0;
    
    res.json({
      success: true,
      data: {
        user: {
          id: user.id,
          name: user.name,
          firstName: user.firstName,
          lastName: user.lastName,
          email: user.email,
          avatar: user.profile?.avatar || null
        },
        stats: {
          enrolledClasses: enrolledClasses.length,
          overallProgress: overallProgress,
          upcomingItems: upcomingItems.length,
          unreadMessages: unreadCount
        },
        upcomingItems: upcomingItems.slice(0, 10),
        announcements: activeAnnouncements.map(a => ({
          id: a.id,
          title: a.title,
          message: a.message,
          createdAt: a.createdAt
        })),
        enrolledClasses: enrolledClasses.map(c => ({
          id: c.id,
          name: c.name,
          shortName: c.shortName,
          teacherId: c.teacherId
        }))
      }
    });
  } catch (error) {
    console.error('Dashboard API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// USER PROFILE
// ============================================================================

/**
 * GET /api/profile
 * Get current user's full profile
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: { id, name, username, email, firstName, lastName, phone, address, city, state, zip, role, profile }
 * }
 */
router.get('/profile', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const user = await userModel.findById(userId);
    
    if (!user) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }
    
    // Remove sensitive data
    const { password, ...safeUser } = user;
    
    res.json({
      success: true,
      data: {
        ...safeUser,
        profile: user.profile || {}
      }
    });
  } catch (error) {
    console.error('Profile API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * PUT /api/profile
 * Update current user's profile
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: { id, name, firstName, lastName, email, phone }
 * }
 */
router.put('/profile', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const updates = req.body;
    
    // Fields that can be updated
    const allowedFields = [
      'firstName', 'lastName', 'phone', 'address', 'city', 
      'state', 'zip', 'dateOfBirth', 'emergencyContact', 
      'emergencyPhone'
    ];
    
    const safeUpdates = {};
    allowedFields.forEach(field => {
      if (updates[field] !== undefined) {
        safeUpdates[field] = updates[field];
      }
    });
    
    await userModel.update(userId, safeUpdates);
    const updatedUser = await userModel.findById(userId);
    
    res.json({
      success: true,
      data: {
        id: updatedUser.id,
        name: updatedUser.name,
        firstName: updatedUser.firstName,
        lastName: updatedUser.lastName,
        email: updatedUser.email,
        phone: updatedUser.phone
      }
    });
  } catch (error) {
    console.error('Profile update API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// CLASSES
// ============================================================================

/**
 * GET /api/classes
 * Get all enrolled classes for the current user
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: [{
 *     id, name, shortName, teacherId, progress,
 *     stats: { lectures: {completed, total}, simulations: {completed, total}, tests: {completed, total, average}, assignments: {completed, total, average} }
 *   }]
 * }
 */
router.get('/classes', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const allClasses = await classModel.getAllClasses();
    
    const enrolledClasses = allClasses.filter(c => 
      (c.studentIds || []).includes(userId)
    );
    
    const classesWithProgress = enrolledClasses.map(klass => {
      const lectures = klass.lectures || [];
      const simulations = klass.simulations || [];
      const tests = klass.tests || [];
      const assignments = klass.assignments || [];
      const grades = klass.grades || [];
      
      // Calculate lecture progress
      let lecturesCompleted = 0;
      lectures.forEach(l => {
        const completions = l.completions || [];
        if (completions.some(c => c.studentId === userId && c.completed)) lecturesCompleted++;
      });
      
      // Calculate simulation progress
      let simulationsCompleted = 0;
      simulations.forEach(l => {
        const completions = l.completions || [];
        if (completions.some(c => c.studentId === userId && c.completed)) simulationsCompleted++;
      });
      
      // Calculate test scores
      const testGrades = grades.filter(g => g.testId && g.studentId === userId);
      const avgTestScore = testGrades.length > 0 
        ? Math.round(testGrades.reduce((sum, g) => sum + g.score, 0) / testGrades.length)
        : null;
      
      // Calculate assignment scores
      const asgGrades = grades.filter(g => g.assignmentId && g.studentId === userId);
      const avgAsgScore = asgGrades.length > 0
        ? Math.round(asgGrades.reduce((sum, g) => sum + g.score, 0) / asgGrades.length)
        : null;
      
      // Overall progress
      const totalItems = lectures.length + simulations.length + tests.length;
      const completedItems = lecturesCompleted + simulationsCompleted + testGrades.length;
      const progress = totalItems > 0 ? Math.round((completedItems / totalItems) * 100) : 0;
      
      return {
        id: klass.id,
        name: klass.name,
        shortName: klass.shortName,
        teacherId: klass.teacherId,
        progress: progress,
        stats: {
          lectures: { completed: lecturesCompleted, total: lectures.length },
          simulations: { completed: simulationsCompleted, total: simulations.length },
          tests: { completed: testGrades.length, total: tests.length, average: avgTestScore },
          assignments: { completed: asgGrades.length, total: assignments.length, average: avgAsgScore }
        }
      };
    });
    
    res.json({
      success: true,
      data: classesWithProgress
    });
  } catch (error) {
    console.error('Classes API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * GET /api/classes/:id
 * Get detailed class information
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     id, name, shortName, description,
 *     teacher: { id, name, email },
 *     lectures: [{ id, title, url, dripDay, completed, completedAt }],
 *     simulations: [{ id, title, url, dripDay, completed, completedAt }],
 *     tests: [{ id, title, timeLimit, dueDate, official, score, gradedAt, completed }],
 *     assignments: [{ id, title, url, dueDate, dripDay, score, gradedAt, completed }],
 *     startDate, endDate
 *   }
 * }
 */
router.get('/classes/:id', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const classId = parseInt(req.params.id);
    
    const klass = await classModel.findClassById(classId);
    
    if (!klass) {
      return res.status(404).json({ success: false, error: 'Class not found' });
    }
    
    // Check enrollment
    if (!(klass.studentIds || []).includes(userId) && req.session.role !== 'admin') {
      return res.status(403).json({ success: false, error: 'Not enrolled in this class' });
    }
    
    // Get teacher info
    const teacher = klass.teacherId ? await userModel.findById(klass.teacherId) : null;
    
    // Process lectures with completion status
    const lectures = (klass.lectures || []).map(lecture => {
      const completions = lecture.completions || [];
      const isCompleted = completions.some(c => c.studentId === userId && c.completed);
      return {
        id: lecture.id,
        title: lecture.title,
        url: lecture.url,
        dripDay: lecture.dripDay,
        completed: isCompleted,
        completedAt: completions.find(c => c.studentId === userId)?.completedAt || null
      };
    });
    
    // Process simulations with completion status
    const simulations = (klass.simulations || []).map(sim => {
      const completions = sim.completions || [];
      const isCompleted = completions.some(c => c.studentId === userId && c.completed);
      return {
        id: sim.id,
        title: sim.title,
        url: sim.url,
        dripDay: sim.dripDay,
        completed: isCompleted,
        completedAt: completions.find(c => c.studentId === userId)?.completedAt || null
      };
    });
    
    // Process tests with grades
    const grades = klass.grades || [];
    const tests = (klass.tests || []).map(test => {
      const grade = grades.find(g => g.testId === test.id && g.studentId === userId);
      return {
        id: test.id,
        title: test.title,
        timeLimit: test.timeLimit,
        dueDate: test.dueDate,
        official: test.official,
        score: grade?.score || null,
        gradedAt: grade?.gradedAt || null,
        completed: !!grade
      };
    });
    
    // Process assignments with grades
    const assignments = (klass.assignments || []).map(asg => {
      const grade = grades.find(g => g.assignmentId === asg.id && g.studentId === userId);
      return {
        id: asg.id,
        title: asg.title,
        url: asg.url,
        dueDate: asg.dueDate,
        dripDay: asg.dripDay,
        score: grade?.score || null,
        gradedAt: grade?.gradedAt || null,
        completed: !!grade
      };
    });
    
    res.json({
      success: true,
      data: {
        id: klass.id,
        name: klass.name,
        shortName: klass.shortName,
        description: klass.description || '',
        teacher: teacher ? {
          id: teacher.id,
          name: teacher.name,
          email: teacher.email
        } : null,
        lectures,
        simulations,
        tests,
        assignments,
        startDate: klass.startDate,
        endDate: klass.endDate
      }
    });
  } catch (error) {
    console.error('Class detail API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// LECTURES
// ============================================================================

/**
 * POST /api/classes/:classId/lectures/:lectureId/complete
 * Mark a lecture as complete
 * 
 * Expected Response:
 * {
 *   success: true,
 *   message: 'Lecture marked as complete'
 * }
 */
router.post('/classes/:classId/lectures/:lectureId/complete', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const classId = parseInt(req.params.classId);
    const lectureId = parseInt(req.params.lectureId);
    
    await classModel.markLectureCompleted(classId, lectureId, userId);
    
    res.json({
      success: true,
      message: 'Lecture marked as complete'
    });
  } catch (error) {
    console.error('Mark lecture complete API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// SIMULATIONS
// ============================================================================

/**
 * POST /api/classes/:classId/simulations/:simulationId/complete
 * Mark a simulation as complete
 * 
 * Expected Response:
 * {
 *   success: true,
 *   message: 'Simulation marked as complete'
 * }
 */
router.post('/classes/:classId/simulations/:simulationId/complete', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const classId = parseInt(req.params.classId);
    const simulationId = parseInt(req.params.simulationId);
    
    await classModel.markSimulationCompleted(classId, simulationId, userId);
    
    res.json({
      success: true,
      message: 'Simulation marked as complete'
    });
  } catch (error) {
    console.error('Mark simulation complete API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// TESTS
// ============================================================================

/**
 * GET /api/classes/:classId/tests/:testId
 * Get test questions for taking a test
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     test: { id, title, timeLimit, questionCount },
 *     questions: [{ index, question, options: [], picture, explanation: null }],
 *     progress: { currentQuestion, answers: {}, startedAt } | null
 *   }
 * }
 */
router.get('/classes/:classId/tests/:testId', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const classId = parseInt(req.params.classId);
    const testId = parseInt(req.params.testId);
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      return res.status(404).json({ success: false, error: 'Class not found' });
    }
    
    // Check enrollment
    if (!(klass.studentIds || []).includes(userId) && req.session.role !== 'admin') {
      return res.status(403).json({ success: false, error: 'Not enrolled in this class' });
    }
    
    const test = (klass.tests || []).find(t => t.id === testId);
    if (!test) {
      return res.status(404).json({ success: false, error: 'Test not found' });
    }
    
    // Get questions from test library
    const questions = await testModel.getQuestionsByTest(test.title);
    
    // Remove correct answers from response
    const questionsForStudent = questions.map((q, idx) => ({
      index: idx,
      question: q.question,
      options: q.options,
      picture: q.picture || null,
      explanation: null // Don't send explanation until after submission
    }));
    
    // Check for existing progress
    let progress = null;
    try {
      progress = await testProgressModel.getProgress(userId, classId, test.title);
    } catch (e) {
      // Progress tracking might not exist
    }
    
    res.json({
      success: true,
      data: {
        test: {
          id: test.id,
          title: test.title,
          timeLimit: test.timeLimit || 90,
          questionCount: questions.length
        },
        questions: questionsForStudent,
        progress: progress ? {
          currentQuestion: progress.current_question,
          answers: progress.answers || {},
          startedAt: progress.started_at
        } : null
      }
    });
  } catch (error) {
    console.error('Get test API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * POST /api/classes/:classId/tests/:testId/submit
 * Submit test answers
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     score: 85,
 *     correct: 17,
 *     total: 20,
 *     passed: true,
 *     results: [{ questionIndex, question, studentAnswer, correctAnswer, isCorrect, explanation }]
 *   }
 * }
 */
router.post('/classes/:classId/tests/:testId/submit', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const classId = parseInt(req.params.classId);
    const testId = parseInt(req.params.testId);
    const { answers } = req.body; // { 0: 1, 1: 2, 2: 0, ... } (questionIndex: answerIndex)
    
    console.log('Submit test - userId:', userId, 'classId:', classId, 'testId:', testId);
    console.log('Answers received:', answers);
    
    if (!answers || typeof answers !== 'object') {
      return res.status(400).json({ success: false, error: 'Invalid answers format' });
    }
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      console.log('Class not found:', classId);
      return res.status(404).json({ success: false, error: 'Class not found' });
    }
    
    const test = (klass.tests || []).find(t => t.id === testId);
    if (!test) {
      console.log('Test not found:', testId, 'in class:', classId);
      return res.status(404).json({ success: false, error: 'Test not found' });
    }
    
    console.log('Found test:', test.title);
    
    // Get questions to grade
    const questions = await testModel.getQuestionsByTest(test.title);
    
    if (!questions || questions.length === 0) {
      console.log('No questions found for test:', test.title);
      return res.status(404).json({ success: false, error: 'No questions found for this test' });
    }
    
    console.log('Grading test with', questions.length, 'questions');
    
    // Calculate score
    let correct = 0;
    const results = questions.map((q, idx) => {
      const studentAnswer = answers[idx];
      const isCorrect = studentAnswer === q.answer;
      if (isCorrect) correct++;
      
      return {
        questionIndex: idx,
        question: q.question,
        studentAnswer: studentAnswer,
        correctAnswer: q.answer,
        isCorrect: isCorrect,
        explanation: q.explanation || null
      };
    });
    
    const score = Math.round((correct / questions.length) * 100);
    
    console.log('Test score:', score, '(', correct, '/', questions.length, ')');
    
    // Save grade
    await classModel.recordGrade(classId, testId, userId, score);
    
    console.log('Grade saved successfully');
    
    // Clear test progress
    try {
      await testProgressModel.deleteProgress(userId, classId, test.title);
    } catch (e) {
      console.log('Could not clear progress (may not exist):', e.message);
      // Progress tracking might not exist
    }
    
    res.json({
      success: true,
      data: {
        score: score,
        correct: correct,
        total: questions.length,
        passed: score >= 70,
        results: results
      }
    });
  } catch (error) {
    console.error('Submit test API error:', error);
    console.error('Error stack:', error.stack);
    res.status(500).json({ success: false, error: 'Server error: ' + error.message });
  }
});

/**
 * POST /api/classes/:classId/tests/:testId/progress
 * Save test progress (for resuming later)
 * 
 * Expected Response:
 * {
 *   success: true,
 *   message: 'Progress saved',
 *   data: { currentQuestion, answers: {}, savedAt }
 * }
 */
router.post('/classes/:classId/tests/:testId/progress', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const classId = parseInt(req.params.classId);
    const testId = parseInt(req.params.testId);
    const { currentQuestion, answers } = req.body;
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      return res.status(404).json({ success: false, error: 'Class not found' });
    }
    
    const test = (klass.tests || []).find(t => t.id === testId);
    if (!test) {
      return res.status(404).json({ success: false, error: 'Test not found' });
    }
    
    await testProgressModel.saveProgress(userId, classId, test.title, answers, currentQuestion);
    
    res.json({
      success: true,
      message: 'Progress saved'
    });
  } catch (error) {
    console.error('Save test progress API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * GET /api/classes/:classId/tests/:testId/resume
 * Check if test can be resumed and get saved progress
 * 
 * Expected Response:
 * {
 *   success: true,
 *   canResume: true,
 *   progress: { currentQuestion, answers: {}, startedAt, testTitle } | null,
 *   completed: false
 * }
 * OR
 * {
 *   success: true,
 *   canResume: false,
 *   completed: true,
 *   message: 'Test already completed',
 *   score: 85
 * }
 */
router.get('/classes/:classId/tests/:testId/resume', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const classId = parseInt(req.params.classId);
    const testId = parseInt(req.params.testId);
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      return res.status(404).json({ success: false, error: 'Class not found' });
    }
    
    // Check enrollment
    if (!(klass.studentIds || []).includes(userId) && req.session.role !== 'admin') {
      return res.status(403).json({ success: false, error: 'Not enrolled in this class' });
    }
    
    const test = (klass.tests || []).find(t => t.id === testId);
    if (!test) {
      return res.status(404).json({ success: false, error: 'Test not found' });
    }
    
    // Check if test is already completed
    const grades = klass.grades || [];
    const existingGrade = grades.find(g => g.testId === testId && g.studentId === userId);
    
    if (existingGrade) {
      return res.json({
        success: true,
        canResume: false,
        completed: true,
        message: 'Test already completed',
        score: existingGrade.score
      });
    }
    
    // Check for saved progress
    let progress = null;
    try {
      progress = await testProgressModel.getProgress(userId, classId, test.title);
    } catch (e) {
      // Progress tracking might not exist
    }
    
    if (progress) {
      const answers = JSON.parse(progress.answers || '{}');
      const answeredCount = Object.keys(answers).length;
      
      return res.json({
        success: true,
        canResume: true,
        completed: false,
        progress: {
          currentQuestion: progress.current_question,
          answeredCount: answeredCount,
          totalQuestions: await testModel.getQuestionsByTest(test.title).then(q => q.length),
          startedAt: progress.started_at,
          savedAt: progress.updated_at || progress.started_at
        }
      });
    } else {
      return res.json({
        success: true,
        canResume: false,
        completed: false,
        message: 'No saved progress found. Start a new test.'
      });
    }
  } catch (error) {
    console.error('Check resume API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * DELETE /api/classes/:classId/tests/:testId/progress
 * Clear saved test progress (restart test)
 * 
 * Expected Response:
 * {
 *   success: true,
 *   message: 'Progress cleared'
 * }
 */
router.delete('/classes/:classId/tests/:testId/progress', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const classId = parseInt(req.params.classId);
    const testId = parseInt(req.params.testId);
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      return res.status(404).json({ success: false, error: 'Class not found' });
    }
    
    const test = (klass.tests || []).find(t => t.id === testId);
    if (!test) {
      return res.status(404).json({ success: false, error: 'Test not found' });
    }
    
    try {
      await testProgressModel.deleteProgress(userId, classId, test.title);
    } catch (e) {
      // Progress might not exist
    }
    
    res.json({
      success: true,
      message: 'Test progress cleared. You can start fresh.'
    });
  } catch (error) {
    console.error('Clear progress API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * GET /api/tests/in-progress
 * Get all tests currently in progress for the student
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: [{
 *     classId, className,
 *     testId, testTitle, timeLimit,
 *     currentQuestion, answeredCount,
 *     startedAt, savedAt
 *   }]
 * }
 */
router.get('/tests/in-progress', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const allClasses = await classModel.getAllClasses();
    const enrolledClasses = allClasses.filter(c => 
      (c.studentIds || []).includes(userId)
    );
    
    const testsInProgress = [];
    
    for (const klass of enrolledClasses) {
      const tests = klass.tests || [];
      const grades = klass.grades || [];
      
      for (const test of tests) {
        // Check if test is completed
        const isCompleted = grades.some(g => g.testId === test.id && g.studentId === userId);
        
        if (!isCompleted) {
          // Check for progress
          try {
            const progress = await testProgressModel.getProgress(userId, klass.id, test.title);
            if (progress) {
              const answers = JSON.parse(progress.answers || '{}');
              testsInProgress.push({
                classId: klass.id,
                className: klass.name,
                testId: test.id,
                testTitle: test.title,
                timeLimit: test.timeLimit || 90,
                currentQuestion: progress.current_question,
                answeredCount: Object.keys(answers).length,
                startedAt: progress.started_at,
                savedAt: progress.updated_at || progress.started_at
              });
            }
          } catch (e) {
            // No progress for this test
          }
        }
      }
    }
    
    res.json({
      success: true,
      data: testsInProgress
    });
  } catch (error) {
    console.error('Get in-progress tests API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * GET /api/classes/:classId/tests/:testId/results
 * Get test results after completion
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     testTitle, score, passed, letterGrade,
 *     totalQuestions, correctAnswers,
 *     completedAt, canRetake
 *   }
 * }
 */
router.get('/classes/:classId/tests/:testId/results', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const classId = parseInt(req.params.classId);
    const testId = parseInt(req.params.testId);
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      return res.status(404).json({ success: false, error: 'Class not found' });
    }
    
    const test = (klass.tests || []).find(t => t.id === testId);
    if (!test) {
      return res.status(404).json({ success: false, error: 'Test not found' });
    }
    
    // Get grade for this test
    const grades = klass.grades || [];
    const grade = grades.find(g => g.testId === testId && g.studentId === userId);
    
    if (!grade) {
      return res.status(404).json({ success: false, error: 'No results found. Test may not be completed yet.' });
    }
    
    // Get questions to show results
    const questions = await testModel.getQuestionsByTest(test.title);
    
    // Calculate details
    const score = grade.score;
    const passed = score >= 70;
    const letterGrade = score >= 90 ? 'A' : score >= 80 ? 'B' : score >= 70 ? 'C' : score >= 60 ? 'D' : 'F';
    
    res.json({
      success: true,
      data: {
        testTitle: test.title,
        score: score,
        passed: passed,
        letterGrade: letterGrade,
        totalQuestions: questions.length,
        correctAnswers: Math.round((score / 100) * questions.length),
        completedAt: grade.gradedAt,
        canRetake: true // You can add logic here if there's a limit
      }
    });
  } catch (error) {
    console.error('Get test results API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * GET /api/study/tests
 * Get all available tests for study mode
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: [{ title, questionCount, timeLimit }]
 * }
 */
router.get('/study/tests', requireStudent, async (req, res) => {
  try {
    const tests = await testModel.getAllTests();
    
    res.json({
      success: true,
      data: tests.map(t => ({
        title: t.title,
        questionCount: t.questionCount,
        timeLimit: t.timeLimit || 90
      }))
    });
  } catch (error) {
    console.error('Get study tests API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * GET /api/study/tests/:testTitle
 * Get test questions for study mode with correct answers
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     testTitle,
 *     questionCount,
 *     questions: [{ index, question, options: [], correctAnswer, explanation, picture }]
 *   }
 * }
 */
router.get('/study/tests/:testTitle', requireStudent, async (req, res) => {
  try {
    const testTitle = decodeURIComponent(req.params.testTitle);
    const questions = await testModel.getQuestionsByTest(testTitle);
    
    if (!questions || questions.length === 0) {
      return res.status(404).json({ success: false, error: 'Test not found' });
    }
    
    // In study mode, we can show answers and explanations
    const questionsWithAnswers = questions.map((q, idx) => ({
      index: idx,
      question: q.question,
      options: q.options || [],
      correctAnswer: q.answer,
      explanation: q.explanation || null,
      picture: q.picture || null
    }));
    
    res.json({
      success: true,
      data: {
        testTitle: testTitle,
        questionCount: questions.length,
        questions: questionsWithAnswers
      }
    });
  } catch (error) {
    console.error('Get study test API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * POST /api/study/tests/:testTitle/practice
 * Submit practice test for immediate feedback (not graded)
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     score, correct, total, passed,
 *     results: [{ questionIndex, question, studentAnswer, correctAnswer, isCorrect, explanation }]
 *   }
 * }
 */
router.post('/study/tests/:testTitle/practice', requireStudent, async (req, res) => {
  try {
    const testTitle = decodeURIComponent(req.params.testTitle);
    const { answers } = req.body;
    
    const questions = await testModel.getQuestionsByTest(testTitle);
    
    if (!questions || questions.length === 0) {
      return res.status(404).json({ success: false, error: 'Test not found' });
    }
    
    // Calculate score
    let correct = 0;
    const results = questions.map((q, idx) => {
      const studentAnswer = answers[idx];
      const isCorrect = studentAnswer === q.answer;
      if (isCorrect) correct++;
      
      return {
        questionIndex: idx,
        question: q.question,
        studentAnswer: studentAnswer,
        correctAnswer: q.answer,
        isCorrect: isCorrect,
        explanation: q.explanation || null
      };
    });
    
    const score = Math.round((correct / questions.length) * 100);
    
    res.json({
      success: true,
      data: {
        score: score,
        correct: correct,
        total: questions.length,
        passed: score >= 70,
        results: results,
        isPractice: true // Not saved to grades
      }
    });
  } catch (error) {
    console.error('Practice test API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// SIMULATIONS
// ============================================================================

/**
 * GET /api/classes/:classId/simulations
 * Get all simulations for a class
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: [{
 *     id, title, url, dripDay,
 *     completed, completedAt
 *   }]
 * }
 */
router.get('/classes/:classId/simulations', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const classId = parseInt(req.params.classId);
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      return res.status(404).json({ success: false, error: 'Class not found' });
    }
    
    // Check enrollment
    if (!(klass.studentIds || []).includes(userId) && req.session.role !== 'admin') {
      return res.status(403).json({ success: false, error: 'Not enrolled in this class' });
    }
    
    const simulations = klass.simulations || [];
    
    res.json({
      success: true,
      data: simulations.map(sim => {
        const completions = sim.completions || [];
        const completion = completions.find(c => c.studentId === userId);
        return {
          id: sim.id,
          title: sim.title,
          url: sim.url,
          dripDay: sim.dripDay,
          completed: completion ? completion.completed : false,
          completedAt: completion ? completion.completedAt : null
        };
      })
    });
  } catch (error) {
    console.error('Get simulations API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// CLASS LINKS
// ============================================================================

/**
 * GET /api/classes/:classId/links
 * Get class links (external resources)
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: [{ id, title, url, description }]
 * }
 */
router.get('/classes/:classId/links', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const classId = parseInt(req.params.classId);
    
    const klass = await classModel.findClassById(classId);
    if (!klass) {
      return res.status(404).json({ success: false, error: 'Class not found' });
    }
    
    // Check enrollment
    if (!(klass.studentIds || []).includes(userId) && req.session.role !== 'admin') {
      return res.status(403).json({ success: false, error: 'Not enrolled in this class' });
    }
    
    const links = klass.links || [];
    
    res.json({
      success: true,
      data: links.map(link => ({
        id: link.id || link.title,
        title: link.title,
        url: link.url,
        description: link.description || null
      }))
    });
  } catch (error) {
    console.error('Get class links API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// GRADES
// ============================================================================

/**
 * GET /api/grades
 * Get all grades for the current user across all classes
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     classes: [{ classId, className, grades: [{ testId, testTitle, score, gradedAt }], average }],
 *     overallGPA: 3.2
 *   }
 * }
 */
router.get('/grades', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const allClasses = await classModel.getAllClasses();
    
    const enrolledClasses = allClasses.filter(c => 
      (c.studentIds || []).includes(userId)
    );
    
    const gradesData = enrolledClasses.map(klass => {
      const grades = klass.grades || [];
      const tests = klass.tests || [];
      const assignments = klass.assignments || [];
      
      // Test grades
      const testGrades = tests.map(test => {
        const grade = grades.find(g => g.testId === test.id && g.studentId === userId);
        return {
          id: test.id,
          title: test.title,
          score: grade?.score || null,
          gradedAt: grade?.gradedAt || null
        };
      });
      
      // Assignment grades
      const assignmentGrades = assignments.map(asg => {
        const grade = grades.find(g => g.assignmentId === asg.id && g.studentId === userId);
        return {
          id: asg.id,
          title: asg.title,
          score: grade?.score || null,
          gradedAt: grade?.gradedAt || null
        };
      });
      
      // Calculate averages
      const validTestScores = testGrades.filter(g => g.score !== null).map(g => g.score);
      const validAsgScores = assignmentGrades.filter(g => g.score !== null).map(g => g.score);
      
      const testAvg = validTestScores.length > 0 
        ? Math.round(validTestScores.reduce((a, b) => a + b, 0) / validTestScores.length)
        : null;
      const asgAvg = validAsgScores.length > 0
        ? Math.round(validAsgScores.reduce((a, b) => a + b, 0) / validAsgScores.length)
        : null;
      
      // Calculate overall grade (50% tests, 30% assignments, 20% simulations)
      const simulations = klass.simulations || [];
      let simulationCompletion = 0;
      if (simulations.length > 0) {
        const completedSimulations = simulations.filter(sim => {
          const completions = sim.completions || [];
          return completions.some(c => c.studentId === userId && c.completed);
        }).length;
        simulationCompletion = Math.round((completedSimulations / simulations.length) * 100);
      }
      
      let overallGrade = null;
      if (testAvg !== null || asgAvg !== null) {
        const testComponent = (testAvg || 0) * 0.5;
        const asgComponent = (asgAvg || 0) * 0.3;
        const simulationComponent = simulationCompletion * 0.2;
        overallGrade = Math.round(testComponent + asgComponent + simulationComponent);
      }
      
      // Letter grade
      let letterGrade = null;
      if (overallGrade !== null) {
        if (overallGrade >= 90) letterGrade = 'A';
        else if (overallGrade >= 80) letterGrade = 'B';
        else if (overallGrade >= 70) letterGrade = 'C';
        else if (overallGrade >= 60) letterGrade = 'D';
        else letterGrade = 'F';
      }
      
      return {
        classId: klass.id,
        className: klass.name,
        classShortName: klass.shortName,
        testAverage: testAvg,
        assignmentAverage: asgAvg,
        simulationCompletion: simulationCompletion,
        overallGrade: overallGrade,
        letterGrade: letterGrade,
        tests: testGrades,
        assignments: assignmentGrades
      };
    });
    
    // Calculate GPA
    const gradedClasses = gradesData.filter(g => g.letterGrade !== null);
    let gpa = null;
    if (gradedClasses.length > 0) {
      const gpaMap = { A: 4.0, B: 3.0, C: 2.0, D: 1.0, F: 0.0 };
      const totalPoints = gradedClasses.reduce((sum, g) => sum + gpaMap[g.letterGrade], 0);
      gpa = (totalPoints / gradedClasses.length).toFixed(2);
    }
    
    res.json({
      success: true,
      data: {
        gpa: gpa,
        classes: gradesData
      }
    });
  } catch (error) {
    console.error('Grades API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// MESSAGES
// ============================================================================

/**
 * GET /api/messages
 * Get all messages for the current user (inbox and sent)
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     conversations: [{
 *       otherUser: { id, name },
 *       lastMessage: { subject, body, time, isRead, fromMe },
 *       unreadCount
 *     }]
 *   }
 * }
 */
router.get('/messages', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const messages = await messageModel.getMailbox(userId);
    
    // Group messages by conversation partner
    const conversations = {};
    for (const msg of messages) {
      const otherUserId = msg.senderId === userId ? msg.recipientId : msg.senderId;
      if (!conversations[otherUserId]) {
        // Get other user info
        const otherUser = await userModel.findById(otherUserId);
        conversations[otherUserId] = {
          otherUserId: otherUserId,
          otherUserName: otherUser ? otherUser.name : 'Unknown',
          lastMessage: msg.subject || msg.body,
          lastMessageAt: msg.sentAt,
          unreadCount: 0,
          messages: []
        };
      }
      conversations[otherUserId].messages.push(msg);
      if (msg.recipientId === userId && !msg.readAt) {
        conversations[otherUserId].unreadCount++;
      }
    }
    
    res.json({
      success: true,
      data: Object.values(conversations).map(conv => ({
        otherUserId: conv.otherUserId,
        otherUserName: conv.otherUserName,
        lastMessage: conv.lastMessage,
        lastMessageAt: conv.lastMessageAt,
        unreadCount: conv.unreadCount
      }))
    });
  } catch (error) {
    console.error('Messages API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * GET /api/messages/:otherUserId
 * Get conversation with a specific user
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     otherUser: { id, name, email },
 *     messages: [{ id, senderId, subject, body, sentAt, readAt, isFromMe }]
 *   }
 * }
 */
router.get('/messages/:otherUserId', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const otherUserId = parseInt(req.params.otherUserId);
    
    // Get all messages from mailbox
    const allMessages = await messageModel.getMailbox(userId);
    
    // Filter to messages between these two users
    const messages = allMessages.filter(m => 
      (m.senderId === userId && m.recipientId === otherUserId) ||
      (m.senderId === otherUserId && m.recipientId === userId)
    ).sort((a, b) => new Date(a.sentAt) - new Date(b.sentAt));
    
    // Mark as read (update messages where user is recipient)
    for (const msg of messages) {
      if (msg.recipientId === userId && !msg.readAt) {
        await db.query('UPDATE mdtslms_messages SET readAt = ? WHERE id = ?', [new Date().toISOString(), msg.id]);
      }
    }
    
    // Get other user info
    const otherUser = await userModel.findById(otherUserId);
    
    res.json({
      success: true,
      data: {
        otherUser: otherUser ? {
          id: otherUser.id,
          name: otherUser.name,
          email: otherUser.email
        } : null,
        messages: messages.map(m => ({
          id: m.id,
          senderId: m.senderId,
          subject: m.subject,
          body: m.body,
          sentAt: m.sentAt,
          readAt: m.readAt,
          isFromMe: m.senderId === userId
        }))
      }
    });
  } catch (error) {
    console.error('Conversation API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * POST /api/messages/:otherUserId
 * Send a message to another user
 * 
 * Expected Response:
 * {
 *   success: true,
 *   message: 'Message sent',
 *   data: { messageId, subject, body, sentAt }
 * }
 */
router.post('/messages/:otherUserId', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const otherUserId = parseInt(req.params.otherUserId);
    const { subject, body } = req.body;
    
    if (!body || !body.trim()) {
      return res.status(400).json({ success: false, error: 'Message body is required' });
    }
    
    const message = await messageModel.sendMessage(userId, otherUserId, subject || 'No Subject', body.trim());
    
    res.json({
      success: true,
      data: {
        id: message.id,
        senderId: userId,
        subject: message.subject,
        body: message.body,
        sentAt: message.sentAt
      }
    });
  } catch (error) {
    console.error('Send message API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// DISCUSSIONS
// ============================================================================

/**
 * GET /api/classes/:classId/discussions
 * Get discussions for a class
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: [{ id, userId, userName, message, createdAt }]
 * }
 */
router.get('/classes/:classId/discussions', requireStudent, async (req, res) => {
  try {
    const classId = parseInt(req.params.classId);
    const discussions = await discussionModel.getByClass(classId);
    
    // Get user names for all messages
    const userIds = [...new Set(discussions.map(d => d.userId))];
    const users = {};
    for (const id of userIds) {
      const user = await userModel.findById(id);
      users[id] = user ? user.name : 'Unknown';
    }
    
    res.json({
      success: true,
      data: discussions.map(d => ({
        id: d.id,
        userId: d.userId,
        userName: users[d.userId],
        message: d.message,
        createdAt: d.createdAt
      }))
    });
  } catch (error) {
    console.error('Discussions API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * POST /api/classes/:classId/discussions
 * Post a discussion message
 * 
 * Expected Response:
 * {
 *   success: true,
 *   message: 'Discussion message posted'
 * }
 */
router.post('/classes/:classId/discussions', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const classId = parseInt(req.params.classId);
    const { message } = req.body;
    
    if (!message || !message.trim()) {
      return res.status(400).json({ success: false, error: 'Message is required' });
    }
    
    await discussionModel.addMessage(classId, userId, message.trim());
    
    res.json({
      success: true,
      message: 'Discussion message posted'
    });
  } catch (error) {
    console.error('Post discussion API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// DOCUMENTS
// ============================================================================

/**
 * GET /api/documents
 * Get all documents pending signature for the current user
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: [{
 *     id, title, description, status,
 *     createdAt, partyStatus, signedAt
 *   }]
 * }
 */
router.get('/documents', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const user = await userModel.findById(userId);
    
    if (!user) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }
    
    const documents = await signatureDocumentModel.getDocumentsByEmail(user.email);
    
    res.json({
      success: true,
      data: documents.map(doc => ({
        id: doc.id,
        title: doc.title,
        description: doc.description,
        status: doc.status,
        createdAt: doc.created_at,
        partyStatus: doc.party_status,
        signedAt: doc.signed_at
      }))
    });
  } catch (error) {
    console.error('Documents API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// ATTENDANCE
// ============================================================================

/**
 * GET /api/attendance
 * Get attendance records for the current user
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: [{ date, status, note }]
 * }
 */
router.get('/attendance', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const user = await userModel.findById(userId);
    
    if (!user) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }
    
    const attendance = (user.profile?.attendance || []).sort((a, b) => 
      new Date(b.date) - new Date(a.date)
    );
    
    res.json({
      success: true,
      data: {
        totalDays: attendance.length,
        records: attendance.map(a => ({
          date: a.date,
          className: a.className || 'Class',
          studentName: a.studentName || user.name
        }))
      }
    });
  } catch (error) {
    console.error('Attendance API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// TODO/CHECKLIST
// ============================================================================

/**
 * GET /api/todos
 * Get todo/checklist items for the current user
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: [{
 *     id, templateId, templateName, assignedAt,
 *     items: [{ id, title, completed, completedAt, subitems: [{...}] }]
 *   }]
 * }
 */
router.get('/todos', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const assignments = await todoModel.getStudentAssignments(userId);
    
    res.json({
      success: true,
      data: assignments.map(a => ({
        id: a.id,
        templateId: a.templateId,
        templateName: a.templateName,
        assignedAt: a.assignedAt,
        items: (a.items || []).map(item => ({
          id: item.id,
          title: item.title,
          completed: item.completed,
          completedAt: item.completedAt,
          subitems: (item.subitems || []).map(sub => ({
            id: sub.id,
            title: sub.title,
            completed: sub.completed,
            completedAt: sub.completedAt
          }))
        }))
      }))
    });
  } catch (error) {
    console.error('Todos API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * POST /api/todos/:assignmentId/items/:itemId/toggle
 * Toggle a todo item completion status
 * 
 * Expected Response:
 * {
 *   success: true,
 *   message: 'Item updated',
 *   data: { completed, completedAt }
 * }
 */
router.post('/todos/:assignmentId/items/:itemId/toggle', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const assignmentId = parseInt(req.params.assignmentId);
    const itemId = parseInt(req.params.itemId);
    const { completed, subitemId } = req.body;
    
    if (subitemId) {
      await todoModel.toggleSubitem(assignmentId, itemId, subitemId, completed);
    } else {
      await todoModel.toggleItem(assignmentId, itemId, completed);
    }
    
    res.json({
      success: true,
      message: 'Todo item updated'
    });
  } catch (error) {
    console.error('Toggle todo API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// ANNOUNCEMENTS
// ============================================================================

/**
 * GET /api/announcements
 * Get all active announcements
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: [{ id, message, createdAt, expiresAt }]
 * }
 */
router.get('/announcements', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const announcements = await announcementModel.forStudent(userId);
    
    res.json({
      success: true,
      data: announcements.map(a => ({
        id: a.id,
        message: a.message,
        message: a.message,
        createdAt: a.createdAt,
        expiresAt: a.expiresAt
      }))
    });
  } catch (error) {
    console.error('Announcements API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// TUITION/FINANCIAL
// ============================================================================

/**
 * GET /api/tuition
 * Get tuition breakdown for the current user
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     totalDue, amountPaid, balance,
 *     items: [{ description, amount }]
 *   }
 * }
 */
router.get('/tuition', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const user = await userModel.findById(userId);
    
    if (!user) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }
    
    // Get enrolled classes
    const allClasses = await classModel.getAllClasses();
    const enrolledClasses = allClasses.filter(c => 
      (c.studentIds || []).includes(userId)
    );
    
    // Get courses for pricing
    const courses = await courseModel.getAllCourses();
    const courseLookup = {};
    courses.forEach(c => {
      courseLookup[c.program_name.toLowerCase()] = c;
    });
    
    let totalTuition = 0;
    const tuitionBreakdown = [];
    
    enrolledClasses.forEach(klass => {
      const course = courseLookup[klass.name.toLowerCase()] || 
                     courseLookup[klass.shortName?.toLowerCase()];
      
      if (course) {
        const cost = parseFloat(course.program_cost) || 0;
        totalTuition += cost;
        tuitionBreakdown.push({
          className: klass.name,
          courseName: course.program_name,
          cost: cost,
          hours: course.program_hours || 0
        });
      }
    });
    
    // Add manual tuition courses
    const manualCourses = user.profile?.manualTuitionCourses || [];
    manualCourses.forEach(manual => {
      const cost = parseFloat(manual.cost) || 0;
      totalTuition += cost;
      tuitionBreakdown.push({
        className: manual.courseName,
        courseName: manual.courseName,
        cost: cost,
        hours: manual.clockHours || 0,
        isManual: true
      });
    });
    
    const registrationFee = 75.00;
    
    res.json({
      success: true,
      data: {
        tuitionBreakdown: tuitionBreakdown,
        subtotal: totalTuition,
        registrationFee: registrationFee,
        total: totalTuition + registrationFee
      }
    });
  } catch (error) {
    console.error('Tuition API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// CONTACTS (Teachers, Admins)
// ============================================================================

/**
 * GET /api/contacts
 * Get list of teachers and admins the student can message
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     teachers: [{ id, name, email, role: 'teacher' }],
 *     admins: [{ id, name, email, role: 'admin' }]
 *   }
 * }
 */
router.get('/contacts', requireStudent, async (req, res) => {
  try {
    const userId = req.session.user.id;
    
    // Get enrolled classes to find teachers
    const allClasses = await classModel.getAllClasses();
    const enrolledClasses = allClasses.filter(c => 
      (c.studentIds || []).includes(userId)
    );
    
    const teacherIds = [...new Set(enrolledClasses.map(c => c.teacherId).filter(Boolean))];
    
    // Get teachers
    const teachers = [];
    for (const id of teacherIds) {
      const teacher = await userModel.findById(id);
      if (teacher) {
        teachers.push({
          id: teacher.id,
          name: teacher.name,
          email: teacher.email,
          role: 'teacher'
        });
      }
    }
    
    // Get admins
    const admins = await userModel.findByRole('admin');
    const adminContacts = admins.map(admin => ({
      id: admin.id,
      name: admin.name,
      email: admin.email,
      role: 'admin'
    }));
    
    res.json({
      success: true,
      data: {
        teachers,
        admins: adminContacts
      }
    });
  } catch (error) {
    console.error('Contacts API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// TEACHER DASHBOARD & CALENDAR
// ============================================================================

const requireTeacher = (req, res, next) => {
  if (!req.session || !['teacher', 'admin'].includes(req.session.role)) {
    return res.status(403).json({ 
      success: false, 
      error: 'Teacher access required' 
    });
  }
  next();
};

/**
 * GET /api/teacher/dashboard
 * Get teacher dashboard with classes and schedule
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     user: { id, name, email, role },
 *     stats: { totalClasses, totalStudents, unreadMessages },
 *     classes: [{ id, name, shortName, studentCount, startDate, endDate }],
 *     announcements: [{ id, message, createdAt, expiresAt }]
 *   }
 * }
 */
router.get('/teacher/dashboard', requireTeacher, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const user = await userModel.findById(userId);
    
    if (!user) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }
    
    // Get classes taught by this teacher
    const allClasses = await classModel.getAllClasses();
    const myClasses = allClasses.filter(c => c.teacherId === userId);
    
    // Get announcements
    const announcements = await announcementModel.forTeacher(userId);
    
    // Get messages
    const messages = await messageModel.getMailbox(userId);
    const unreadCount = messages.filter(m => m.recipientId === userId && !m.readAt).length;
    
    // Calculate stats
    let totalStudents = 0;
    myClasses.forEach(c => {
      totalStudents += (c.studentIds || []).length;
    });
    
    res.json({
      success: true,
      data: {
        user: {
          id: user.id,
          name: user.name,
          email: user.email,
          role: user.role
        },
        stats: {
          totalClasses: myClasses.length,
          totalStudents: totalStudents,
          unreadMessages: unreadCount
        },
        classes: myClasses.map(c => ({
          id: c.id,
          name: c.name,
          shortName: c.shortName,
          studentCount: (c.studentIds || []).length,
          startDate: c.startDate,
          endDate: c.endDate
        })),
        recentAnnouncements: announcements.slice(0, 5)
      }
    });
  } catch (error) {
    console.error('Teacher dashboard API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * GET /api/teacher/schedule
 * Get teacher's weekly schedule
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     thisWeek: [{ type, title, dueDate, className, classId }],
 *     nextWeek: [{ type, title, dueDate, className, classId }]
 *   }
 * }
 */
router.get('/teacher/schedule', requireTeacher, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const allClasses = await classModel.getAllClasses();
    const myClasses = allClasses.filter(c => c.teacherId === userId);
    
    // Get current week and next week
    const now = new Date();
    const startOfWeek = new Date(now);
    startOfWeek.setDate(now.getDate() - now.getDay()); // Start on Sunday
    startOfWeek.setHours(0, 0, 0, 0);
    
    const endOfWeek = new Date(startOfWeek);
    endOfWeek.setDate(startOfWeek.getDate() + 6);
    endOfWeek.setHours(23, 59, 59, 999);
    
    const startOfNextWeek = new Date(endOfWeek);
    startOfNextWeek.setDate(endOfWeek.getDate() + 1);
    startOfNextWeek.setHours(0, 0, 0, 0);
    
    const endOfNextWeek = new Date(startOfNextWeek);
    endOfNextWeek.setDate(startOfNextWeek.getDate() + 6);
    endOfNextWeek.setHours(23, 59, 59, 999);
    
    // Collect this week's events
    const thisWeekEvents = [];
    const nextWeekEvents = [];
    
    myClasses.forEach(klass => {
      // Add tests
      (klass.tests || []).forEach(test => {
        if (test.dueDate) {
          const dueDate = new Date(test.dueDate);
          if (dueDate >= startOfWeek && dueDate <= endOfWeek) {
            thisWeekEvents.push({
              type: 'test',
              title: test.title,
              dueDate: test.dueDate,
              className: klass.name,
              classId: klass.id
            });
          } else if (dueDate >= startOfNextWeek && dueDate <= endOfNextWeek) {
            nextWeekEvents.push({
              type: 'test',
              title: test.title,
              dueDate: test.dueDate,
              className: klass.name,
              classId: klass.id
            });
          }
        }
      });
      
      // Add assignments
      (klass.assignments || []).forEach(assignment => {
        if (assignment.dueDate) {
          const dueDate = new Date(assignment.dueDate);
          if (dueDate >= startOfWeek && dueDate <= endOfWeek) {
            thisWeekEvents.push({
              type: 'assignment',
              title: assignment.title,
              dueDate: assignment.dueDate,
              className: klass.name,
              classId: klass.id
            });
          } else if (dueDate >= startOfNextWeek && dueDate <= endOfNextWeek) {
            nextWeekEvents.push({
              type: 'assignment',
              title: assignment.title,
              dueDate: assignment.dueDate,
              className: klass.name,
              classId: klass.id
            });
          }
        }
      });
      
      // Add class schedule if available
      if (klass.schedule) {
        (klass.schedule || []).forEach(session => {
          const sessionDate = new Date(session.date);
          if (sessionDate >= startOfWeek && sessionDate <= endOfWeek) {
            thisWeekEvents.push({
              type: 'class',
              title: `${klass.name} - ${session.topic || 'Class Session'}`,
              date: session.date,
              time: session.time,
              className: klass.name,
              classId: klass.id
            });
          } else if (sessionDate >= startOfNextWeek && sessionDate <= endOfNextWeek) {
            nextWeekEvents.push({
              type: 'class',
              title: `${klass.name} - ${session.topic || 'Class Session'}`,
              date: session.date,
              time: session.time,
              className: klass.name,
              classId: klass.id
            });
          }
        });
      }
    });
    
    // Sort by date
    thisWeekEvents.sort((a, b) => new Date(a.dueDate || a.date) - new Date(b.dueDate || b.date));
    nextWeekEvents.sort((a, b) => new Date(a.dueDate || a.date) - new Date(b.dueDate || b.date));
    
    res.json({
      success: true,
      data: {
        thisWeek: {
          start: startOfWeek.toISOString(),
          end: endOfWeek.toISOString(),
          events: thisWeekEvents
        },
        nextWeek: {
          start: startOfNextWeek.toISOString(),
          end: endOfNextWeek.toISOString(),
          events: nextWeekEvents
        }
      }
    });
  } catch (error) {
    console.error('Teacher schedule API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

// ============================================================================
// ADMIN DASHBOARD & FEATURES
// ============================================================================

const requireAdmin = (req, res, next) => {
  if (!req.session || req.session.role !== 'admin') {
    return res.status(403).json({ 
      success: false, 
      error: 'Admin access required' 
    });
  }
  next();
};

/**
 * GET /api/admin/dashboard
 * Get admin dashboard with system overview
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     user: { id, name, email, role },
 *     stats: { totalStudents, totalTeachers, totalClasses, totalActiveClasses, unreadMessages },
 *     recentActivity: [{ type, description, timestamp }]
 *   }
 * }
 */
router.get('/admin/dashboard', requireAdmin, async (req, res) => {
  try {
    const userId = req.session.user.id;
    const user = await userModel.findById(userId);
    
    if (!user) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }
    
    // Get all classes
    const allClasses = await classModel.getAllClasses();
    
    // Get all students
    const students = await userModel.findByRole('student');
    const activeStudents = students.filter(s => s.active !== false);
    
    // Get all teachers
    const teachers = await userModel.findByRole('teacher');
    
    // Get messages
    const messages = await messageModel.getMailbox(userId);
    const unreadCount = messages.filter(m => m.recipientId === userId && !m.readAt).length;
    
    // Get announcements
    const announcements = await announcementModel.forAdmin();
    
    res.json({
      success: true,
      data: {
        user: {
          id: user.id,
          name: user.name,
          email: user.email,
          role: user.role
        },
        stats: {
          totalClasses: allClasses.length,
          totalStudents: activeStudents.length,
          totalTeachers: teachers.length,
          unreadMessages: unreadCount
        },
        recentAnnouncements: announcements.slice(0, 5)
      }
    });
  } catch (error) {
    console.error('Admin dashboard API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * GET /api/admin/schedule
 * Get admin's view of this week and next week highlights
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: {
 *     thisWeek: { start, end, highlights: [{ type, title, date, className, classId, studentCount }] },
 *     nextWeek: { start, end, highlights: [...] }
 *   }
 * }
 */
router.get('/admin/schedule', requireAdmin, async (req, res) => {
  try {
    const allClasses = await classModel.getAllClasses();
    
    // Get current week and next week
    const now = new Date();
    const startOfWeek = new Date(now);
    startOfWeek.setDate(now.getDate() - now.getDay()); // Start on Sunday
    startOfWeek.setHours(0, 0, 0, 0);
    
    const endOfWeek = new Date(startOfWeek);
    endOfWeek.setDate(startOfWeek.getDate() + 6);
    endOfWeek.setHours(23, 59, 59, 999);
    
    const startOfNextWeek = new Date(endOfWeek);
    startOfNextWeek.setDate(endOfWeek.getDate() + 1);
    startOfNextWeek.setHours(0, 0, 0, 0);
    
    const endOfNextWeek = new Date(startOfNextWeek);
    endOfNextWeek.setDate(startOfNextWeek.getDate() + 6);
    endOfNextWeek.setHours(23, 59, 59, 999);
    
    // Collect this week's highlights
    const thisWeekHighlights = [];
    const nextWeekHighlights = [];
    
    allClasses.forEach(klass => {
      // Class start/end dates
      if (klass.startDate) {
        const startDate = new Date(klass.startDate);
        if (startDate >= startOfWeek && startDate <= endOfWeek) {
          thisWeekHighlights.push({
            type: 'class-start',
            title: `${klass.name} - Class Starts`,
            date: klass.startDate,
            className: klass.name,
            classId: klass.id,
            studentCount: (klass.studentIds || []).length
          });
        } else if (startDate >= startOfNextWeek && startDate <= endOfNextWeek) {
          nextWeekHighlights.push({
            type: 'class-start',
            title: `${klass.name} - Class Starts`,
            date: klass.startDate,
            className: klass.name,
            classId: klass.id,
            studentCount: (klass.studentIds || []).length
          });
        }
      }
      
      if (klass.endDate) {
        const endDate = new Date(klass.endDate);
        if (endDate >= startOfWeek && endDate <= endOfWeek) {
          thisWeekHighlights.push({
            type: 'class-end',
            title: `${klass.name} - Class Ends`,
            date: klass.endDate,
            className: klass.name,
            classId: klass.id,
            studentCount: (klass.studentIds || []).length
          });
        } else if (endDate >= startOfNextWeek && endDate <= endOfNextWeek) {
          nextWeekHighlights.push({
            type: 'class-end',
            title: `${klass.name} - Class Ends`,
            date: klass.endDate,
            className: klass.name,
            classId: klass.id,
            studentCount: (klass.studentIds || []).length
          });
        }
      }
      
      // Tests
      (klass.tests || []).forEach(test => {
        if (test.dueDate) {
          const dueDate = new Date(test.dueDate);
          if (dueDate >= startOfWeek && dueDate <= endOfWeek) {
            thisWeekHighlights.push({
              type: 'test',
              title: `${klass.name} - ${test.title}`,
              dueDate: test.dueDate,
              className: klass.name,
              classId: klass.id
            });
          } else if (dueDate >= startOfNextWeek && dueDate <= endOfNextWeek) {
            nextWeekHighlights.push({
              type: 'test',
              title: `${klass.name} - ${test.title}`,
              dueDate: test.dueDate,
              className: klass.name,
              classId: klass.id
            });
          }
        }
      });
      
      // Assignments
      (klass.assignments || []).forEach(assignment => {
        if (assignment.dueDate) {
          const dueDate = new Date(assignment.dueDate);
          if (dueDate >= startOfWeek && dueDate <= endOfWeek) {
            thisWeekHighlights.push({
              type: 'assignment',
              title: `${klass.name} - ${assignment.title}`,
              dueDate: assignment.dueDate,
              className: klass.name,
              classId: klass.id
            });
          } else if (dueDate >= startOfNextWeek && dueDate <= endOfNextWeek) {
            nextWeekHighlights.push({
              type: 'assignment',
              title: `${klass.name} - ${assignment.title}`,
              dueDate: assignment.dueDate,
              className: klass.name,
              classId: klass.id
            });
          }
        }
      });
    });
    
    // Sort by date
    thisWeekHighlights.sort((a, b) => new Date(a.dueDate || a.date) - new Date(b.dueDate || b.date));
    nextWeekHighlights.sort((a, b) => new Date(a.dueDate || a.date) - new Date(b.dueDate || b.date));
    
    res.json({
      success: true,
      data: {
        thisWeek: {
          start: startOfWeek.toISOString(),
          end: endOfWeek.toISOString(),
          highlights: thisWeekHighlights
        },
        nextWeek: {
          start: startOfNextWeek.toISOString(),
          end: endOfNextWeek.toISOString(),
          highlights: nextWeekHighlights
        }
      }
    });
  } catch (error) {
    console.error('Admin schedule API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * GET /api/admin/students
 * Get all students for admin
 * 
 * Expected Response:
 * {
 *   success: true,
 *   data: [{ id, name, username, email, phone, active, status }]
 * }
 */
router.get('/admin/students', requireAdmin, async (req, res) => {
  try {
    const students = await userModel.findByRole('student');
    
    res.json({
      success: true,
      data: students.map(s => ({
        id: s.id,
        name: s.name,
        username: s.username,
        email: s.email,
        phone: s.phone,
        active: s.active !== false,
        status: s.status
      }))
    });
  } catch (error) {
    console.error('Get students API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

/**
 * POST /api/admin/students/:studentId/contact
 * Contact a student (send message)
 * 
 * Expected Response:
 * {
 *   success: true,
 *   message: 'Message sent to student',
 *   data: { messageId, sentAt }
 * }
 */
router.post('/admin/students/:studentId/contact', requireAdmin, async (req, res) => {
  try {
    const adminId = req.session.user.id;
    const studentId = parseInt(req.params.studentId);
    const { subject, body } = req.body;
    
    if (!body || !body.trim()) {
      return res.status(400).json({ success: false, error: 'Message body is required' });
    }
    
    const student = await userModel.findById(studentId);
    if (!student) {
      return res.status(404).json({ success: false, error: 'Student not found' });
    }
    
    const message = await messageModel.sendMessage(adminId, studentId, subject || 'Message from Admin', body.trim());
    
    res.json({
      success: true,
      message: 'Message sent to student',
      data: {
        id: message.id,
        sentAt: message.sentAt,
        studentName: student.name
      }
    });
  } catch (error) {
    console.error('Contact student API error:', error);
    res.status(500).json({ success: false, error: 'Server error' });
  }
});

module.exports = router;
