const crypto = require('crypto');
const db = require('./db');

function hashPassword(plainPassword, existingSalt) {
  const pwd = String(plainPassword);
  const salt = existingSalt || crypto.randomBytes(16).toString('hex');
  const hash = crypto.pbkdf2Sync(pwd, salt, 1000, 64, 'sha512').toString('hex');
  return { salt, hash };
}

function mapRow(row) {
  if (!row) return row;
  if (row.profile && typeof row.profile === 'string') {
    try { row.profile = JSON.parse(row.profile); } catch (_) { row.profile = null; }
  }
    row.active = row.active === undefined || row.active === null ? true : !!row.active;
  return row;
}


async function findByUsername(username) {
  const [rows] = await db.query('SELECT * FROM mdtslms_users WHERE username = ?', [username]);
  return mapRow(rows[0]);
}

async function findById(id) {
  const [rows] = await db.query('SELECT * FROM mdtslms_users WHERE id = ?', [id]);
  return mapRow(rows[0]);
}

function verifyPassword(user, candidatePassword) {
  if (!user || !user.salt || !user.hash) return false;
  const { hash } = hashPassword(candidatePassword, user.salt);
  return hash === user.hash;
}
async function createTeacher({ name, username, email, password }) {
   const { salt, hash } = hashPassword(password);
  const [result] = await db.query(
     'INSERT INTO mdtslms_users (name, username, email, role, status, salt, hash, active) VALUES (?,?,?,?,?,?,?,?)',
    [name, username, email, 'teacher', 'approved', salt, hash, 1]
  );
  return { id: result.insertId };
}

// Generate a random, unused six-digit student ID (100000–999999)
async function generateStudentId() {
  const [rows] = await db.query('SELECT profile FROM mdtslms_users');
  const used = new Set();
  let max = 99999;
  for (const row of rows) {
    if (!row.profile) continue;
    let parsed;
    try { parsed = JSON.parse(row.profile); } catch (_) { continue; }
    const sid = parsed && parsed.studentId;
    const num = Number(sid);
    if (Number.isInteger(num) && String(num).length === 6) {
      used.add(String(num).padStart(6, '0'));
      if (num > max) max = num;
    }
  }
  // Try random selection a reasonable number of times
  for (let i = 0; i < 2000; i++) {
    const n = Math.floor(100000 + Math.random() * 900000);
    const sid = String(n);
    if (!used.has(sid)) return sid;
  }
  // Fallback to sequential next if random attempts are exhausted
  const next = Math.max(100000, max + 1);
  return String(next).padStart(6, '0');
}

async function createStudent({ username, name, email, password, studentId,
  firstName, lastName, suffix, address, city, state, zip, course, affiliateProgram,
  phones, ssn, emergencyContact, admissionDate, startDate, endDate, classTime, classDays,
  tuition, grievanceAck, financialAid, referralName, referralEmail, gender, race }) {
  const { salt, hash } = hashPassword(password);
  const docNow = new Date().toISOString();
  const unsigned = { agreed: false, signedAt: null, signatureDataUrl: '' };
  const doc = (type, version, sig) => (
    sig
      ? { type, version, agreed: true, signedAt: docNow, signatureDataUrl: sig }
      : { type, version, ...unsigned }
  );

  // Ensure studentId is a unique 6-digit value
  try {
    if (!(studentId && /^\d{6}$/.test(String(studentId)))) {
      studentId = await generateStudentId();
    } else {
      // Check collision
      const [rows] = await db.query('SELECT profile FROM mdtslms_users');
      for (const row of rows) {
        if (!row.profile) continue;
        try {
          const parsed = JSON.parse(row.profile);
          if (parsed && String(parsed.studentId) === String(studentId)) {
            studentId = await generateStudentId();
            break;
          }
        } catch (_) {}
      }
    }
  } catch (_) {
    // Best-effort; fall back to provided value if checks fail
  }

  const profile = {
    studentId,
    firstName, lastName, suffix: suffix || '',
    address: { line1: address, city, state, zip },
    phones: phones || {},
    ssn,
    gender: gender || '',
    race: race || '',
    emergencyContact: emergencyContact || {},
    program: { admissionDate, startDate, endDate, classTime, classDays },
    course,
    affiliateProgram,
    tuition: tuition || {},
    financialAidRequested: !!financialAid,
    grievanceAcknowledged: !!grievanceAck,
    uploads: [],
    documents: [
       doc('registration-agreement', 'v1.0', ''),
      doc('code-of-conduct', 'v1.0', ''),
      doc('cancellation-policy', 'v1.0', ''),
      doc('notice-to-buyer', 'v1.0', ''),
      doc('electronic-use-agreement', 'v1.0', ''),
      doc('contract-acceptance', 'v1.0', ''),
      { type: 'representatives-certification', version: 'v1.0', ...unsigned, requiredRole: 'admin' },
      { type: 'school-official', version: 'v1.0', ...unsigned, requiredRole: 'admin' }
    ]
  };
   if (referralName || referralEmail) {
    profile.referral = { name: referralName, email: referralEmail };
  }

  const [result] = await db.query(
    `INSERT INTO mdtslms_users (username, name, email, role, salt, hash, status, appliedAt, profile, active)
     VALUES (?, ?, ?, 'student', ?, ?, 'pending', ?, ?, ?)`,
    [username, name, email, salt, hash, docNow, JSON.stringify(profile), 1]
  );
   return mapRow({
    id: result.insertId,
    username,
    name,
    email,
    role: 'student',
    status: 'pending',
    profile
  });
}

async function createAdmin({ name, username, email, password }) {
  const { salt, hash } = hashPassword(password);
  const [result] = await db.query(
    'INSERT INTO mdtslms_users (name, username, email, role, status, salt, hash, active) VALUES (?,?,?,?,?,?,?,?)',
    [name, username, email, 'admin', 'approved', salt, hash, 1]
  );
  return { id: result.insertId };
}

async function updateProfile(id, updates) {
  const user = await findById(id);
  if (!user) return null;
  const merge = (target, src) => {
    if (!src) return target;
    for (const key of Object.keys(src)) {
      const val = src[key];
      if (val && typeof val === 'object' && !Array.isArray(val)) {
        target[key] = merge(target[key] || {}, val);
      } else if (val !== undefined) {
        target[key] = val;
      }
    }
    return target;
  };
  user.profile = merge(user.profile || {}, updates);
  await db.query('UPDATE mdtslms_users SET profile=? WHERE id=?', [JSON.stringify(user.profile), id]);
  return user.profile;
}

async function signDocument(id, docType, signatureDataUrl) {
  const user = await findById(id);
  if (!user) return null;
  // Prevent signature updates once the student has been approved
  if (user.status === 'approved') return user;

  user.profile = user.profile || {};
  user.profile.documents = user.profile.documents || [];
  const doc = user.profile.documents.find(d => d.type === docType);
  if (!doc) return user;

  doc.signatureDataUrl = signatureDataUrl;
  doc.signedAt = new Date().toISOString();

  await db.query('UPDATE mdtslms_users SET profile=? WHERE id=?', [JSON.stringify(user.profile), id]);
  return user;
}
async function markApplicationComplete(id) {
  const user = await findById(id);
  if (!user) return null;
  user.profile = user.profile || {};
  user.profile.applicationCompleted = new Date().toISOString();
  await db.query('UPDATE mdtslms_users SET profile=? WHERE id=?', [JSON.stringify(user.profile), id]);
  return user;
}
async function setActive(id, active) {
  await db.query('UPDATE mdtslms_users SET active=? WHERE id=?', [active ? 1 : 0, id]);
  return findById(id);
}
async function findByEmail(email) {
  const [rows] = await db.query('SELECT * FROM mdtslms_users WHERE email = ?', [email]);
  return mapRow(rows[0]);
}


async function updatePassword(username, newPassword) {
  const { salt, hash } = hashPassword(newPassword);
  const [result] = await db.query('UPDATE mdtslms_users SET salt=?, hash=? WHERE username=?', [salt, hash, username]);
  return result.affectedRows > 0;
}

async function setPasswordHash(id, salt, hash) {
  if (!salt || !hash) return false;
  const [result] = await db.query('UPDATE mdtslms_users SET salt=?, hash=? WHERE id=?', [salt, hash, id]);
  return result.affectedRows > 0;
}

async function setResetToken(username, token, expires) {
  const user = await findByUsername(username);
  if (!user) return false;
  user.profile = user.profile || {};
  user.profile.resetToken = { token, expires };
  await db.query('UPDATE mdtslms_users SET profile=? WHERE id=?', [JSON.stringify(user.profile), user.id]);
  return true;
}

async function findByResetToken(token) {
  const [rows] = await db.query('SELECT * FROM mdtslms_users');
  for (const row of rows) {
    const user = mapRow(row);
    if (user.profile && user.profile.resetToken && user.profile.resetToken.token === token) {
      if (user.profile.resetToken.expires && user.profile.resetToken.expires < Date.now()) {
        return null;
      }
      return user;
    }
  }
  return null;
}

async function clearResetToken(id) {
  const user = await findById(id);
  if (!user) return false;
  if (user.profile && user.profile.resetToken) {
    delete user.profile.resetToken;
    await db.query('UPDATE mdtslms_users SET profile=? WHERE id=?', [JSON.stringify(user.profile), id]);
  }
  return true;
}

async function setStep2Token(id, token, expires) {
  const user = await findById(id);
  if (!user) return false;
  user.profile = user.profile || {};
  user.profile.step2Token = { token, expires };
  await db.query('UPDATE mdtslms_users SET profile=? WHERE id=?', [JSON.stringify(user.profile), id]);
  return true;
}

async function findByStep2Token(token) {
  const [rows] = await db.query('SELECT * FROM mdtslms_users');
  for (const row of rows) {
    const user = mapRow(row);
    if (user.profile && user.profile.step2Token && user.profile.step2Token.token === token) {
      if (user.profile.step2Token.expires && user.profile.step2Token.expires < Date.now()) {
        return null;
      }
      return user;
    }
  }
  return null;
}

async function clearStep2Token(id) {
  const user = await findById(id);
  if (!user) return false;
  if (user.profile && user.profile.step2Token) {
    delete user.profile.step2Token;
    await db.query('UPDATE mdtslms_users SET profile=? WHERE id=?', [JSON.stringify(user.profile), id]);
  }
  return true;
}

async function addUploads(id, uploads) {
 const user = await findById(id);
  if (!user) return null;
  user.profile = user.profile || {};
  user.profile.uploads = user.profile.uploads || [];
  user.profile.uploads.push(...uploads);
  await db.query('UPDATE mdtslms_users SET profile=? WHERE id=?', [JSON.stringify(user.profile), id]);
  return user;
}

async function addLinks(id, links) {
  const user = await findById(id);
  if (!user) return null;
  user.profile = user.profile || {};
  user.profile.links = user.profile.links || [];
  user.profile.links.push(...links);
  await db.query('UPDATE mdtslms_users SET profile=? WHERE id=?', [JSON.stringify(user.profile), id]);
  return user;
}

async function removeLink(id, index) {
  const user = await findById(id);
  if (!user) return null;
  const idx = Number(index);
  if (!Number.isInteger(idx) || idx < 0) return user.profile?.links || [];
  user.profile = user.profile || {};
  const links = Array.isArray(user.profile.links) ? user.profile.links : [];
  if (idx >= links.length) return links;
  links.splice(idx, 1);
  user.profile.links = links;
  await db.query('UPDATE mdtslms_users SET profile=? WHERE id=?', [JSON.stringify(user.profile), id]);
  return links;
}

async function updateDocuments(id, documents) {
  const user = await findById(id);
  if (!user) return null;
  user.profile = user.profile || {};
  user.profile.idDocument = documents.idDocument || user.profile.idDocument;
  user.profile.transcriptDocument = documents.transcriptDocument || user.profile.transcriptDocument;
  await db.query('UPDATE mdtslms_users SET profile=? WHERE id=?', [JSON.stringify(user.profile), id]);
  return user;
}

async function addCertificate(id, cert) {
  const user = await findById(id);
  if (!user) return null;
  user.profile = user.profile || {};
  user.profile.certificates = user.profile.certificates || [];
  user.profile.certificates.push({
    title: cert.title || '',
    classId: cert.classId,
    testId: cert.testId,
    date: cert.date || new Date().toISOString(),
    url: cert.url || ''
  });
  await db.query('UPDATE mdtslms_users SET profile=? WHERE id=?', [JSON.stringify(user.profile), id]);
  return user.profile.certificates;
}

let ensuredUserAcceptedAt = false;
async function ensureUserAcceptedAtColumn() {
  if (ensuredUserAcceptedAt) return;
  try {
    await db.query('ALTER TABLE mdtslms_users ADD COLUMN IF NOT EXISTS acceptedAt DATETIME NULL');
  } catch (e) {
    // ignore if not supported; subsequent update will hard-add if needed
  } finally {
    ensuredUserAcceptedAt = true;
  }
}

async function setStatus(id, status) {
  const finishedAt = (status === 'approved' || status === 'declined') ? new Date().toISOString() : null;
  await db.query('UPDATE mdtslms_users SET status=?, finishedAt=? WHERE id=?', [status, finishedAt, id]);
  if (status === 'approved') {
    const ts = new Date().toISOString().slice(0, 19).replace('T', ' ');
    await ensureUserAcceptedAtColumn();
    try {
      await db.query('UPDATE mdtslms_users SET acceptedAt=? WHERE id=?', [ts, id]);
    } catch (e) {
      try { await db.query('ALTER TABLE mdtslms_users ADD COLUMN acceptedAt DATETIME NULL'); } catch (_) {}
      await db.query('UPDATE mdtslms_users SET acceptedAt=? WHERE id=?', [ts, id]);
    }
  }
  return findById(id);
}

async function getAllpre() {
  const [rows] = await db.query('SELECT * FROM mdtslms_pre_registrations');
  return rows.map(mapRow);
}

async function getAll() {
  const [rows] = await db.query('SELECT * FROM mdtslms_users');
  return rows.map(mapRow);
}

async function getAllTestEngine() {
  const [rows] = await db.query('SELECT * FROM mdtsapps_myclass.users');
  return rows; // Return raw rows since this is a different table structure
}

async function setActiveTestEngine(id, active) {
  const status = active ? 'active' : 'inactive';
  
  if (active) {
    // When setting to active, also update StartDate to current date
    const currentDate = new Date().toISOString().slice(0, 19).replace('T', ' ');
    await db.query('UPDATE mdtsapps_myclass.users SET status=?, StartDate=? WHERE id=?', [status, currentDate, id]);
  } else {
    // When deactivating, only update status
    await db.query('UPDATE mdtsapps_myclass.users SET status=? WHERE id=?', [status, id]);
  }
  
  const [rows] = await db.query('SELECT * FROM mdtsapps_myclass.users WHERE id=?', [id]);
  return rows[0];
}

async function getByRole(role) {
  const [rows] = await db.query('SELECT * FROM mdtslms_users WHERE role = ?', [role]);
  return rows.map(mapRow);
}

async function listAdmins() {
  const [rows] = await db.query(
    "SELECT * FROM mdtslms_users WHERE role = 'admin' AND (active IS NULL OR active = 1) ORDER BY name"
  );
  return rows.map(mapRow);
}

async function listTeachers() {
  const [rows] = await db.query(
    "SELECT * FROM mdtslms_users WHERE role = 'teacher' AND (active IS NULL OR active = 1) ORDER BY name"
  );
  return rows.map(mapRow);
}

async function deleteById(id) {
  const [result] = await db.query('DELETE FROM mdtslms_users WHERE id = ?', [id]);
  return result.affectedRows > 0;
}

module.exports = {
  hashPassword,
  verifyPassword,
  findByUsername,
    findByEmail,

  findById,
    generateStudentId,

  createStudent,
  createTeacher,
  createAdmin,
  setStatus,
    setActive,
  addUploads,
  addLinks,
  updateDocuments,
    signDocument,
        markApplicationComplete,

  updateProfile,

  updatePassword,
  setPasswordHash,
    setResetToken,
  findByResetToken,
  clearResetToken,
  setStep2Token,
  findByStep2Token,
  clearStep2Token,
  getAll,
  getAllpre,
  getAllTestEngine,
  setActiveTestEngine,
  getByRole,
  findByRole: getByRole, // Alias for compatibility
  listAdmins,
  listTeachers,
  deleteById
};

// --- Last Contacted helpers ---
let ensuredUserLastContacted = false;
async function ensureUserLastContactedColumn() {
  if (ensuredUserLastContacted) return;
  try {
    await db.query('ALTER TABLE mdtslms_users ADD COLUMN IF NOT EXISTS lastContacted DATETIME NULL');
  } catch (e) {
    // Ignore if not supported or already exists
  } finally {
    ensuredUserLastContacted = true;
  }
}

async function setLastContacted(id, when = new Date()) {
  await ensureUserLastContactedColumn();
  const ts = new Date(when).toISOString().slice(0, 19).replace('T', ' ');
  try {
    await db.query('UPDATE mdtslms_users SET lastContacted=? WHERE id=?', [ts, id]);
  } catch (e) {
    try {
      await db.query('ALTER TABLE mdtslms_users ADD COLUMN lastContacted DATETIME NULL');
    } catch (_) {}
    await db.query('UPDATE mdtslms_users SET lastContacted=? WHERE id=?', [ts, id]);
  }
  return true;
}

async function updateBasicInfo(id, updates) {
  const { name, email, username } = updates;
  const setClause = [];
  const values = [];
  
  if (name !== undefined) {
    setClause.push('name = ?');
    values.push(name);
  }
  if (email !== undefined) {
    setClause.push('email = ?');
    values.push(email);
  }
  if (username !== undefined) {
    setClause.push('username = ?');
    values.push(username);
  }
  
  if (setClause.length === 0) return;
  
  values.push(id);
  const query = `UPDATE mdtslms_users SET ${setClause.join(', ')} WHERE id = ?`;
  await db.query(query, values);
}

module.exports.setLastContacted = setLastContacted;
module.exports.updateBasicInfo = updateBasicInfo;
module.exports.addCertificate = addCertificate;
module.exports.removeLink = removeLink;

// Activate user in mdtsapps_myclass.users table by email (for re-enrollment)
async function activateTestEngineByEmail(email) {
  if (!email) return null;
  
  const currentDate = new Date().toISOString().slice(0, 19).replace('T', ' ');
  
  // Update status to active and StartDate to current date for matching email
  const [result] = await db.query(
    'UPDATE mdtsapps_myclass.users SET status=?, StartDate=? WHERE email=?', 
    ['active', currentDate, email]
  );
  
  if (result.affectedRows > 0) {
    const [rows] = await db.query('SELECT * FROM mdtsapps_myclass.users WHERE email=?', [email]);
    return rows[0];
  }
  
  return null;
}

module.exports.activateTestEngineByEmail = activateTestEngineByEmail;

// --- Account Merge Functions ---
// Find all student accounts with the same email address
async function findAccountsByEmail(email) {
  if (!email) return [];
  const [rows] = await db.query(
    'SELECT * FROM mdtslms_users WHERE email = ? AND role = ? ORDER BY id ASC',
    [email.toLowerCase().trim(), 'student']
  );
  return rows.map(mapRow);
}

// Get all student accounts grouped by email (for displaying merged view)
async function getAccountsGroupedByEmail() {
  const [rows] = await db.query(
    `SELECT * FROM mdtslms_users WHERE role = 'student' ORDER BY email, id ASC`
  );
  
  const grouped = {};
  for (const row of rows) {
    const mapped = mapRow(row);
    const email = (mapped.email || '').toLowerCase().trim();
    if (!email) {
      // Accounts without email get their own group by id
      grouped[`no-email-${mapped.id}`] = [mapped];
    } else {
      if (!grouped[email]) grouped[email] = [];
      grouped[email].push(mapped);
    }
  }
  
  return grouped;
}

// Merge multiple accounts into one primary account
// Keeps the primary account and transfers relevant data from others
async function mergeAccounts(primaryId, secondaryIds, options = {}) {
  const { deleteSecondary = false, adminId = null } = options;
  
  const primary = await findById(primaryId);
  if (!primary) throw new Error('Primary account not found');
  
  const mergeHistory = [];
  const secondaryAccounts = [];
  
  for (const secId of secondaryIds) {
    const secondary = await findById(secId);
    if (!secondary) continue;
    secondaryAccounts.push(secondary);
    
    // Record merge history
    mergeHistory.push({
      mergedFromId: secondary.id,
      mergedFromName: secondary.name,
      mergedFromStatus: secondary.status,
      mergedAt: new Date().toISOString(),
      mergedBy: adminId,
      originalProfile: secondary.profile
    });
  }
  
  // Build merged profile
  const profile = primary.profile || {};
  profile.mergeHistory = profile.mergeHistory || [];
  profile.mergeHistory.push(...mergeHistory);
  
  // Collect status history from all accounts
  profile.statusHistory = profile.statusHistory || [];
  
  // Add primary's current status to history if not already there
  if (primary.status) {
    const lastHistory = profile.statusHistory[profile.statusHistory.length - 1];
    if (!lastHistory || lastHistory.status !== primary.status) {
      profile.statusHistory.push({
        status: primary.status,
        accountId: primary.id,
        timestamp: primary.appliedAt || primary.acceptedAt || new Date().toISOString()
      });
    }
  }
  
  // Add secondary accounts' statuses to history
  for (const secondary of secondaryAccounts) {
    if (secondary.status) {
      profile.statusHistory.push({
        status: secondary.status,
        accountId: secondary.id,
        timestamp: secondary.appliedAt || secondary.acceptedAt || new Date().toISOString()
      });
    }
  }
  
  // Sort status history by timestamp
  profile.statusHistory.sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp));
  
  // Merge uploads from secondary accounts
  const allUploads = [...(profile.uploads || [])];
  for (const secondary of secondaryAccounts) {
    const secUploads = secondary.profile?.uploads || [];
    for (const upload of secUploads) {
      // Avoid duplicates by filename
      if (!allUploads.find(u => u.filename === upload.filename)) {
        allUploads.push({ ...upload, mergedFrom: secondary.id });
      }
    }
  }
  profile.uploads = allUploads;
  
  // Merge documents (keep most recent signed versions)
  const allDocs = [...(profile.documents || [])];
  for (const secondary of secondaryAccounts) {
    const secDocs = secondary.profile?.documents || [];
    for (const secDoc of secDocs) {
      const existingIdx = allDocs.findIndex(d => d.type === secDoc.type);
      if (existingIdx === -1) {
        allDocs.push({ ...secDoc, mergedFrom: secondary.id });
      } else if (secDoc.signedAt && (!allDocs[existingIdx].signedAt || 
                 new Date(secDoc.signedAt) > new Date(allDocs[existingIdx].signedAt))) {
        // Keep newer signed document
        allDocs[existingIdx] = { ...secDoc, mergedFrom: secondary.id };
      }
    }
  }
  profile.documents = allDocs;
  
  // Merge certificates
  const allCerts = [...(profile.certificates || [])];
  for (const secondary of secondaryAccounts) {
    const secCerts = secondary.profile?.certificates || [];
    for (const cert of secCerts) {
      if (!allCerts.find(c => c.name === cert.name && c.issuedAt === cert.issuedAt)) {
        allCerts.push({ ...cert, mergedFrom: secondary.id });
      }
    }
  }
  profile.certificates = allCerts;
  
  // Merge links
  const allLinks = [...(profile.links || [])];
  for (const secondary of secondaryAccounts) {
    const secLinks = secondary.profile?.links || [];
    for (const link of secLinks) {
      if (!allLinks.find(l => l.url === link.url)) {
        allLinks.push({ ...link, mergedFrom: secondary.id });
      }
    }
  }
  profile.links = allLinks;
  
  // Determine best status (prefer approved > alumni > pending > others)
  const statusPriority = { 'approved': 4, 'alumni': 3, 'pending': 2, 'interested': 1 };
  let bestStatus = primary.status;
  let bestPriority = statusPriority[primary.status] || 0;
  
  for (const secondary of secondaryAccounts) {
    const priority = statusPriority[secondary.status] || 0;
    if (priority > bestPriority) {
      bestStatus = secondary.status;
      bestPriority = priority;
    }
  }
  
  // Update primary account with merged data
  await db.query('UPDATE mdtslms_users SET profile=?, status=? WHERE id=?', 
    [JSON.stringify(profile), bestStatus, primaryId]);
  
  // Mark secondary accounts as merged (or delete them)
  for (const secondary of secondaryAccounts) {
    if (deleteSecondary) {
      await db.query('DELETE FROM mdtslms_users WHERE id=?', [secondary.id]);
    } else {
      // Mark as merged - set a merged flag in profile
      const secProfile = secondary.profile || {};
      secProfile.mergedIntoPrimary = primaryId;
      secProfile.mergedAt = new Date().toISOString();
      await db.query('UPDATE mdtslms_users SET profile=?, status=? WHERE id=?',
        [JSON.stringify(secProfile), 'merged', secondary.id]);
    }
  }
  
  return await findById(primaryId);
}

// Auto-merge accounts by email - finds duplicates and merges them
async function autoMergeByEmail(email, adminId = null) {
  const accounts = await findAccountsByEmail(email);
  if (accounts.length <= 1) return null; // Nothing to merge
  
  // Determine primary account (prefer approved > alumni > earliest created)
  const statusPriority = { 'approved': 4, 'alumni': 3, 'pending': 2, 'interested': 1 };
  accounts.sort((a, b) => {
    const priorityDiff = (statusPriority[b.status] || 0) - (statusPriority[a.status] || 0);
    if (priorityDiff !== 0) return priorityDiff;
    // If same priority, use the one with most complete profile
    const aCompleteness = (a.profile?.documents?.filter(d => d.signedAt)?.length || 0);
    const bCompleteness = (b.profile?.documents?.filter(d => d.signedAt)?.length || 0);
    if (bCompleteness !== aCompleteness) return bCompleteness - aCompleteness;
    // Finally, prefer earliest account
    return a.id - b.id;
  });
  
  const primary = accounts[0];
  const secondaryIds = accounts.slice(1).map(a => a.id);
  
  return await mergeAccounts(primary.id, secondaryIds, { deleteSecondary: false, adminId });
}

// Get account history/journey for display
function getAccountJourney(accounts) {
  const journey = [];
  
  // Sort accounts by creation date
  const sorted = [...accounts].sort((a, b) => {
    const dateA = new Date(a.appliedAt || a.createdAt || 0);
    const dateB = new Date(b.appliedAt || b.createdAt || 0);
    return dateA - dateB;
  });
  
  for (const account of sorted) {
    journey.push({
      accountId: account.id,
      status: account.status,
      name: account.name,
      studentId: account.profile?.studentId,
      date: account.appliedAt || account.createdAt,
      acceptedAt: account.acceptedAt,
      becameAlumniAt: account.profile?.becameAlumniAt
    });
  }
  
  return journey;
}

module.exports.findAccountsByEmail = findAccountsByEmail;
module.exports.getAccountsGroupedByEmail = getAccountsGroupedByEmail;
module.exports.mergeAccounts = mergeAccounts;
module.exports.autoMergeByEmail = autoMergeByEmail;
module.exports.getAccountJourney = getAccountJourney;