const db = require('./db');

async function init() {
  await db.query(`
    CREATE TABLE IF NOT EXISTS todo_templates (
      id INT AUTO_INCREMENT PRIMARY KEY,
      name VARCHAR(255) NOT NULL,
      description TEXT,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )
  `);

  await db.query(`
    CREATE TABLE IF NOT EXISTS todo_template_items (
      id INT AUTO_INCREMENT PRIMARY KEY,
      template_id INT NOT NULL,
      title VARCHAR(255) NOT NULL,
      position INT DEFAULT 0,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      CONSTRAINT fk_template_item_template FOREIGN KEY (template_id) REFERENCES todo_templates(id) ON DELETE CASCADE
    )
  `);

  await db.query(`
    CREATE TABLE IF NOT EXISTS todo_template_subitems (
      id INT AUTO_INCREMENT PRIMARY KEY,
      item_id INT NOT NULL,
      title VARCHAR(255) NOT NULL,
      position INT DEFAULT 0,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      CONSTRAINT fk_subitem_item FOREIGN KEY (item_id) REFERENCES todo_template_items(id) ON DELETE CASCADE
    )
  `);

  await db.query(`
    CREATE TABLE IF NOT EXISTS student_todo_assignments (
      id INT AUTO_INCREMENT PRIMARY KEY,
      student_id INT NOT NULL,
      template_id INT NOT NULL,
      assigned_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      UNIQUE KEY uniq_student_template (student_id, template_id),
      CONSTRAINT fk_assignment_template FOREIGN KEY (template_id) REFERENCES todo_templates(id) ON DELETE CASCADE
    )
  `);

  await db.query(`
    CREATE TABLE IF NOT EXISTS student_todo_items (
      id INT AUTO_INCREMENT PRIMARY KEY,
      assignment_id INT NOT NULL,
      parent_item_id INT DEFAULT NULL,
      template_item_id INT DEFAULT NULL,
      template_subitem_id INT DEFAULT NULL,
      title VARCHAR(255) NOT NULL,
      is_subitem TINYINT(1) DEFAULT 0,
      position INT DEFAULT 0,
      completed TINYINT(1) DEFAULT 0,
      completed_at DATETIME DEFAULT NULL,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      CONSTRAINT fk_student_todo_assignment FOREIGN KEY (assignment_id) REFERENCES student_todo_assignments(id) ON DELETE CASCADE,
      CONSTRAINT fk_student_todo_parent FOREIGN KEY (parent_item_id) REFERENCES student_todo_items(id) ON DELETE CASCADE
    )
  `);
}

init().catch(console.error);

async function getAllTemplates() {
  const [templates] = await db.query('SELECT * FROM todo_templates ORDER BY created_at DESC');
  if (!templates.length) return [];

  const templateIds = templates.map(t => t.id);
  const [items] = await db.query(
    `SELECT * FROM todo_template_items WHERE template_id IN (${templateIds.map(() => '?').join(',')})
     ORDER BY position ASC, id ASC`,
    templateIds
  );

  const itemIds = items.map(i => i.id);
  const subItemsQuery = itemIds.length
    ? await db.query(
        `SELECT * FROM todo_template_subitems WHERE item_id IN (${itemIds.map(() => '?').join(',')})
         ORDER BY position ASC, id ASC`,
        itemIds
      )
    : [ [] ];

  const subItems = subItemsQuery[0];
  const subItemsByItem = subItems.reduce((acc, sub) => {
    (acc[sub.item_id] = acc[sub.item_id] || []).push(sub);
    return acc;
  }, {});

  const itemsByTemplate = items.reduce((acc, item) => {
    (acc[item.template_id] = acc[item.template_id] || []).push({
      ...item,
      subItems: subItemsByItem[item.id] || []
    });
    return acc;
  }, {});

  return templates.map(t => ({
    ...t,
    items: itemsByTemplate[t.id] || []
  }));
}

async function createTemplate(name, description) {
  const trimmed = (name || '').trim();
  if (!trimmed) throw new Error('Template name is required');
  await db.query('INSERT INTO todo_templates (name, description) VALUES (?, ?)', [trimmed, description || null]);
}

async function addTemplateItem(templateId, title) {
  const trimmed = (title || '').trim();
  if (!trimmed) throw new Error('Item title is required');
  const [[row]] = await db.query(
    'SELECT COALESCE(MAX(position), 0) + 1 AS nextPos FROM todo_template_items WHERE template_id = ?',
    [templateId]
  );
  const position = (row && row.nextPos) || 1;
  await db.query(
    'INSERT INTO todo_template_items (template_id, title, position) VALUES (?, ?, ?)',
    [templateId, trimmed, position]
  );
}

async function addTemplateSubitem(itemId, title) {
  const trimmed = (title || '').trim();
  if (!trimmed) throw new Error('Sub-item title is required');
  const [[row]] = await db.query(
    'SELECT COALESCE(MAX(position), 0) + 1 AS nextPos FROM todo_template_subitems WHERE item_id = ?',
    [itemId]
  );
  const position = (row && row.nextPos) || 1;
  await db.query(
    'INSERT INTO todo_template_subitems (item_id, title, position) VALUES (?, ?, ?)',
    [itemId, trimmed, position]
  );
}

async function assignTemplateToStudent(studentId, templateId) {
  const [[existing]] = await db.query(
    'SELECT id FROM student_todo_assignments WHERE student_id = ? AND template_id = ?',
    [studentId, templateId]
  );
  if (existing) return existing.id;

  const [assignmentResult] = await db.query(
    'INSERT INTO student_todo_assignments (student_id, template_id) VALUES (?, ?)',
    [studentId, templateId]
  );
  const assignmentId = assignmentResult.insertId;

  const [items] = await db.query(
    'SELECT * FROM todo_template_items WHERE template_id = ? ORDER BY position ASC, id ASC',
    [templateId]
  );
  if (!items.length) return assignmentId;

  const itemIds = items.map(i => i.id);
  const subItemsQuery = itemIds.length
    ? await db.query(
        `SELECT * FROM todo_template_subitems WHERE item_id IN (${itemIds.map(() => '?').join(',')})
         ORDER BY position ASC, id ASC`,
        itemIds
      )
    : [ [] ];
  const subItems = subItemsQuery[0];
  const subByItem = subItems.reduce((acc, sub) => {
    (acc[sub.item_id] = acc[sub.item_id] || []).push(sub);
    return acc;
  }, {});

  const itemIdMap = new Map();
  for (const item of items) {
    const [result] = await db.query(
      `INSERT INTO student_todo_items
        (assignment_id, parent_item_id, template_item_id, template_subitem_id, title, is_subitem, position)
       VALUES (?, NULL, ?, NULL, ?, 0, ?)`,
      [assignmentId, item.id, item.title, item.position || 0]
    );
    itemIdMap.set(item.id, result.insertId);

    const children = subByItem[item.id] || [];
    for (const child of children) {
      await db.query(
        `INSERT INTO student_todo_items
          (assignment_id, parent_item_id, template_item_id, template_subitem_id, title, is_subitem, position)
         VALUES (?, ?, ?, ?, ?, 1, ?)`,
        [assignmentId, result.insertId, item.id, child.id, child.title, child.position || 0]
      );
    }
  }

  return assignmentId;
}

async function getStudentAssignments(studentId) {
  const [assignments] = await db.query(
    `SELECT a.*, t.name AS templateName, t.description AS templateDescription
     FROM student_todo_assignments a
     JOIN todo_templates t ON t.id = a.template_id
     WHERE a.student_id = ?
     ORDER BY a.assigned_at DESC`,
    [studentId]
  );
  if (!assignments.length) return [];

  const assignmentIds = assignments.map(a => a.id);
  const [items] = await db.query(
    `SELECT *
     FROM student_todo_items
     WHERE assignment_id IN (${assignmentIds.map(() => '?').join(',')})
     ORDER BY parent_item_id IS NOT NULL, position ASC, id ASC`,
    assignmentIds
  );

  const assignmentsMap = new Map();
  assignments.forEach(a => {
    assignmentsMap.set(a.id, {
      id: a.id,
      templateId: a.template_id,
      templateName: a.templateName,
      templateDescription: a.templateDescription,
      assignedAt: a.assigned_at,
      total: 0,
      completed: 0,
      items: [],
      _itemLookup: new Map()
    });
  });

  items.forEach(item => {
    const container = assignmentsMap.get(item.assignment_id);
    if (!container) return;

    const record = {
      id: item.id,
      title: item.title,
      isSubitem: !!item.is_subitem,
      parentId: item.parent_item_id,
      completed: !!item.completed,
      completedAt: item.completed_at,
      subItems: []
    };

    container.total += 1;
    if (record.completed) container.completed += 1;

    if (record.isSubitem && record.parentId && container._itemLookup.has(record.parentId)) {
      container._itemLookup.get(record.parentId).subItems.push(record);
    } else {
      container.items.push(record);
      container._itemLookup.set(record.id, record);
    }
  });

  return Array.from(assignmentsMap.values()).map(a => {
    const progress = a.total ? Math.round((a.completed / a.total) * 100) : 0;
    delete a._itemLookup;
    return { ...a, progress };
  });
}

async function setTodoItemCompletion(studentId, itemId, completed) {
  const [[item]] = await db.query(
    `SELECT i.*, a.student_id
     FROM student_todo_items i
     JOIN student_todo_assignments a ON a.id = i.assignment_id
     WHERE i.id = ?`,
    [itemId]
  );
  if (!item || item.student_id !== studentId) throw new Error('Item not found');

  const now = new Date().toISOString().slice(0, 19).replace('T', ' ');
  await db.query(
    'UPDATE student_todo_items SET completed = ?, completed_at = ? WHERE id = ?',
    [completed ? 1 : 0, completed ? now : null, itemId]
  );
  return { assignmentId: item.assignment_id, completedAt: completed ? now : null };
}

async function addStudentTodoItem(studentId, assignmentId, title, parentItemId) {
  const [[assignment]] = await db.query(
    'SELECT * FROM student_todo_assignments WHERE id = ? AND student_id = ?',
    [assignmentId, studentId]
  );
  if (!assignment) throw new Error('Assignment not found');

  const trimmed = (title || '').trim();
  if (!trimmed) throw new Error('Title is required');

  let isSubitem = 0;
  let parentId = null;

  if (parentItemId) {
    const [[parent]] = await db.query(
      'SELECT * FROM student_todo_items WHERE id = ? AND assignment_id = ?',
      [parentItemId, assignmentId]
    );
    if (!parent) throw new Error('Parent item not found');
    parentId = parent.id;
    isSubitem = 1;
  }

  const positionQuery = await db.query(
    parentId
      ? 'SELECT COALESCE(MAX(position), 0) + 1 AS nextPos FROM student_todo_items WHERE assignment_id = ? AND parent_item_id = ?'
      : 'SELECT COALESCE(MAX(position), 0) + 1 AS nextPos FROM student_todo_items WHERE assignment_id = ? AND parent_item_id IS NULL',
    parentId ? [assignmentId, parentId] : [assignmentId]
  );
  const [[row]] = positionQuery;
  const position = (row && row.nextPos) || 1;

  const [result] = await db.query(
    `INSERT INTO student_todo_items
      (assignment_id, parent_item_id, template_item_id, template_subitem_id, title, is_subitem, position)
     VALUES (?, ?, NULL, NULL, ?, ?, ?)`,
    [assignmentId, parentId, trimmed, isSubitem, position]
  );

  const insertedId = result.insertId;
  const [[created]] = await db.query('SELECT * FROM student_todo_items WHERE id = ?', [insertedId]);
  return {
    id: created.id,
    assignmentId,
    parentItemId: created.parent_item_id,
    title: created.title,
    isSubitem: !!created.is_subitem,
    completed: !!created.completed,
    completedAt: created.completed_at,
    position: created.position
  };
}

async function deleteStudentAssignment(studentId, assignmentId) {
  await db.query(
    'DELETE FROM student_todo_assignments WHERE id = ? AND student_id = ?',
    [assignmentId, studentId]
  );
  return true;
}

module.exports = {
  getAllTemplates,
  createTemplate,
  addTemplateItem,
  addTemplateSubitem,
  assignTemplateToStudent,
  getStudentAssignments,
  setTodoItemCompletion,
  addStudentTodoItem,
  deleteStudentAssignment
};
