const db = require('./db');

async function create({
  name,
  email,
  phone,
  phoneCarrier,
  address,
  zip,
  state,
  serving,
  branch,
  course,
  applicantType,
  referral,
  referralEmail,
  consent
}) {
  const [result] = await db.query(
    'INSERT INTO mdtslms_pre_registrations (name, email, phone, phoneCarrier, address, zip, state, serving, branch, course, applicantType, referral, referralEmail, consent, createdAt) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?, ?, NOW())',
    [
      name,
      email || null,
      phone,
      phoneCarrier || null,
      address || null,
      zip,
      state,
      serving ? 1 : 0,
      branch || null,
      course,
      applicantType,
      referral || null,
      referralEmail || null,
      consent ? 1 : 0
    ]
  );
  return { id: result.insertId };
}

let ensuredPreLastContacted = false;
async function ensurePreLastContactedColumn() {
  if (ensuredPreLastContacted) return;
  try {
    await db.query('ALTER TABLE mdtslms_pre_registrations ADD COLUMN IF NOT EXISTS lastContacted DATETIME NULL');
  } catch (e) {
    // ignore if exists
  } finally {
    ensuredPreLastContacted = true;
  }
}

let ensuredSignatureCollected = false;
async function ensureSignatureCollectedColumn() {
  if (ensuredSignatureCollected) return;
  try {
    await db.query('ALTER TABLE mdtslms_pre_registrations ADD COLUMN signatureCollected TINYINT(1) NOT NULL DEFAULT 0');
  } catch (e) {
    if (e && e.code !== 'ER_DUP_FIELDNAME') {
      console.error('ensureSignatureCollectedColumn error', e);
    }
  } finally {
    ensuredSignatureCollected = true;
  }
}

async function setLastContacted(id, when = new Date()) {
  await ensurePreLastContactedColumn();
  const ts = new Date(when).toISOString().slice(0, 19).replace('T', ' ');
  try {
    await db.query('UPDATE mdtslms_pre_registrations SET lastContacted=? WHERE id=?', [ts, id]);
  } catch (e) {
    try {
      await db.query('ALTER TABLE mdtslms_pre_registrations ADD COLUMN lastContacted DATETIME NULL');
    } catch (_) {}
    await db.query('UPDATE mdtslms_pre_registrations SET lastContacted=? WHERE id=?', [ts, id]);
  }
  return true;
}

async function setSignatureCollected(id, collected) {
  await ensureSignatureCollectedColumn();
  const flag = collected ? 1 : 0;
  await db.query('UPDATE mdtslms_pre_registrations SET signatureCollected=? WHERE id=?', [flag, id]);
  return true;
}

async function getAll() {
  await ensureSignatureCollectedColumn();
  const [rows] = await db.query('SELECT * FROM mdtslms_pre_registrations ORDER BY createdAt DESC');
  return rows;
}

async function update(id, {
  name,
  email,
  phone,
  phoneCarrier,
  address,
  zip,
  state,
  serving,
  branch,
  course,
  applicantType,
  referral,
  referralEmail
}) {
  const [result] = await db.query(
    `UPDATE mdtslms_pre_registrations 
     SET name=?, email=?, phone=?, phoneCarrier=?, address=?, zip=?, state=?, 
         serving=?, branch=?, course=?, applicantType=?, referral=?, referralEmail=?
     WHERE id=?`,
    [
      name,
      email || null,
      phone,
      phoneCarrier || null,
      address || null,
      zip,
      state,
      serving ? 1 : 0,
      branch || null,
      course,
      applicantType,
      referral || null,
      referralEmail || null,
      id
    ]
  );
  return { affectedRows: result.affectedRows };
}

async function findById(id) {
  await ensureSignatureCollectedColumn();
  const [rows] = await db.query('SELECT * FROM mdtslms_pre_registrations WHERE id = ?', [id]);
  return rows[0] || null;
}

async function deleteById(id) {
  const [result] = await db.query('DELETE FROM mdtslms_pre_registrations WHERE id = ?', [id]);
  return { affectedRows: result.affectedRows };
}

async function findByEmail(email) {
  await ensureSignatureCollectedColumn();
  const [rows] = await db.query('SELECT * FROM mdtslms_pre_registrations WHERE email = ? ORDER BY createdAt DESC LIMIT 1', [email]);
  return rows[0] || null;
}

module.exports = { create, getAll, setLastContacted, setSignatureCollected, update, findById, deleteById, findByEmail };
