const db = require('./db');

async function init() {
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_partners (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    notes TEXT NULL,
    createdAt DATETIME DEFAULT CURRENT_TIMESTAMP
  )`);
  await db.query('ALTER TABLE mdtslms_partners ADD COLUMN website VARCHAR(255) NULL').catch(() => {});
  await db.query('ALTER TABLE mdtslms_partners ADD COLUMN phone VARCHAR(100) NULL').catch(() => {});
  await db.query('ALTER TABLE mdtslms_partners ADD COLUMN address TEXT NULL').catch(() => {});
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_partner_contacts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    partnerId INT NOT NULL,
    role VARCHAR(120) NULL,
    name VARCHAR(255) NOT NULL,
    phone VARCHAR(100) NULL,
    email VARCHAR(255) NULL,
    notes TEXT NULL,
    createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX(partnerId),
    FOREIGN KEY (partnerId) REFERENCES mdtslms_partners(id) ON DELETE CASCADE
  )`);
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_partner_documents (
    id INT AUTO_INCREMENT PRIMARY KEY,
    contactId INT NOT NULL,
    originalName VARCHAR(255) NOT NULL,
    storedName VARCHAR(255) NOT NULL,
    uploadedAt DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX(contactId),
    FOREIGN KEY (contactId) REFERENCES mdtslms_partner_contacts(id) ON DELETE CASCADE
  )`);
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_partner_contact_notes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    contactId INT NOT NULL,
    note TEXT NOT NULL,
    createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX(contactId),
    FOREIGN KEY (contactId) REFERENCES mdtslms_partner_contacts(id) ON DELETE CASCADE
  )`);
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_partner_notes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    partnerId INT NOT NULL,
    note TEXT NOT NULL,
    createdBy INT NULL,
    createdByName VARCHAR(255) NULL,
    createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX(partnerId),
    FOREIGN KEY (partnerId) REFERENCES mdtslms_partners(id) ON DELETE CASCADE
  )`);
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_partner_calendar_events (
    id INT AUTO_INCREMENT PRIMARY KEY,
    partnerId INT NOT NULL,
    calendarEventId INT NOT NULL,
    createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX(partnerId),
    INDEX(calendarEventId)
  )`);
}
init().catch(console.error);

async function list() {
  const [partners] = await db.query('SELECT * FROM mdtslms_partners ORDER BY name ASC');
  const [contacts] = await db.query('SELECT * FROM mdtslms_partner_contacts ORDER BY name ASC');
  const contactIds = contacts.map(c => c.id);
  const partnerIds = partners.map(p => p.id);
  let docsByContact = new Map();
  let notesByContact = new Map();
  let partnerNotesByPartner = new Map();
  let partnerEventsByPartner = new Map();
  let docsByPartner = new Map();
  let allContactDocs = [];
  if (contactIds.length) {
    const placeholders = contactIds.map(() => '?').join(',');
    const [docs] = await db.query(`SELECT * FROM mdtslms_partner_documents WHERE contactId IN (${placeholders}) ORDER BY uploadedAt DESC`, contactIds);
    allContactDocs = docs;
    docs.forEach((doc) => {
      const list = docsByContact.get(doc.contactId) || [];
      list.push(doc);
      docsByContact.set(doc.contactId, list);
    });
    const [notes] = await db.query(`SELECT * FROM mdtslms_partner_contact_notes WHERE contactId IN (${placeholders}) ORDER BY createdAt DESC`, contactIds);
    notes.forEach((note) => {
      const list = notesByContact.get(note.contactId) || [];
      list.push(note);
      notesByContact.set(note.contactId, list);
    });
  }
  const contactsById = new Map(contacts.map((c) => [c.id, c]));
  if (partnerIds.length) {
    const partnerPlaceholders = partnerIds.map(() => '?').join(',');
    const [pNotes] = await db.query(`SELECT * FROM mdtslms_partner_notes WHERE partnerId IN (${partnerPlaceholders}) ORDER BY createdAt DESC`, partnerIds);
    pNotes.forEach((note) => {
      const list = partnerNotesByPartner.get(note.partnerId) || [];
      list.push(note);
      partnerNotesByPartner.set(note.partnerId, list);
    });
    if (allContactDocs.length) {
      allContactDocs.forEach((doc) => {
        const contact = contactsById.get(doc.contactId);
        if (!contact) return;
        const partnerList = docsByPartner.get(contact.partnerId) || [];
        partnerList.push({
          ...doc,
          contactName: contact.name || ''
        });
        docsByPartner.set(contact.partnerId, partnerList);
      });
    }
    const [pEvents] = await db.query(
      `SELECT pe.*, ce.title, ce.start_at AS startAt, ce.end_at AS endAt
       FROM mdtslms_partner_calendar_events pe
       JOIN mdtslms_calendar_events ce ON ce.id = pe.calendarEventId
       WHERE pe.partnerId IN (${partnerPlaceholders})
       ORDER BY ce.start_at DESC`,
      partnerIds
    );
    pEvents.forEach((row) => {
      const list = partnerEventsByPartner.get(row.partnerId) || [];
      list.push({
        id: row.id,
        calendarEventId: row.calendarEventId,
        title: row.title,
        startAt: row.startAt,
        endAt: row.endAt,
        createdAt: row.createdAt
      });
      partnerEventsByPartner.set(row.partnerId, list);
    });
  }
  const byPartner = new Map();
  contacts.forEach(c => {
    const enriched = { ...c, documents: docsByContact.get(c.id) || [], extraNotes: notesByContact.get(c.id) || [] };
    const p = byPartner.get(c.partnerId) || []; p.push(enriched); byPartner.set(c.partnerId, p);
  });
  return partners.map(p => ({
    ...p,
    contacts: byPartner.get(p.id) || [],
    partnerNotes: partnerNotesByPartner.get(p.id) || [],
    partnerDocuments: docsByPartner.get(p.id) || [],
    partnerEvents: partnerEventsByPartner.get(p.id) || []
  }));
}

async function createPartner({ name, notes, website, phone, address }) {
  const [res] = await db.query(
    'INSERT INTO mdtslms_partners (name, notes, website, phone, address) VALUES (?,?,?,?,?)',
    [name, notes || null, website || null, phone || null, address || null]
  );
  return { id: res.insertId };
}

async function getPartnerById(id) {
  const [rows] = await db.query('SELECT * FROM mdtslms_partners WHERE id=?', [id]);
  return rows[0] || null;
}

async function updatePartner(id, { name, website, phone, address, notes }) {
  const updates = [];
  const params = [];
  if (name !== undefined) { updates.push('name = ?'); params.push(name); }
  if (website !== undefined) { updates.push('website = ?'); params.push(website || null); }
  if (phone !== undefined) { updates.push('phone = ?'); params.push(phone || null); }
  if (address !== undefined) { updates.push('address = ?'); params.push(address || null); }
  if (notes !== undefined) { updates.push('notes = ?'); params.push(notes || null); }
  if (!updates.length) return true;
  params.push(id);
  await db.query(`UPDATE mdtslms_partners SET ${updates.join(', ')} WHERE id = ?`, params);
  return true;
}

async function deletePartner(id) {
  await db.query('DELETE FROM mdtslms_partners WHERE id=?', [id]);
  return true;
}

async function addContact(partnerId, { role, name, phone, email, notes }) {
  const [res] = await db.query(
    'INSERT INTO mdtslms_partner_contacts (partnerId, role, name, phone, email, notes) VALUES (?,?,?,?,?,?)',
    [partnerId, role || null, name, phone || null, (email||null), notes || null]
  );
  return { id: res.insertId };
}

async function deleteContact(id) {
  await db.query('DELETE FROM mdtslms_partner_contacts WHERE id=?', [id]);
  return true;
}

async function getContactById(id) {
  const [rows] = await db.query('SELECT * FROM mdtslms_partner_contacts WHERE id=?', [id]);
  return rows[0] || null;
}

async function addDocument(contactId, { originalName, storedName }) {
  const [res] = await db.query('INSERT INTO mdtslms_partner_documents (contactId, originalName, storedName) VALUES (?,?,?)', [contactId, originalName, storedName]);
  return { id: res.insertId };
}

async function deleteDocument(id) {
  const [rows] = await db.query('SELECT * FROM mdtslms_partner_documents WHERE id=?', [id]);
  if (!rows.length) return null;
  const doc = rows[0];
  await db.query('DELETE FROM mdtslms_partner_documents WHERE id=?', [id]);
  return doc;
}

async function addContactNote(contactId, note) {
  const [res] = await db.query('INSERT INTO mdtslms_partner_contact_notes (contactId, note) VALUES (?, ?)', [contactId, note]);
  return { id: res.insertId };
}

async function deleteContactNote(id) {
  await db.query('DELETE FROM mdtslms_partner_contact_notes WHERE id=?', [id]);
  return true;
}

async function addPartnerNote(partnerId, { note, createdBy, createdByName }) {
  const [res] = await db.query(
    'INSERT INTO mdtslms_partner_notes (partnerId, note, createdBy, createdByName) VALUES (?,?,?,?)',
    [partnerId, note, createdBy || null, createdByName || null]
  );
  return { id: res.insertId };
}

async function deletePartnerNote(id) {
  await db.query('DELETE FROM mdtslms_partner_notes WHERE id=?', [id]);
  return true;
}

async function listContactsByPartner(partnerId) {
  const [rows] = await db.query('SELECT * FROM mdtslms_partner_contacts WHERE partnerId = ? ORDER BY name ASC', [partnerId]);
  const ids = rows.map((row) => row.id);
  if (!ids.length) return rows.map((row) => ({ ...row, documents: [], extraNotes: [] }));
  const placeholders = ids.map(() => '?').join(',');
  const [docs] = await db.query(`SELECT * FROM mdtslms_partner_documents WHERE contactId IN (${placeholders}) ORDER BY uploadedAt DESC`, ids);
  const [notes] = await db.query(`SELECT * FROM mdtslms_partner_contact_notes WHERE contactId IN (${placeholders}) ORDER BY createdAt DESC`, ids);
  const docMap = new Map();
  const noteMap = new Map();
  docs.forEach((doc) => {
    const list = docMap.get(doc.contactId) || [];
    list.push(doc);
    docMap.set(doc.contactId, list);
  });
  notes.forEach((note) => {
    const list = noteMap.get(note.contactId) || [];
    list.push(note);
    noteMap.set(note.contactId, list);
  });
  return rows.map((row) => ({
    ...row,
    documents: docMap.get(row.id) || [],
    extraNotes: noteMap.get(row.id) || []
  }));
}

async function linkCalendarEvent(partnerId, calendarEventId) {
  const [res] = await db.query(
    'INSERT INTO mdtslms_partner_calendar_events (partnerId, calendarEventId) VALUES (?,?)',
    [partnerId, calendarEventId]
  );
  return { id: res.insertId };
}

async function removeLinkedCalendarEvent(id) {
  await db.query('DELETE FROM mdtslms_partner_calendar_events WHERE id = ?', [id]);
  return true;
}

module.exports = {
  list,
  createPartner,
  getPartnerById,
  updatePartner,
  deletePartner,
  addContact,
  deleteContact,
  getContactById,
  addDocument,
  deleteDocument,
  addContactNote,
  deleteContactNote,
  addPartnerNote,
  deletePartnerNote,
  listContactsByPartner,
  linkCalendarEvent,
  removeLinkedCalendarEvent
};
