const db = require('./db');

// Initialize tables for leads and contacts
async function init() {
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_leads (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255),
    email VARCHAR(255) UNIQUE,
    phone VARCHAR(100),
    interestPercent INT DEFAULT 0,
    source VARCHAR(50),
    createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
    updatedAt DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    lastContacted DATETIME NULL,
    contactCount INT DEFAULT 0
  )`);
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_lead_contacts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    leadId INT NOT NULL,
    direction VARCHAR(10),
    method VARCHAR(20),
    note TEXT,
    createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX(leadId),
    FOREIGN KEY (leadId) REFERENCES mdtslms_leads(id) ON DELETE CASCADE
  )`);
}
init().catch(console.error);

async function upsertByEmail({ name, email, phone, interestPercent = 0, source = '' }) {
  if (!email) return null;
  const [rows] = await db.query('SELECT * FROM mdtslms_leads WHERE email = ?', [email]);
  const pct = Math.max(0, Math.min(100, Number(interestPercent) || 0));
  if (rows.length) {
    const lead = rows[0];
    const newPct = Math.max(lead.interestPercent || 0, pct);
    const newName = (lead.name && lead.name.trim()) ? lead.name : (name || '').trim();
    const newPhone = (lead.phone && lead.phone.trim()) ? lead.phone : (phone || '').trim();
    await db.query(
      'UPDATE mdtslms_leads SET name=?, phone=?, interestPercent=?, source=IF(?<>\'\', ?, source) WHERE id=?',
      [newName || null, newPhone || null, newPct, source, source, lead.id]
    );
    return { id: lead.id };
  } else {
    const [result] = await db.query(
      'INSERT INTO mdtslms_leads (name, email, phone, interestPercent, source) VALUES (?,?,?,?,?)',
      [(name || '').trim() || null, email.trim(), (phone || '').trim() || null, pct, source]
    );
    return { id: result.insertId };
  }
}

async function addContact(leadId, { direction = 'outbound', method = 'other', note = '' }) {
  await db.query(
    'INSERT INTO mdtslms_lead_contacts (leadId, direction, method, note) VALUES (?,?,?,?)',
    [leadId, direction, method, note]
  );
  await db.query(
    'UPDATE mdtslms_leads SET lastContacted = NOW(), contactCount = contactCount + 1 WHERE id = ?',
    [leadId]
  );
  return true;
}

async function updateInterest(leadId, interestPercent) {
  const pct = Math.max(0, Math.min(100, Number(interestPercent) || 0));
  await db.query('UPDATE mdtslms_leads SET interestPercent=? WHERE id=?', [pct, leadId]);
}

async function getAll() {
  const [rows] = await db.query('SELECT * FROM mdtslms_leads ORDER BY updatedAt DESC');
  return rows;
}

async function getContacts(leadId) {
  const [rows] = await db.query('SELECT * FROM mdtslms_lead_contacts WHERE leadId=? ORDER BY createdAt DESC', [leadId]);
  return rows;
}

async function deleteLead(id) {
  await db.query('DELETE FROM mdtslms_leads WHERE id=?', [id]);
  return true;
}

async function deleteContact(contactId) {
  const [rows] = await db.query('SELECT leadId FROM mdtslms_lead_contacts WHERE id=?', [contactId]);
  if (!rows.length) return false;
  const leadId = rows[0].leadId;
  await db.query('DELETE FROM mdtslms_lead_contacts WHERE id=?', [contactId]);
  await db.query('UPDATE mdtslms_leads SET contactCount = GREATEST(contactCount - 1, 0) WHERE id=?', [leadId]);
  return true;
}

module.exports = {
  upsertByEmail,
  addContact,
  updateInterest,
  getAll,
  getContacts,
  deleteLead,
  deleteContact,
  getByEmail: async function(email){
    if (!email) return null;
    const [rows] = await db.query('SELECT * FROM mdtslms_leads WHERE email=?', [email]);
    return rows[0] || null;
  },
  getById: async function(id){
    const [rows] = await db.query('SELECT * FROM mdtslms_leads WHERE id=?', [id]);
    return rows[0] || null;
  }
};
