const db = require('./db');

// Initialize the interviews table
async function init() {
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_interviews (
    id INT AUTO_INCREMENT PRIMARY KEY,
    studentName VARCHAR(255) NOT NULL,
    studentEmail VARCHAR(255) NOT NULL,
    studentPhone VARCHAR(100) NULL,
    scheduledAt DATETIME NOT NULL,
    durationMinutes INT NOT NULL DEFAULT 30,
    status VARCHAR(20) NOT NULL DEFAULT 'scheduled',
    assignedAdminId INT NULL,
    notes TEXT NULL,
    createdAt DATETIME DEFAULT CURRENT_TIMESTAMP,
    updatedAt DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX (scheduledAt),
    INDEX (assignedAdminId)
  )`);
}
init().catch(console.error);

function mapRow(row) { return row; }

async function create({ studentName, studentEmail, studentPhone, scheduledAt, durationMinutes = 30, notes = '', assignedAdminId = null }) {
  const ts = new Date(scheduledAt);
  const sqlTs = ts.toISOString().slice(0, 19).replace('T', ' ');
  const [result] = await db.query(
    'INSERT INTO mdtslms_interviews (studentName, studentEmail, studentPhone, scheduledAt, durationMinutes, status, assignedAdminId, notes) VALUES (?,?,?,?,?, \'scheduled\', ?, ?)',
    [studentName, studentEmail, studentPhone || null, sqlTs, durationMinutes, assignedAdminId ? Number(assignedAdminId) : null, notes || null]
  );
  return findById(result.insertId);
}

async function findById(id) {
  const [rows] = await db.query('SELECT * FROM mdtslms_interviews WHERE id=?', [id]);
  return mapRow(rows[0] || null);
}

async function listAll() {
  const [rows] = await db.query('SELECT * FROM mdtslms_interviews ORDER BY scheduledAt DESC');
  return rows.map(mapRow);
}

async function listBetween(start, end) {
  const s = new Date(start).toISOString().slice(0,19).replace('T',' ');
  const e = new Date(end).toISOString().slice(0,19).replace('T',' ');
  const [rows] = await db.query('SELECT * FROM mdtslms_interviews WHERE scheduledAt BETWEEN ? AND ? ORDER BY scheduledAt ASC', [s, e]);
  return rows.map(mapRow);
}

async function assignAdmin(id, adminId) {
  const aid = adminId ? Number(adminId) : null;
  await db.query('UPDATE mdtslms_interviews SET assignedAdminId=? WHERE id=?', [aid, id]);
  return findById(id);
}

async function setStatus(id, status) {
  await db.query('UPDATE mdtslms_interviews SET status=? WHERE id=?', [status, id]);
  return findById(id);
}

async function remove(id) {
  const [result] = await db.query('DELETE FROM mdtslms_interviews WHERE id=?', [id]);
  return result.affectedRows > 0;
}

module.exports = {
  create,
  findById,
  listAll,
  listBetween,
  assignAdmin,
  setStatus,
  remove
};
