const pool = require('./db');

// Initialize courses table if it doesn't exist
async function initializeTable() {
  try {
    await pool.query(`
      CREATE TABLE IF NOT EXISTS courses (
        id INT AUTO_INCREMENT PRIMARY KEY,
        program_name VARCHAR(255) NOT NULL UNIQUE,
        program_hours INT DEFAULT 0,
        program_cost DECIMAL(10, 2) DEFAULT 0.00,
        credential_awarded VARCHAR(100) DEFAULT 'Certificate',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);
    console.log('Courses table initialized');
  } catch (err) {
    console.error('Error initializing courses table:', err);
  }
}

// Get all courses
async function getAllCourses() {
  try {
    const [rows] = await pool.query('SELECT * FROM courses ORDER BY program_name ASC');
    return rows;
  } catch (err) {
    console.error('Error getting courses:', err);
    return [];
  }
}

// Get course by name
async function getCourseByName(programName) {
  try {
    const [rows] = await pool.query('SELECT * FROM courses WHERE program_name = ?', [programName]);
    return rows[0] || null;
  } catch (err) {
    console.error('Error getting course by name:', err);
    return null;
  }
}

// Add or update a course
async function upsertCourse(programName, hours, cost, credential) {
  try {
    await pool.query(
      `INSERT INTO courses (program_name, program_hours, program_cost, credential_awarded)
       VALUES (?, ?, ?, ?)
       ON DUPLICATE KEY UPDATE
       program_hours = VALUES(program_hours),
       program_cost = VALUES(program_cost),
       credential_awarded = VALUES(credential_awarded)`,
      [programName, hours || 0, cost || 0, credential || 'Certificate']
    );
    return true;
  } catch (err) {
    console.error('Error upserting course:', err);
    return false;
  }
}

// Update a course
async function updateCourse(id, data) {
  try {
    const { program_hours, program_cost, credential_awarded } = data;
    await pool.query(
      'UPDATE courses SET program_hours = ?, program_cost = ?, credential_awarded = ? WHERE id = ?',
      [program_hours || 0, program_cost || 0, credential_awarded || 'Certificate', id]
    );
    return true;
  } catch (err) {
    console.error('Error updating course:', err);
    return false;
  }
}

// Update course program name (including database and dropdowns)
async function updateProgramName(id, newName) {
  try {
    // Get current course data
    const [rows] = await pool.query('SELECT * FROM courses WHERE id = ?', [id]);
    if (!rows || rows.length === 0) {
      return { success: false, error: 'Course not found' };
    }
    
    const oldName = rows[0].program_name;
    const trimmedNewName = newName.trim();
    
    // Check if name is the same
    if (oldName === trimmedNewName) {
      return { success: true, message: 'No change needed' };
    }
    
    // Check if new name already exists
    const [existing] = await pool.query('SELECT * FROM courses WHERE program_name = ? AND id != ?', [trimmedNewName, id]);
    if (existing && existing.length > 0) {
      return { success: false, error: 'A course with this name already exists' };
    }
    
    // Update database
    await pool.query('UPDATE courses SET program_name = ? WHERE id = ?', [trimmedNewName, id]);
    
    return { success: true, oldName, newName: trimmedNewName };
  } catch (err) {
    console.error('Error updating program name:', err);
    return { success: false, error: 'Failed to update program name' };
  }
}

// Delete a course
async function deleteCourse(id) {
  try {
    await pool.query('DELETE FROM courses WHERE id = ?', [id]);
    return true;
  } catch (err) {
    console.error('Error deleting course:', err);
    return false;
  }
}

// Sync courses from dropdowns.json
async function syncFromDropdowns(coursesList) {
  try {
    for (const courseName of coursesList) {
      const existing = await getCourseByName(courseName);
      if (!existing) {
        // Add with default values - will be updated through admin panel
        await upsertCourse(courseName, 0, 0, 'Certificate');
      }
    }
    return true;
  } catch (err) {
    console.error('Error syncing courses:', err);
    return false;
  }
}

// Initialize table on module load
initializeTable();

module.exports = {
  getAllCourses,
  getCourseByName,
  upsertCourse,
  updateCourse,
  updateProgramName,
  deleteCourse,
  syncFromDropdowns,
  initializeTable
};
