const db = require('./db');

function mapClass(row) {
  if (!row) return row;
 ['studentIds', 'schedule', 'lectures', 'simulations', 'assignments', 'tests', 'grades', 'attendance', 'checklist', 'submissions', 'studentEnrollments', 'links'].forEach(k => {    if (row[k] && typeof row[k] === 'string') {
      try { row[k] = JSON.parse(row[k]); } catch (_) { row[k] = []; }
    } else if (!row[k]) {
      row[k] = [];
    }
  });
  return row;
}



async function getAllClasses() {
  const [rows] = await db.query('SELECT * FROM mdtslms_classes');
  return rows.map(mapClass);
}

async function findClassById(id) {
  const [rows] = await db.query('SELECT * FROM mdtslms_classes WHERE id = ?', [id]);
  return mapClass(rows[0]);
}

let classCreatedAtSupported = null;
let classClockHoursSupported = null;

async function ensureClassCreatedAtColumn() {
  if (classCreatedAtSupported !== null) return classCreatedAtSupported;
  try {
    const [columns] = await db.query("SHOW COLUMNS FROM mdtslms_classes LIKE 'createdAt'");
    if (!columns.length) {
      await db.query('ALTER TABLE mdtslms_classes ADD COLUMN createdAt DATETIME NULL');
    }
    classCreatedAtSupported = true;
  } catch (e) {
    console.error('ensureClassCreatedAtColumn error:', e.message || e);
    classCreatedAtSupported = false;
  }
  return classCreatedAtSupported;
}

async function ensureClassClockHoursColumn() {
  if (classClockHoursSupported !== null) return classClockHoursSupported;
  try {
    const [columns] = await db.query("SHOW COLUMNS FROM mdtslms_classes LIKE 'clockHours'");
    if (!columns.length) {
      await db.query('ALTER TABLE mdtslms_classes ADD COLUMN clockHours INT DEFAULT 0');
    }
    classClockHoursSupported = true;
  } catch (e) {
    console.error('ensureClassClockHoursColumn error:', e.message || e);
    classClockHoursSupported = false;
  }
  return classClockHoursSupported;
}

let studentEnrollmentsSupported = null;
async function ensureStudentEnrollmentsColumn() {
  if (studentEnrollmentsSupported !== null) return studentEnrollmentsSupported;
  try {
    const [columns] = await db.query("SHOW COLUMNS FROM mdtslms_classes LIKE 'studentEnrollments'");
    if (!columns.length) {
      await db.query('ALTER TABLE mdtslms_classes ADD COLUMN studentEnrollments JSON NULL');
    }
    studentEnrollmentsSupported = true;
  } catch (e) {
    console.error('ensureStudentEnrollmentsColumn error:', e.message || e);
    studentEnrollmentsSupported = false;
  }
  return studentEnrollmentsSupported;
}

let assistantTeacherIdSupported = null;
async function ensureAssistantTeacherIdColumn() {
  if (assistantTeacherIdSupported !== null) return assistantTeacherIdSupported;
  try {
    const [columns] = await db.query("SHOW COLUMNS FROM mdtslms_classes LIKE 'assistantTeacherId'");
    if (!columns.length) {
      await db.query('ALTER TABLE mdtslms_classes ADD COLUMN assistantTeacherId INT NULL');
    }
    assistantTeacherIdSupported = true;
  } catch (e) {
    console.error('ensureAssistantTeacherIdColumn error:', e.message || e);
    assistantTeacherIdSupported = false;
  }
  return assistantTeacherIdSupported;
}

let syllabusColumnsSupported = null;
async function ensureSyllabusColumns() {
  if (syllabusColumnsSupported !== null) return syllabusColumnsSupported;
  try {
    const [pathCols] = await db.query("SHOW COLUMNS FROM mdtslms_classes LIKE 'syllabusPath'");
    if (!pathCols.length) {
      await db.query('ALTER TABLE mdtslms_classes ADD COLUMN syllabusPath VARCHAR(500) NULL');
    }
    const [nameCols] = await db.query("SHOW COLUMNS FROM mdtslms_classes LIKE 'syllabusFileName'");
    if (!nameCols.length) {
      await db.query('ALTER TABLE mdtslms_classes ADD COLUMN syllabusFileName VARCHAR(255) NULL');
    }
    const [dateCols] = await db.query("SHOW COLUMNS FROM mdtslms_classes LIKE 'syllabusUploadedAt'");
    if (!dateCols.length) {
      await db.query('ALTER TABLE mdtslms_classes ADD COLUMN syllabusUploadedAt DATETIME NULL');
    }
    syllabusColumnsSupported = true;
  } catch (e) {
    console.error('ensureSyllabusColumns error:', e.message || e);
    syllabusColumnsSupported = false;
  }
  return syllabusColumnsSupported;
}

let locationSupported = null;
async function ensureLocationColumn() {
  if (locationSupported !== null) return locationSupported;
  try {
    const [columns] = await db.query("SHOW COLUMNS FROM mdtslms_classes LIKE 'location'");
    if (!columns.length) {
      await db.query('ALTER TABLE mdtslms_classes ADD COLUMN location VARCHAR(500) NULL');
    }
    locationSupported = true;
  } catch (e) {
    console.error('ensureLocationColumn error:', e.message || e);
    locationSupported = false;
  }
  return locationSupported;
}

async function createClass({ schoolYear, cohort, name, shortName, description, location, teacherId, assistantTeacherId, schedule, weeks, startDate, endDate, clockHours }) {
  const hasCreatedAt = await ensureClassCreatedAtColumn();
  const hasClockHours = await ensureClassClockHoursColumn();
  const hasAssistantTeacherId = await ensureAssistantTeacherIdColumn();
  const hasLocation = await ensureLocationColumn();
  const ts = new Date().toISOString().slice(0, 19).replace('T', ' ');
  const columns = [
    'schoolYear',
    'cohort',
    'name',
    'shortName',
    'description',
    'teacherId',
    'weeks',
    'startDate',
    'endDate',
    'studentIds',
    'schedule',
    'lectures',
    'simulations',
    'assignments',
    'tests',
    'grades',
    'attendance',
    'checklist'
  ];
  const values = [
    schoolYear,
    cohort,
    name,
    shortName,
    description,
    teacherId,
    weeks || 0,
    startDate,
    endDate,
    JSON.stringify([]),
    JSON.stringify(schedule || []),
    JSON.stringify([]),
    JSON.stringify([]),
    JSON.stringify([]),
    JSON.stringify([]),
    JSON.stringify([]),
    JSON.stringify([]),
    JSON.stringify([])
  ];

  if (hasCreatedAt) {
    columns.push('createdAt');
    values.push(ts);
  }
  
  if (hasClockHours) {
    columns.push('clockHours');
    values.push(clockHours || 0);
  }

  if (hasAssistantTeacherId) {
    columns.push('assistantTeacherId');
    values.push(assistantTeacherId || null);
  }

  if (hasLocation) {
    columns.push('location');
    values.push(location || null);
  }

  const placeholders = columns.map(() => '?').join(',');
  const sql = `INSERT INTO mdtslms_classes (${columns.join(',')}) VALUES (${placeholders})`;
  const [result] = await db.query(sql, values);
  return findClassById(result.insertId);
}

async function addStudent(classId, studentId) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  klass.studentIds = klass.studentIds || [];
  klass.studentEnrollments = klass.studentEnrollments || [];
  
  // Add to studentIds if not already present
  if (!klass.studentIds.includes(studentId)) {
    klass.studentIds.push(studentId);
  }
  
  // Track enrollment date and clock hours completed
  const existing = klass.studentEnrollments.find(e => Number(e.studentId) === Number(studentId));
  if (!existing) {
    klass.studentEnrollments.push({
      studentId: Number(studentId),
      enrolledAt: new Date().toISOString(),
      clockHoursCompleted: 0
    });
  } else if (existing.clockHoursCompleted === undefined) {
    // Backward compatibility: add clockHoursCompleted to existing enrollments
    existing.clockHoursCompleted = 0;
  }
  
  await db.query('UPDATE mdtslms_classes SET studentIds=?, studentEnrollments=? WHERE id=?', [
    JSON.stringify(klass.studentIds),
    JSON.stringify(klass.studentEnrollments),
    classId
  ]);
  return klass;
}

async function removeStudent(classId, studentId) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  const before = Array.isArray(klass.studentIds) ? klass.studentIds : [];
  const after = before.filter(id => Number(id) !== Number(studentId));
  
  // Also remove from enrollments
  klass.studentEnrollments = (klass.studentEnrollments || []).filter(
    e => Number(e.studentId) !== Number(studentId)
  );
  
  await db.query('UPDATE mdtslms_classes SET studentIds=?, studentEnrollments=? WHERE id=?', [
    JSON.stringify(after),
    JSON.stringify(klass.studentEnrollments),
    classId
  ]);
  return true;
}

async function toggleStudentLock(classId, studentId) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  
  klass.studentEnrollments = klass.studentEnrollments || [];
  let enrollment = klass.studentEnrollments.find(e => Number(e.studentId) === Number(studentId));
  
  if (!enrollment) {
    // Create enrollment entry if it doesn't exist (for backwards compatibility)
    enrollment = {
      studentId: Number(studentId),
      enrolledAt: new Date().toISOString(),
      clockHoursCompleted: 0,
      locked: false
    };
    klass.studentEnrollments.push(enrollment);
  }
  
  // Toggle the locked status
  enrollment.locked = !enrollment.locked;
  
  await db.query('UPDATE mdtslms_classes SET studentEnrollments=? WHERE id=?', [
    JSON.stringify(klass.studentEnrollments),
    classId
  ]);
  
  return klass;
}

async function duplicateClass(id) {
  const original = await findClassById(id);
  if (!original) return null;
  const hasCreatedAt = await ensureClassCreatedAtColumn();
  const hasAssistantTeacherId = await ensureAssistantTeacherIdColumn();
  const ts = new Date().toISOString().slice(0, 19).replace('T', ' ');
  const columns = [
    'schoolYear',
    'cohort',
    'name',
    'shortName',
    'description',
    'teacherId',
    'weeks',
    'startDate',
    'endDate',
    'studentIds',
    'schedule',
    'lectures',
    'simulations',
    'assignments',
    'tests',
    'grades',
    'attendance',
    'checklist'
  ];
  const values = [
    original.schoolYear,
    original.cohort,
    `${original.name} (Copy)`,
    original.shortName,
    original.description,
    original.teacherId,
    original.weeks,
    original.startDate,
    original.endDate,
    JSON.stringify([]),
    JSON.stringify(original.schedule || []),
    JSON.stringify(original.lectures || []),
    JSON.stringify(original.simulations || []),
    JSON.stringify(original.assignments || []),
    JSON.stringify(original.tests || []),
    JSON.stringify([]),
    JSON.stringify([]),
    JSON.stringify(original.checklist || [])
  ];

  if (hasCreatedAt) {
    columns.push('createdAt');
    values.push(ts);
  }

  if (hasAssistantTeacherId) {
    columns.push('assistantTeacherId');
    values.push(original.assistantTeacherId || null);
  }

  const placeholders = columns.map(() => '?').join(',');
  const sql = `INSERT INTO mdtslms_classes (${columns.join(',')}) VALUES (${placeholders})`;
  const [result] = await db.query(sql, values);
  return findClassById(result.insertId);
}

async function renameClass(id, name) {
  await db.query('UPDATE mdtslms_classes SET name=? WHERE id=?', [name, id]);
  return findClassById(id);
}

// Update core class fields and schedule
async function updateClass(id, payload) {
  await ensureClassClockHoursColumn();
  await ensureAssistantTeacherIdColumn();
  await ensureLocationColumn();
  const {
    schoolYear,
    cohort,
    name,
    shortName,
    description,
    location,
    teacherId,
    assistantTeacherId,
    weeks,
    startDate,
    endDate,
    schedule,
    clockHours
  } = payload;
  await db.query(
    `UPDATE mdtslms_classes
     SET schoolYear=?, cohort=?, name=?, shortName=?, description=?, location=?, teacherId=?, assistantTeacherId=?, weeks=?, startDate=?, endDate=?, schedule=?, clockHours=?
     WHERE id=?`,
    [
      schoolYear,
      cohort,
      name,
      shortName,
      description,
      location || null,
      teacherId,
      assistantTeacherId || null,
      Number(weeks) || 0,
      startDate,
      endDate,
      JSON.stringify(schedule || []),
      Number(clockHours) || 0,
      id
    ]
  );
  return findClassById(id);
}

// Permanently delete a class
async function deleteClass(id) {
  await db.query('DELETE FROM mdtslms_classes WHERE id = ?', [id]);
  return true;
}

async function addTest(classId, test) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  klass.tests = klass.tests || [];
  const newTest = {
    id: (klass.tests.reduce((m, t) => Math.max(m, t.id), 0) || 0) + 1,
    title: test.title,
    timeLimit: test.timeLimit,
    dueDate: test.dueDate,
    official: test.official !== false // Default to true if not specified
  };
  klass.tests.push(newTest);
  await db.query('UPDATE mdtslms_classes SET tests=? WHERE id=?', [JSON.stringify(klass.tests), classId]);
  return newTest;
}

async function addAssignment(classId, assignment) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  klass.assignments = klass.assignments || [];
  assignment.id = (klass.assignments.reduce((m, a) => Math.max(m, a.id), 0) || 0) + 1;
  klass.assignments.push(assignment);
  await db.query('UPDATE mdtslms_classes SET assignments=? WHERE id=?', [JSON.stringify(klass.assignments), classId]);
  return assignment;
}


async function addLecture(classId, lecture) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  klass.lectures = klass.lectures || [];
  lecture.id = (klass.lectures.reduce((m, l) => Math.max(m, l.id), 0) || 0) + 1;
  klass.lectures.push(lecture);
  await db.query('UPDATE mdtslms_classes SET lectures=? WHERE id=?', [JSON.stringify(klass.lectures), classId]);
  return lecture;
}

async function addSimulation(classId, simulation) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  klass.simulations = klass.simulations || [];
  simulation.id = (klass.simulations.reduce((m, s) => Math.max(m, s.id), 0) || 0) + 1;
  klass.simulations.push(simulation);
  await db.query('UPDATE mdtslms_classes SET simulations=? WHERE id=?', [JSON.stringify(klass.simulations), classId]);
  return simulation;
}

async function addLink(classId, link) {
  try {
    const klass = await findClassById(classId);
    if (!klass) {
      console.error('Class not found:', classId);
      throw new Error('Class not found');
    }
    
    klass.links = klass.links || [];
    link.id = (klass.links.reduce((m, l) => Math.max(m, l.id), 0) || 0) + 1;
    klass.links.push(link);
    
    console.log('Saving links to class:', classId, 'Total links:', klass.links.length);
    
    await db.query('UPDATE mdtslms_classes SET links=? WHERE id=?', [JSON.stringify(klass.links), classId]);
    
    console.log('Links saved successfully');
    return link;
  } catch (error) {
    console.error('Error in addLink function:', error);
    throw error;
  }
}

async function updateChecklist(classId, checklist) {
  await db.query('UPDATE mdtslms_classes SET checklist=? WHERE id=?', [JSON.stringify(checklist || []), classId]);
  return true;
}

// Record an assignment submission from a student
async function addAssignmentSubmission(classId, assignmentId, submission) {
  // Ensure submissions column exists (compatible with MySQL versions without IF NOT EXISTS)
  try {
    const [cols] = await db.query('SHOW COLUMNS FROM mdtslms_classes LIKE "submissions"');
    if (!cols || !cols.length) {
      await db.query('ALTER TABLE mdtslms_classes ADD COLUMN submissions LONGTEXT NULL');
    }
  } catch (e) {
    // Best-effort; continue, and if update fails we’ll bubble up
  }
  const klass = await findClassById(classId);
  if (!klass) return null;
  klass.submissions = Array.isArray(klass.submissions) ? klass.submissions : [];
  const entry = {
    assignmentId: Number(assignmentId),
    studentId: Number(submission.studentId),
    file: submission.file || {},
    note: submission.note || '',
    submittedAt: new Date().toISOString()
  };
  klass.submissions.push(entry);
  try {
    await db.query('UPDATE mdtslms_classes SET submissions=? WHERE id=?', [JSON.stringify(klass.submissions), classId]);
  } catch (e) {
    // If column still missing, add and retry once
    if (e && e.code === 'ER_BAD_FIELD_ERROR') {
      try { await db.query('ALTER TABLE mdtslms_classes ADD COLUMN submissions LONGTEXT NULL'); } catch (_) {}
      await db.query('UPDATE mdtslms_classes SET submissions=? WHERE id=?', [JSON.stringify(klass.submissions), classId]);
    } else {
      throw e;
    }
  }
  return entry;
}

async function byTeacher(teacherId) {
  await ensureAssistantTeacherIdColumn();
  const [rows] = await db.query('SELECT * FROM mdtslms_classes WHERE teacherId = ? OR assistantTeacherId = ?', [teacherId, teacherId]);
  return rows.map(mapClass);
}


async function upsertItemGrade(classId, key, itemId, studentId, extra) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  klass.grades = klass.grades || [];
  const now = new Date().toISOString();
  const existing = klass.grades.find(g => g[key] === itemId && g.studentId === studentId);
  if (existing) {
    Object.assign(existing, extra, { gradedAt: now });
  } else {
    const entry = { classId, studentId, gradedAt: now, ...extra };
    entry[key] = itemId;
    klass.grades.push(entry);
  }
  await db.query('UPDATE mdtslms_classes SET grades=? WHERE id=?', [JSON.stringify(klass.grades), classId]);
  return true;
}

async function recordGrade(classId, testId, studentId, score) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  klass.grades = klass.grades || [];
  const now = new Date().toISOString();
  const existing = klass.grades.find(g => g.testId === testId && g.studentId === studentId);
  if (existing) {
    existing.score = score;
    existing.attempt = (existing.attempt || 0) + 1;
    existing.gradedAt = now;
  } else {
    klass.grades.push({ classId, studentId, testId, score, attempt: 1, gradedAt: now });
  }
  await db.query('UPDATE mdtslms_classes SET grades=? WHERE id=?', [JSON.stringify(klass.grades), classId]);
  return true;
}

// Enhanced attendance record with detailed entries per student
// attendanceData: { date, scheduledHours, timeIn, timeOut, entries: [{ studentId, hoursCompleted, present, tardy, reason }], instructorSignature, signedAt, takenBy, takenById }
async function recordDetailedAttendance(classId, attendanceData) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  klass.attendance = klass.attendance || [];
  klass.studentEnrollments = klass.studentEnrollments || [];
  
  const { date, scheduledHours, timeIn, timeOut, entries, instructorSignature, signedAt, takenBy, takenById } = attendanceData;
  
  // Find existing record for this date
  const existingIndex = klass.attendance.findIndex(a => a.date === date);
  
  // Build the attendance record - store entries directly with all data
  const attendanceRecord = {
    date,
    scheduledHours: parseFloat(scheduledHours) || 0,
    timeIn: timeIn || '',
    timeOut: timeOut || '',
    entries: (entries || []).map(e => ({
      studentId: Number(e.studentId),
      hoursCompleted: parseFloat(e.hoursCompleted) || 0,
      present: Boolean(e.present),
      tardy: Boolean(e.tardy),
      reason: e.reason || ''
    })),
    instructorSignature: instructorSignature || null,
    signedAt: signedAt || null,
    takenBy: takenBy || null,
    takenById: takenById || null
  };
  
  console.log('SAVE DEBUG:', {
    date: attendanceRecord.date,
    numEntries: attendanceRecord.entries.length,
    entries: attendanceRecord.entries.map(e => ({ id: e.studentId, present: e.present, hours: e.hoursCompleted }))
  });
  
  if (existingIndex >= 0) {
    klass.attendance[existingIndex] = attendanceRecord;
  } else {
    klass.attendance.push(attendanceRecord);
  }
  
  // Update clock hours for present students
  if (attendanceRecord.scheduledHours > 0) {
    attendanceRecord.entries.forEach(entry => {
      if (entry.present) {
        const enrollment = klass.studentEnrollments.find(e => Number(e.studentId) === entry.studentId);
        if (enrollment) {
          if (enrollment.clockHoursCompleted === undefined) {
            enrollment.clockHoursCompleted = 0;
          }
          // For simplicity, just set hours (could track per-date to prevent duplicates)
        }
      }
    });
  }
  
  await db.query('UPDATE mdtslms_classes SET attendance=?, studentEnrollments=? WHERE id=?', [
    JSON.stringify(klass.attendance),
    JSON.stringify(klass.studentEnrollments),
    classId
  ]);
  
  return true;
}

// Get detailed attendance record for a specific student
async function getStudentAttendanceRecord(classId, studentId) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  
  const studentIdNum = Number(studentId);
  const student = klass.studentEnrollments?.find(e => Number(e.studentId) === studentIdNum);
  
  const attendanceRecords = (klass.attendance || [])
    .map(record => {
      // Find this student's entry in the attendance record
      const entry = (record.entries || []).find(e => Number(e.studentId) === studentIdNum);
      
      // Check legacy present array
      const legacyPresent = (record.present || []).some(id => Number(id) === studentIdNum);
      
      // Only include this record if the student has an entry OR is in the legacy present array
      if (!entry && !legacyPresent) {
        return null; // Skip this record - student has no attendance data for this date
      }
      
      // Simple logic: if entry exists, use its data. If not, check legacy present array
      let isPresent = false;
      let hoursCompleted = 0;
      let isTardy = false;
      let reason = '';
      
      if (entry) {
        // New format: entry has present field directly
        isPresent = entry.present === true;
        hoursCompleted = isPresent ? (entry.hoursCompleted || record.scheduledHours || 0) : 0;
        isTardy = entry.tardy === true;
        reason = entry.reason || '';
      } else if (legacyPresent) {
        // Legacy format: check old present array
        isPresent = true;
        hoursCompleted = record.scheduledHours || 0;
      }
      
      return {
        date: record.date,
        scheduledHours: record.scheduledHours || 0,
        timeIn: record.timeIn || '',
        timeOut: record.timeOut || '',
        hoursCompleted: hoursCompleted,
        absent: !isPresent,
        tardy: isTardy,
        reason: reason,
        instructorSignature: record.instructorSignature || null,
        signedAt: record.signedAt || null
      };
    })
    .filter(record => record !== null) // Remove null entries
    .sort((a, b) => a.date.localeCompare(b.date));
  
  // Calculate totals
  const totalScheduledHours = attendanceRecords.reduce((sum, r) => sum + (r.scheduledHours || 0), 0);
  const totalCompletedHours = attendanceRecords.reduce((sum, r) => sum + (r.hoursCompleted || 0), 0);
  const percentageCompleted = totalScheduledHours > 0 ? ((totalCompletedHours / totalScheduledHours) * 100).toFixed(1) : 0;
  
  return {
    studentId: studentIdNum,
    enrollment: student,
    records: attendanceRecords,
    summary: {
      totalScheduledHours,
      totalCompletedHours,
      percentageCompleted,
      meetsRequirement: parseFloat(percentageCompleted) >= 80
    }
  };
}

// Delete a specific attendance entry for a student on a date
async function deleteStudentAttendanceEntry(classId, studentId, date) {
  const klass = await findClassById(classId);
  if (!klass) return { success: false, error: 'Class not found' };
  
  const studentIdNum = Number(studentId);
  
  console.log('DELETE DEBUG - Looking for date:', date);
  console.log('DELETE DEBUG - Available dates:', (klass.attendance || []).map(r => r.date));
  
  // Find the attendance record for this date
  const attendanceIndex = (klass.attendance || []).findIndex(record => record.date === date);
  
  console.log('DELETE DEBUG - Found at index:', attendanceIndex);
  
  if (attendanceIndex === -1) {
    return { success: false, error: 'Attendance record not found for this date' };
  }
  
  const attendanceRecord = klass.attendance[attendanceIndex];
  
  console.log('DELETE DEBUG - Before deletion, entries:', attendanceRecord.entries?.length);
  console.log('DELETE DEBUG - Entry student IDs:', attendanceRecord.entries?.map(e => e.studentId));
  console.log('DELETE DEBUG - Looking for studentId:', studentIdNum);
  
  // Remove the student's entry from this attendance record
  if (attendanceRecord.entries && Array.isArray(attendanceRecord.entries)) {
    const beforeCount = attendanceRecord.entries.length;
    attendanceRecord.entries = attendanceRecord.entries.filter(entry => Number(entry.studentId) !== studentIdNum);
    console.log('DELETE DEBUG - Entries removed:', beforeCount - attendanceRecord.entries.length);
    console.log('DELETE DEBUG - After deletion, entries:', attendanceRecord.entries.length);
  }
  
  // Also remove from legacy present array if it exists
  if (attendanceRecord.present && Array.isArray(attendanceRecord.present)) {
    attendanceRecord.present = attendanceRecord.present.filter(id => Number(id) !== studentIdNum);
  }
  
  // If this attendance record now has no entries at all, optionally remove it entirely
  // (commenting this out to keep the date record even if all students are removed)
  // if (!attendanceRecord.entries || attendanceRecord.entries.length === 0) {
  //   klass.attendance.splice(attendanceIndex, 1);
  // }
  
  // Save the updated attendance
  console.log('DELETE DEBUG - Saving updated attendance array');
  await db.query('UPDATE mdtslms_classes SET attendance=? WHERE id=?', [
    JSON.stringify(klass.attendance),
    classId
  ]);
  
  console.log('DELETE DEBUG - Save complete');
  
  return { success: true };
}

async function recordAttendance(classId, date, presentIds) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  klass.attendance = klass.attendance || [];
  klass.studentEnrollments = klass.studentEnrollments || [];
  
  // Calculate clock hours per session based on schedule for the given date
  let hoursPerSession = 0;
  if (klass.schedule && Array.isArray(klass.schedule)) {
    const dateObj = new Date(date);
    const dayNames = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
    const dayName = dayNames[dateObj.getDay()];
    
    const scheduleEntry = klass.schedule.find(s => s.day === dayName);
    if (scheduleEntry && scheduleEntry.start && scheduleEntry.end) {
      const startTime = scheduleEntry.start.split(':');
      const endTime = scheduleEntry.end.split(':');
      const startHours = parseInt(startTime[0]) + parseInt(startTime[1]) / 60;
      const endHours = parseInt(endTime[0]) + parseInt(endTime[1]) / 60;
      hoursPerSession = Math.max(0, endHours - startHours);
    }
  }
  
  // Update attendance record
  const existing = klass.attendance.find(a => a.date === date);
  const previousPresent = existing ? existing.present : [];
  
  if (existing) {
    existing.present = presentIds;
  } else {
    klass.attendance.push({ date, present: presentIds });
  }
  
  // Add clock hours to students who are marked present
  if (hoursPerSession > 0 && presentIds && presentIds.length > 0) {
    presentIds.forEach(studentId => {
      const enrollment = klass.studentEnrollments.find(e => Number(e.studentId) === Number(studentId));
      if (enrollment) {
        // Initialize if not exists
        if (enrollment.clockHoursCompleted === undefined) {
          enrollment.clockHoursCompleted = 0;
        }
        // Only add hours if this student wasn't previously marked present for this date
        if (!previousPresent.includes(studentId)) {
          enrollment.clockHoursCompleted += hoursPerSession;
        }
      }
    });
    
    // Update both attendance and studentEnrollments
    await db.query('UPDATE mdtslms_classes SET attendance=?, studentEnrollments=? WHERE id=?', [
      JSON.stringify(klass.attendance),
      JSON.stringify(klass.studentEnrollments),
      classId
    ]);
  } else {
    // Just update attendance if no hours to add
    await db.query('UPDATE mdtslms_classes SET attendance=? WHERE id=?', [JSON.stringify(klass.attendance), classId]);
  }
  
  return true;
}

async function byTeacher(teacherId) {
  await ensureAssistantTeacherIdColumn();
  const [rows] = await db.query('SELECT * FROM mdtslms_classes WHERE teacherId = ? OR assistantTeacherId = ?', [teacherId, teacherId]);
  return rows.map(mapClass);
}
async function upsertGrade(classId, testId, studentId, score) {
  return upsertItemGrade(classId, 'testId', testId, studentId, { score });
}

async function upsertAssignmentGrade(classId, assignmentId, studentId, score) {
  return upsertItemGrade(classId, 'assignmentId', assignmentId, studentId, { score });
}

async function upsertLabStatus(classId, labId, studentId, passed) {
  return upsertItemGrade(classId, 'labId', labId, studentId, { passed: !!passed });
}

// Update helpers
async function updateLecture(classId, lectureId, fields) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  const list = Array.isArray(klass.lectures) ? klass.lectures : [];
  const item = list.find(l => Number(l.id) === Number(lectureId));
  if (!item) return null;
  if (fields.title !== undefined) item.title = fields.title;
  if (fields.date !== undefined) item.date = fields.date;
  if (fields.dripDay !== undefined) item.dripDay = fields.dripDay ? Number(fields.dripDay) : null;
  await db.query('UPDATE mdtslms_classes SET lectures=? WHERE id=?', [JSON.stringify(list), classId]);
  return item;
}

async function updateSimulation(classId, simId, fields) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  const list = Array.isArray(klass.simulations) ? klass.simulations : [];
  const item = list.find(s => Number(s.id) === Number(simId));
  if (!item) return null;
  if (fields.title !== undefined) item.title = fields.title;
  if (fields.date !== undefined) item.date = fields.date;
  if (fields.dripDay !== undefined) item.dripDay = fields.dripDay ? Number(fields.dripDay) : null;
  await db.query('UPDATE mdtslms_classes SET simulations=? WHERE id=?', [JSON.stringify(list), classId]);
  return item;
}

async function updateAssignment(classId, assignmentId, fields) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  const list = Array.isArray(klass.assignments) ? klass.assignments : [];
  const item = list.find(a => Number(a.id) === Number(assignmentId));
  if (!item) return null;
  if (fields.title !== undefined) item.title = fields.title;
  if (fields.date !== undefined) item.date = fields.date;
  if (fields.dripDay !== undefined) item.dripDay = fields.dripDay ? Number(fields.dripDay) : null;
  await db.query('UPDATE mdtslms_classes SET assignments=? WHERE id=?', [JSON.stringify(list), classId]);
  return item;
}

async function updateTest(classId, testId, fields) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  const list = Array.isArray(klass.tests) ? klass.tests : [];
  const item = list.find(t => Number(t.id) === Number(testId));
  if (!item) return null;
  if (fields.title !== undefined) item.title = fields.title;
  if (fields.dueDate !== undefined) item.dueDate = fields.dueDate;
  if (fields.official !== undefined) item.official = fields.official;
  if (fields.dripDay !== undefined) item.dripDay = fields.dripDay ? Number(fields.dripDay) : null;
  await db.query('UPDATE mdtslms_classes SET tests=? WHERE id=?', [JSON.stringify(list), classId]);
  return item;
}

async function updateLink(classId, linkId, fields) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  const list = Array.isArray(klass.links) ? klass.links : [];
  const item = list.find(l => Number(l.id) === Number(linkId));
  if (!item) return null;
  if (fields.title !== undefined) item.title = fields.title;
  if (fields.url !== undefined) item.url = fields.url;
  if (fields.description !== undefined) item.description = fields.description;
  await db.query('UPDATE mdtslms_classes SET links=? WHERE id=?', [JSON.stringify(list), classId]);
  return item;
}

// Removal helpers
async function removeLecture(classId, lectureId) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  const list = Array.isArray(klass.lectures) ? klass.lectures : [];
  const next = list.filter(l => Number(l.id) !== Number(lectureId));
  await db.query('UPDATE mdtslms_classes SET lectures=? WHERE id=?', [JSON.stringify(next), classId]);
  return true;
}

async function removeSimulation(classId, simulationId) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  const list = Array.isArray(klass.simulations) ? klass.simulations : [];
  const next = list.filter(s => Number(s.id) !== Number(simulationId));
  await db.query('UPDATE mdtslms_classes SET simulations=? WHERE id=?', [JSON.stringify(next), classId]);
  return true;
}

async function removeAssignment(classId, assignmentId) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  const list = Array.isArray(klass.assignments) ? klass.assignments : [];
  const next = list.filter(a => Number(a.id) !== Number(assignmentId));
  await db.query('UPDATE mdtslms_classes SET assignments=? WHERE id=?', [JSON.stringify(next), classId]);
  return true;
}

async function removeTest(classId, testId) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  const list = Array.isArray(klass.tests) ? klass.tests : [];
  const next = list.filter(t => Number(t.id) !== Number(testId));
  await db.query('UPDATE mdtslms_classes SET tests=? WHERE id=?', [JSON.stringify(next), classId]);
  return true;
}

async function removeLink(classId, linkId) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  const list = Array.isArray(klass.links) ? klass.links : [];
  const next = list.filter(l => Number(l.id) !== Number(linkId));
  await db.query('UPDATE mdtslms_classes SET links=? WHERE id=?', [JSON.stringify(next), classId]);
  return true;
}

// Drip schedule utility: filter content items based on class start date
// Day 1 is the first day of the class (startDate), not individual enrollment date
function filterContentByDripSchedule(items, classStartDate) {
  if (!Array.isArray(items) || !items.length) return items;
  if (!classStartDate) return items;
  
  const startDate = new Date(classStartDate);
  const now = new Date();
  const daysSinceStart = Math.floor((now - startDate) / (1000 * 60 * 60 * 24));
  
  return items.filter(item => {
    // If no dripDay is set, show the item (backwards compatibility)
    if (item.dripDay === null || item.dripDay === undefined) return true;
    
    const dripDay = Number(item.dripDay);
    // dripDay 1 means day 1 (class start date), dripDay 2 means day 2 (next day), etc.
    return daysSinceStart >= (dripDay - 1);
  });
}

// Get days since class start date (Day 1 = class start date)
function getDaysSinceClassStart(classStartDate) {
  if (!classStartDate) return null;
  
  const startDate = new Date(classStartDate);
  const now = new Date();
  const daysSinceStart = Math.floor((now - startDate) / (1000 * 60 * 60 * 24)) + 1; // +1 so day 1 is start date
  return daysSinceStart > 0 ? daysSinceStart : null;
}

// Get student's completed clock hours for a class
function getStudentClockHours(studentId, studentEnrollments) {
  if (!Array.isArray(studentEnrollments) || !studentEnrollments.length) return 0;
  
  const enrollment = studentEnrollments.find(e => Number(e.studentId) === Number(studentId));
  if (!enrollment) return 0;
  
  return enrollment.clockHoursCompleted || 0;
}

// Update grades array for a class
async function updateGrades(classId, grades) {
  await db.query('UPDATE mdtslms_classes SET grades=? WHERE id=?', [JSON.stringify(grades), classId]);
}

// Update tests array for a class
async function updateTests(classId, tests) {
  await db.query('UPDATE mdtslms_classes SET tests=? WHERE id=?', [JSON.stringify(tests), classId]);
}

// Update student enrollments array for a class
async function updateStudentEnrollments(classId, enrollments) {
  await ensureStudentEnrollmentsColumn();
  await db.query('UPDATE mdtslms_classes SET studentEnrollments=? WHERE id=?', [JSON.stringify(enrollments), classId]);
}

// Update syllabus for a class
async function updateSyllabus(classId, syllabusPath, syllabusFileName) {
  await ensureSyllabusColumns();
  const now = new Date().toISOString().slice(0, 19).replace('T', ' ');
  await db.query(
    'UPDATE mdtslms_classes SET syllabusPath=?, syllabusFileName=?, syllabusUploadedAt=? WHERE id=?',
    [syllabusPath, syllabusFileName, now, classId]
  );
}

// Remove syllabus from a class
async function removeSyllabus(classId) {
  await ensureSyllabusColumns();
  await db.query(
    'UPDATE mdtslms_classes SET syllabusPath=NULL, syllabusFileName=NULL, syllabusUploadedAt=NULL WHERE id=?',
    [classId]
  );
}

// Mark a lecture as completed for a student
async function markLectureCompleted(classId, lectureId, studentId) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  
  klass.lectures = klass.lectures || [];
  const lecture = klass.lectures.find(l => l.id === lectureId);
  if (!lecture) return null;
  
  lecture.completions = lecture.completions || [];
  const existing = lecture.completions.find(c => c.studentId === studentId);
  
  if (existing) {
    existing.completed = true;
    existing.completedAt = new Date().toISOString();
  } else {
    lecture.completions.push({
      studentId: studentId,
      completed: true,
      completedAt: new Date().toISOString()
    });
  }
  
  await db.query('UPDATE mdtslms_classes SET lectures=? WHERE id=?', [JSON.stringify(klass.lectures), classId]);
  return lecture;
}

// Mark a simulation as completed for a student
async function markSimulationCompleted(classId, simulationId, studentId) {
  const klass = await findClassById(classId);
  if (!klass) return null;
  
  klass.simulations = klass.simulations || [];
  const simulation = klass.simulations.find(s => s.id === simulationId);
  if (!simulation) return null;
  
  simulation.completions = simulation.completions || [];
  const existing = simulation.completions.find(c => c.studentId === studentId);
  
  if (existing) {
    existing.completed = true;
    existing.completedAt = new Date().toISOString();
  } else {
    simulation.completions.push({
      studentId: studentId,
      completed: true,
      completedAt: new Date().toISOString()
    });
  }
  
  await db.query('UPDATE mdtslms_classes SET simulations=? WHERE id=?', [JSON.stringify(klass.simulations), classId]);
  return simulation;
}

module.exports = {
  getAllClasses,
  findClassById,
  createClass,
  addStudent,
  addAssignmentSubmission,
  addTest,
  addLecture,
  addSimulation,
  addAssignment,
  addLink,
  updateLecture,
  updateSimulation,
  updateAssignment,
  updateTest,
  updateLink,
  removeStudent,
  toggleStudentLock,
  removeLecture,
  removeSimulation,
  removeAssignment,
  removeTest,
  removeLink,

  recordGrade,
  byTeacher,
  recordAttendance,
  recordDetailedAttendance,
  getStudentAttendanceRecord,
  deleteStudentAttendanceEntry,
  upsertGrade,
  upsertAssignmentGrade,
  upsertLabStatus,
  duplicateClass,
  updateChecklist,
  renameClass,
  updateClass,
  deleteClass,
  updateGrades,
  updateTests,
  updateStudentEnrollments,
  updateSyllabus,
  removeSyllabus,
  ensureSyllabusColumns,
  markLectureCompleted,
  markSimulationCompleted,
  
  // Drip schedule utilities
  filterContentByDripSchedule,
  getDaysSinceClassStart,
  
  // Clock hours tracking
  getStudentClockHours
};
