const fs = require('fs');
const path = require('path');

const SETTINGS_PATH = path.join(__dirname, '..', 'data', 'careerAdvisorSettings.json');

/**
 * Default settings for career advisor interview availability
 */
const DEFAULT_SETTINGS = {
  availableTimes: [
    '9:00 AM',
    '10:00 AM',
    '11:00 AM',
    '1:00 PM',
    '2:00 PM',
    '3:00 PM',
    '4:00 PM'
  ],
  availableDays: [1, 2, 3, 4, 5], // Monday-Friday (0=Sunday, 6=Saturday)
  daysAhead: 40, // How many days in advance to show
  specificDates: [], // Specific dates to include (YYYY-MM-DD)
  excludedDates: [], // Specific dates to exclude (YYYY-MM-DD)
  datePattern: 'biweekly', // 'biweekly' (15th/30th), 'weekly', 'all', 'specific'
  updatedAt: new Date().toISOString()
};

/**
 * Load settings from JSON file
 */
function loadSettings() {
  try {
    if (fs.existsSync(SETTINGS_PATH)) {
      const data = fs.readFileSync(SETTINGS_PATH, 'utf8');
      const settings = JSON.parse(data);
      // Merge with defaults to ensure all properties exist
      return { ...DEFAULT_SETTINGS, ...settings };
    }
  } catch (err) {
    console.error('Failed to load career advisor settings:', err.message);
  }
  return DEFAULT_SETTINGS;
}

/**
 * Save settings to JSON file
 */
function saveSettings(settings) {
  try {
    const dataToSave = {
      ...settings,
      updatedAt: new Date().toISOString()
    };
    fs.writeFileSync(SETTINGS_PATH, JSON.stringify(dataToSave, null, 2), 'utf8');
    return true;
  } catch (err) {
    console.error('Failed to save career advisor settings:', err.message);
    return false;
  }
}

/**
 * Get current settings
 */
function getSettings() {
  return loadSettings();
}

/**
 * Update settings
 */
function updateSettings(updates) {
  const current = loadSettings();
  const newSettings = {
    ...current,
    ...updates
  };
  saveSettings(newSettings);
  return newSettings;
}

/**
 * Reset to default settings
 */
function resetToDefaults() {
  saveSettings(DEFAULT_SETTINGS);
  return DEFAULT_SETTINGS;
}

/**
 * Get available dates based on settings
 */
function getAvailableDates() {
  const settings = loadSettings();
  const dates = [];
  const today = new Date();
  today.setHours(0, 0, 0, 0);
  const maxDate = new Date(today);
  maxDate.setDate(maxDate.getDate() + settings.daysAhead);

  if (settings.datePattern === 'specific') {
    // Use only specific dates
    return settings.specificDates
      .map(d => new Date(d + 'T12:00:00'))
      .filter(d => d >= today && d <= maxDate && !settings.excludedDates.includes(d.toISOString().split('T')[0]))
      .map(d => d.toISOString().split('T')[0])
      .sort();
  }

  if (settings.datePattern === 'biweekly') {
    // Original logic: 15th and last day of month
    for (let i = 0; i < 3; i++) {
      const targetDate = new Date(today.getFullYear(), today.getMonth() + i, 1);
      const year = targetDate.getFullYear();
      const month = targetDate.getMonth();
      
      // Check 15th
      let midMonth = new Date(year, month, 15);
      let midDay = midMonth.getDay();
      
      // Adjust to available days if needed
      if (!settings.availableDays.includes(midDay)) {
        // Find nearest available day
        for (let offset = -2; offset <= 2; offset++) {
          const testDate = new Date(year, month, 15 + offset);
          if (settings.availableDays.includes(testDate.getDay())) {
            midMonth = testDate;
            break;
          }
        }
      }
      
      if (midMonth >= today && midMonth <= maxDate && !settings.excludedDates.includes(midMonth.toISOString().split('T')[0])) {
        dates.push(midMonth.toISOString().split('T')[0]);
      }
      
      // Check last day of month
      let endMonth = new Date(year, month + 1, 0);
      let endDay = endMonth.getDay();
      
      // Adjust to available days if needed
      if (!settings.availableDays.includes(endDay)) {
        for (let offset = -2; offset <= 0; offset++) {
          const testDate = new Date(year, month, endMonth.getDate() + offset);
          if (settings.availableDays.includes(testDate.getDay())) {
            endMonth = testDate;
            break;
          }
        }
      }
      
      if (endMonth >= today && endMonth <= maxDate && !settings.excludedDates.includes(endMonth.toISOString().split('T')[0])) {
        dates.push(endMonth.toISOString().split('T')[0]);
      }
    }
  } else if (settings.datePattern === 'weekly') {
    // Every week on specified days
    for (let i = 0; i <= settings.daysAhead; i++) {
      const date = new Date(today);
      date.setDate(date.getDate() + i);
      const dayOfWeek = date.getDay();
      
      if (settings.availableDays.includes(dayOfWeek) && !settings.excludedDates.includes(date.toISOString().split('T')[0])) {
        dates.push(date.toISOString().split('T')[0]);
      }
    }
  } else if (settings.datePattern === 'all') {
    // Every day within days ahead
    for (let i = 0; i <= settings.daysAhead; i++) {
      const date = new Date(today);
      date.setDate(date.getDate() + i);
      const dayOfWeek = date.getDay();
      
      if (settings.availableDays.includes(dayOfWeek) && !settings.excludedDates.includes(date.toISOString().split('T')[0])) {
        dates.push(date.toISOString().split('T')[0]);
      }
    }
  }

  // Add any specific dates that aren't already included
  settings.specificDates.forEach(dateStr => {
    const date = new Date(dateStr + 'T12:00:00');
    if (date >= today && date <= maxDate && !dates.includes(dateStr) && !settings.excludedDates.includes(dateStr)) {
      dates.push(dateStr);
    }
  });

  return [...new Set(dates)].sort();
}

module.exports = {
  getSettings,
  updateSettings,
  resetToDefaults,
  getAvailableDates,
  DEFAULT_SETTINGS
};
