-- ============================================================================
-- MySQL Database Trigger for Auto Alumni Transition
-- ============================================================================
-- 
-- This approach uses a MySQL EVENT (scheduled task) instead of a cron job
-- or application-level middleware. MySQL will automatically check and update
-- student statuses periodically.
--
-- ADVANTAGES:
-- - Database-level automation (independent of app restarts)
-- - No cron job configuration needed
-- - Runs even if Node.js app is down
-- - Efficient (runs in database, not application layer)
--
-- DISADVANTAGES:
-- - Requires MySQL Event Scheduler to be enabled
-- - Harder to debug/monitor than application code
-- - Less flexible than middleware approach
-- - May not log transitions to application logs
--
-- SETUP INSTRUCTIONS:
-- 1. Enable MySQL Event Scheduler (if not already enabled):
--    SET GLOBAL event_scheduler = ON;
--    Add to my.cnf: event_scheduler=ON
--
-- 2. Run this SQL script to create the event
--
-- 3. Verify event is running:
--    SHOW EVENTS;
--    SELECT * FROM information_schema.events WHERE event_name = 'auto_transition_to_alumni';
-- ============================================================================

-- Drop existing event if it exists
DROP EVENT IF EXISTS auto_transition_to_alumni;

-- Create the event to run daily at 2:00 AM
DELIMITER $$

CREATE EVENT auto_transition_to_alumni
ON SCHEDULE EVERY 1 DAY
STARTS (TIMESTAMP(CURRENT_DATE) + INTERVAL 1 DAY + INTERVAL 2 HOUR)
DO
BEGIN
  DECLARE done INT DEFAULT FALSE;
  DECLARE v_user_id INT;
  DECLARE v_profile JSON;
  DECLARE v_last_class_end DATE;
  DECLARE v_grace_period_end DATE;
  DECLARE v_now DATETIME;
  DECLARE transition_count INT DEFAULT 0;
  
  DECLARE student_cursor CURSOR FOR
    SELECT id, profile
    FROM users
    WHERE role = 'student' AND status = 'approved';
  
  DECLARE CONTINUE HANDLER FOR NOT FOUND SET done = TRUE;
  
  SET v_now = NOW();
  
  OPEN student_cursor;
  
  student_loop: LOOP
    FETCH student_cursor INTO v_user_id, v_profile;
    
    IF done THEN
      LEAVE student_loop;
    END IF;
    
    -- Find the most recent class end date for this student
    SELECT MAX(c.endDate) INTO v_last_class_end
    FROM classes c
    WHERE FIND_IN_SET(v_user_id, c.enrolled) > 0;
    
    -- Check if student has no active classes and grace period has passed
    IF v_last_class_end IS NOT NULL THEN
      SET v_grace_period_end = DATE_ADD(v_last_class_end, INTERVAL 7 DAY);
      
      -- If grace period has passed and no current/future classes
      IF v_grace_period_end < CURDATE() THEN
        -- Check if student has any current or future classes
        IF NOT EXISTS (
          SELECT 1 FROM classes c
          WHERE FIND_IN_SET(v_user_id, c.enrolled) > 0
            AND (c.endDate IS NULL OR c.endDate >= CURDATE())
        ) THEN
          -- Update student status to alumni
          UPDATE users
          SET 
            status = 'alumni',
            profile = JSON_SET(
              COALESCE(profile, '{}'),
              '$.becameAlumniAt', DATE_FORMAT(v_now, '%Y-%m-%dT%H:%i:%s.000Z'),
              '$.autoTransitionedToAlumni', TRUE,
              '$.lastClassEndDate', DATE_FORMAT(v_last_class_end, '%Y-%m-%dT%H:%i:%s.000Z')
            )
          WHERE id = v_user_id;
          
          SET transition_count = transition_count + 1;
        END IF;
      END IF;
    END IF;
  END LOOP;
  
  CLOSE student_cursor;
  
  -- Optional: Log the transition count (requires a log table)
  -- INSERT INTO system_logs (event_type, message, created_at)
  -- VALUES ('alumni_transition', CONCAT('Transitioned ', transition_count, ' students to alumni'), v_now);
  
END$$

DELIMITER ;

-- ============================================================================
-- ALTERNATIVE: Run Every Hour (More Responsive)
-- ============================================================================
-- If you want more frequent checks, use this version instead:
/*
DROP EVENT IF EXISTS auto_transition_to_alumni;

CREATE EVENT auto_transition_to_alumni
ON SCHEDULE EVERY 1 HOUR
STARTS CURRENT_TIMESTAMP
DO
BEGIN
  -- Same logic as above
END;
*/

-- ============================================================================
-- VERIFICATION QUERIES
-- ============================================================================

-- Check if event scheduler is enabled
SHOW VARIABLES LIKE 'event_scheduler';

-- View all events
SHOW EVENTS;

-- View event details
SELECT 
  event_name,
  event_definition,
  interval_value,
  interval_field,
  status,
  last_executed,
  starts,
  ends
FROM information_schema.events
WHERE event_schema = DATABASE()
  AND event_name = 'auto_transition_to_alumni';

-- ============================================================================
-- MANAGEMENT COMMANDS
-- ============================================================================

-- Disable the event (pause auto-transitions)
-- ALTER EVENT auto_transition_to_alumni DISABLE;

-- Enable the event
-- ALTER EVENT auto_transition_to_alumni ENABLE;

-- Delete the event completely
-- DROP EVENT auto_transition_to_alumni;

-- Manually trigger transition logic (for testing)
/*
CALL check_alumni_transitions();  -- If you convert the event logic to a stored procedure
*/
