const db = require('./db');

// Ensure RSVP table exists
async function init() {
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_event_rsvps (
    id INT AUTO_INCREMENT PRIMARY KEY,
    eventId INT,
    eventName VARCHAR(255),
    fullName VARCHAR(255),
    email VARCHAR(255),
    phone VARCHAR(50),
    address VARCHAR(255),
    city VARCHAR(100),
    state VARCHAR(50),
    zip VARCHAR(20),
    branch VARCHAR(100),
    program VARCHAR(100),
    createdAt DATETIME DEFAULT CURRENT_TIMESTAMP
  )`);

  const [columns] = await db.query('SHOW COLUMNS FROM mdtslms_event_rsvps');
  const hasColumn = (name) => columns.some(col => col.Field === name);

  if (!hasColumn('eventName')) {
    await db.query('ALTER TABLE mdtslms_event_rsvps ADD COLUMN eventName VARCHAR(255) AFTER eventId');
  }

  if (!hasColumn('createdAt')) {
    await db.query('ALTER TABLE mdtslms_event_rsvps ADD COLUMN createdAt DATETIME DEFAULT CURRENT_TIMESTAMP AFTER program');
  }

  await db.query('UPDATE mdtslms_event_rsvps SET createdAt = NOW() WHERE createdAt IS NULL');
  await db.query(`
    UPDATE mdtslms_event_rsvps r
    LEFT JOIN mdtslms_events e ON r.eventId = e.id
    SET r.eventName = COALESCE(r.eventName, e.name)
    WHERE r.eventName IS NULL
  `);
}
init().catch(console.error);

async function createRSVP(data) {
  const { eventId, eventName = null, fullName, email, phone, address, city, state, zip, branch, program } = data;
  await db.query(
    'INSERT INTO mdtslms_event_rsvps (eventId, eventName, fullName, email, phone, address, city, state, zip, branch, program) VALUES (?,?,?,?,?,?,?,?,?,?,?)',
    [eventId, eventName, fullName, email, phone, address, city, state, zip, branch, program]
  );
}

async function getAllRSVPs() {
  const [rows] = await db.query(`
    SELECT
      r.id,
      r.eventId,
      COALESCE(r.eventName, e.name) AS eventName,
      COALESCE(r.eventName, e.name) AS event,
      r.fullName,
      r.email,
      r.phone,
      r.address,
      r.city,
      r.state,
      r.zip,
      r.branch,
      r.program,
      r.createdAt
    FROM mdtslms_event_rsvps r
    LEFT JOIN mdtslms_events e ON r.eventId = e.id
    ORDER BY r.createdAt DESC, r.id DESC
  `);
  return rows;
}

async function getById(id) {
  const [rows] = await db.query(
    `SELECT
      r.id,
      r.eventId,
      COALESCE(r.eventName, e.name) AS eventName,
      COALESCE(r.eventName, e.name) AS event,
      r.fullName,
      r.email,
      r.phone,
      r.address,
      r.city,
      r.state,
      r.zip,
      r.branch,
      r.program,
      r.createdAt
    FROM mdtslms_event_rsvps r
    LEFT JOIN mdtslms_events e ON r.eventId = e.id
    WHERE r.id = ?`,
    [id]
  );
  return rows[0];
}

async function deleteById(id) {
  const [res] = await db.query('DELETE FROM mdtslms_event_rsvps WHERE id=?', [id]);
  return res.affectedRows > 0;
}

module.exports = {
  createRSVP,
  getAllRSVPs,
  getById,
  deleteById
};
