const db = require('./db');

// Ensure events table exists
async function init() {
  await db.query(`CREATE TABLE IF NOT EXISTS mdtslms_events (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    eventDate DATE NOT NULL,
    description TEXT,
    attachment VARCHAR(255),
    notes LONGTEXT NULL,
    checklist LONGTEXT NULL
  )`);
  // ensure attachment column exists for older installs
  await db
    .query('ALTER TABLE mdtslms_events ADD COLUMN attachment VARCHAR(255) NULL')
    .catch(() => {});
  await db.query('ALTER TABLE mdtslms_events ADD COLUMN notes LONGTEXT NULL').catch(()=>{});
  await db.query('ALTER TABLE mdtslms_events ADD COLUMN checklist LONGTEXT NULL').catch(()=>{});
  await db.query('ALTER TABLE mdtslms_events ADD COLUMN autoShowPdf BOOLEAN DEFAULT FALSE').catch(()=>{});
}
init().catch(console.error);

async function createEvent({ name, eventDate, description, attachment, autoShowPdf }) {
  const [result] = await db.query(
    'INSERT INTO mdtslms_events (name, eventDate, description, attachment, autoShowPdf) VALUES (?,?,?,?,?)',
    [name, eventDate, description, attachment, autoShowPdf || false]
  );
  return { id: result.insertId, name, eventDate, description, attachment, autoShowPdf };
}

async function getAllEvents() {
  const [rows] = await db.query('SELECT * FROM mdtslms_events ORDER BY eventDate ASC');
  return rows.map(r => {
    try { if (typeof r.checklist === 'string') r.checklist = JSON.parse(r.checklist); } catch(_) { r.checklist = []; }
    return r;
  });
}

async function getEventById(id) {
  const [rows] = await db.query('SELECT * FROM mdtslms_events WHERE id = ?', [id]);
  return rows[0];
}

async function updateEvent(id, { name, eventDate, description, attachment, autoShowPdf }) {
  await db.query(
    'UPDATE mdtslms_events SET name=?, eventDate=?, description=?, attachment=?, autoShowPdf=? WHERE id=?',
    [name, eventDate, description, attachment, autoShowPdf || false, id]
  );
  return getEventById(id);
}

async function deleteEvent(id) {
  await db.query('DELETE FROM mdtslms_events WHERE id = ?', [id]);
  return true;
}

module.exports = {
  createEvent,
  getAllEvents,
  getEventById,
  updateEvent,
  deleteEvent
};
