const fs = require('fs');
const path = require('path');

const DATA_PATH = path.join(__dirname, '..', 'data', 'careerAdvisorInterviews.json');

/**
 * Load career advisor interviews from JSON file
 */
function loadInterviews() {
  try {
    if (fs.existsSync(DATA_PATH)) {
      const data = fs.readFileSync(DATA_PATH, 'utf8');
      return JSON.parse(data);
    }
  } catch (err) {
    console.error('Failed to load career advisor interviews:', err.message);
  }
  return [];
}

/**
 * Save career advisor interviews to JSON file
 */
function saveInterviews(interviews) {
  try {
    fs.writeFileSync(DATA_PATH, JSON.stringify(interviews, null, 2), 'utf8');
    return true;
  } catch (err) {
    console.error('Failed to save career advisor interviews:', err.message);
    return false;
  }
}

/**
 * Get all career advisor interviews
 */
function getAll() {
  return loadInterviews();
}

/**
 * Get interview by ID
 */
function getById(id) {
  const interviews = loadInterviews();
  return interviews.find(i => i.id === id);
}

/**
 * Get interviews for a specific student
 */
function getByStudentId(studentId) {
  const interviews = loadInterviews();
  return interviews.filter(i => i.studentId === studentId);
}

/**
 * Get interviews scheduled for a specific date (YYYY-MM-DD)
 */
function getByDate(date) {
  const interviews = loadInterviews();
  return interviews.filter(i => i.date === date);
}

/**
 * Create a new career advisor interview
 */
function create({ studentId, studentName, studentEmail, studentPhone, date, time, interestArea, summary, resumePath }) {
  const interviews = loadInterviews();
  
  const newInterview = {
    id: Date.now(),
    studentId,
    studentName,
    studentEmail,
    studentPhone,
    date,
    time,
    interestArea: interestArea || '',
    summary: summary || '',
    resumePath: resumePath || null,
    status: 'pending', // pending, confirmed, completed, cancelled
    confirmationSent: false,
    meetingLink: null,
    advisorName: null,
    notes: '',
    createdAt: new Date().toISOString(),
    updatedAt: new Date().toISOString()
  };
  
  interviews.push(newInterview);
  saveInterviews(interviews);
  
  return newInterview;
}

/**
 * Update an existing interview
 */
function update(id, updates) {
  const interviews = loadInterviews();
  const index = interviews.findIndex(i => i.id === id);
  
  if (index === -1) {
    return null;
  }
  
  interviews[index] = {
    ...interviews[index],
    ...updates,
    updatedAt: new Date().toISOString()
  };
  
  saveInterviews(interviews);
  return interviews[index];
}

/**
 * Send confirmation email for an interview
 */
function markConfirmationSent(id, advisorName, meetingLink) {
  return update(id, {
    confirmationSent: true,
    advisorName: advisorName || 'Career Advisor',
    meetingLink: meetingLink || null,
    status: 'confirmed'
  });
}

/**
 * Delete an interview
 */
function deleteInterview(id) {
  const interviews = loadInterviews();
  const filtered = interviews.filter(i => i.id !== id);
  
  if (filtered.length === interviews.length) {
    return false; // No interview found with that ID
  }
  
  saveInterviews(filtered);
  return true;
}

/**
 * Get upcoming interviews (from today forward)
 */
function getUpcoming() {
  const interviews = loadInterviews();
  const today = new Date().toISOString().split('T')[0];
  
  return interviews
    .filter(i => i.date >= today && i.status !== 'cancelled')
    .sort((a, b) => {
      const dateCompare = a.date.localeCompare(b.date);
      if (dateCompare !== 0) return dateCompare;
      return a.time.localeCompare(b.time);
    });
}

/**
 * Get interviews for calendar display (with date range)
 */
function getForCalendar(startDate, endDate) {
  const interviews = loadInterviews();
  
  return interviews.filter(i => {
    return i.date >= startDate && i.date <= endDate && i.status !== 'cancelled';
  });
}

/**
 * Check if a specific time slot is available
 */
function isTimeSlotAvailable(date, time) {
  const interviews = loadInterviews();
  return !interviews.some(i => 
    i.date === date && 
    i.time === time && 
    i.status !== 'cancelled'
  );
}

/**
 * Validate interview date is on allowed days (15th/30th or nearest weekday)
 */
function isValidInterviewDate(dateString) {
  const date = new Date(dateString);
  const day = date.getDate();
  const dayOfWeek = date.getDay(); // 0=Sunday, 6=Saturday
  
  // Must be a weekday
  if (dayOfWeek === 0 || dayOfWeek === 6) {
    return false;
  }
  
  // Must be on or near the 15th or 30th (allowing 13-16 and 28-31)
  const isNearMidMonth = day >= 13 && day <= 16;
  const isNearEndMonth = day >= 28 && day <= 31;
  
  return isNearMidMonth || isNearEndMonth;
}

/**
 * Get next available interview dates (returns array of valid dates)
 * Limited to 40 days out from today
 */
function getNextAvailableDates(daysAhead = 40) {
  const dates = [];
  const today = new Date();
  today.setHours(0, 0, 0, 0);
  const maxDate = new Date(today);
  maxDate.setDate(maxDate.getDate() + daysAhead);
  
  // Check dates for the next 3 months (but filter to daysAhead)
  for (let i = 0; i < 3; i++) {
    const targetDate = new Date(today.getFullYear(), today.getMonth() + i, 1);
    const year = targetDate.getFullYear();
    const month = targetDate.getMonth();
    
    // Check 15th
    let midMonth = new Date(year, month, 15);
    let midDay = midMonth.getDay();
    
    if (midDay === 0) { // Sunday -> Friday
      midMonth = new Date(year, month, 13);
    } else if (midDay === 6) { // Saturday -> Friday
      midMonth = new Date(year, month, 14);
    }
    
    if (midMonth >= today && midMonth <= maxDate) {
      dates.push(midMonth.toISOString().split('T')[0]);
    }
    
    // Check last day of month
    let endMonth = new Date(year, month + 1, 0); // Last day of month
    let endDay = endMonth.getDay();
    
    if (endDay === 0) { // Sunday -> Friday
      endMonth = new Date(year, month, endMonth.getDate() - 2);
    } else if (endDay === 6) { // Saturday -> Friday
      endMonth = new Date(year, month, endMonth.getDate() - 1);
    }
    
    if (endMonth >= today && endMonth <= maxDate) {
      dates.push(endMonth.toISOString().split('T')[0]);
    }
  }
  
  return dates.filter(d => {
    const dateObj = new Date(d + 'T12:00:00');
    return dateObj >= today && dateObj <= maxDate;
  });
}

module.exports = {
  getAll,
  getById,
  getByStudentId,
  getByDate,
  create,
  update,
  markConfirmationSent,
  deleteInterview,
  getUpcoming,
  getForCalendar,
  isTimeSlotAvailable,
  isValidInterviewDate,
  getNextAvailableDates
};
