require('dotenv').config();
const express = require('express');
const path = require('path');
const session = require('express-session');
const bodyParser = require('body-parser');

const authRoutes = require('./routes/auth');
const adminRoutes = require('./routes/admin');
const studentRoutes = require('./routes/student');
const teacherRoutes = require('./routes/teacher');
const paymentRoutes = require('./routes/payment');
const preRegRoutes = require('./routes/preReg');
const eventRoutes = require('./routes/events');
const accountRoutes = require('./routes/account');
const interviewRoutes = require('./routes/interviews');
const documentSigningRoutes = require('./routes/documentSigning');
const dripCampaign = require('./utils/dripCampaign');



const app = express();

// Load branding configuration and expose to views
const branding = require('./branding.json');
app.locals.branding = branding;
app.use((req, res, next) => {
  res.locals.branding = branding;
  next();
});

app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));
app.use(express.static(path.join(__dirname, 'public')));
app.use('/uploads', express.static(path.join(__dirname, 'uploads')));
app.use(session({
  secret: 'dev-secret-change-me',
  resave: false,
  saveUninitialized: false,
   cookie: { maxAge: 3 * 60 * 60 * 1000, sameSite: 'lax', httpOnly: true }

}));
app.use(bodyParser.urlencoded({ extended: true }));
app.use(bodyParser.json());

// Track page views and time spent for logged-in users
app.use((req, res, next) => {
  if (req.session && req.session.user) {
    req.session.pageViews = Array.isArray(req.session.pageViews) ? req.session.pageViews : [];
    req.session.pageViews.push(req.path);
    const now = Date.now();
    if (req.session.lastAction) {
      req.session.timeSpent = (req.session.timeSpent || 0) + (now - req.session.lastAction);
    }
    req.session.lastAction = now;
  }
  next();
});
app.use(express.static(path.join(__dirname, 'public')));
const uploadsPath = path.join(__dirname, 'uploads');
app.use('/uploads', (req, res, next) => {
    return express.static(uploadsPath)(req, res, next);
});

const docsPath = path.join(__dirname, 'docs');
app.use('/docs', (req, res, next) => {
  if (req.path.startsWith('/stxd') && (!req.session || req.session.role !== 'admin')) {
    return res.status(403).send('Unable to access area');
  }
  return express.static(docsPath)(req, res, next);
});
//sfw

// Expose simulations directory statically
const simsPath = path.join(__dirname, 'sims');
app.use('/sims', express.static(simsPath));

app.get('/', (req, res) => res.redirect('/dashboard'));  

app.get('/dashboard', (req, res) => {
  if (!req.session || !req.session.role) return res.redirect('/login');
  const role = req.session.role;
  if (role === 'admin')   return res.redirect('/admin');
  if (role === 'teacher') return res.redirect('/teacher');
  if (role === 'student') return res.redirect('/student');
  return res.redirect('/login');
});

app.use(authRoutes);
app.use(preRegRoutes);

app.use('/admin', adminRoutes);
app.use(paymentRoutes);
app.use(eventRoutes);
app.use(interviewRoutes);
app.use('/document-signing', documentSigningRoutes);


app.use('/student', studentRoutes);
app.use('/teacher', teacherRoutes);
app.use(accountRoutes);

dripCampaign.init();

const PORT = process.env.PORT || 3012;
app.listen(PORT, () => console.log(`School LMS running on http://localhost:${PORT}`));
