# Email Link & Authentication Fix

## Issues Fixed

### 1. Wrong Port in Email Links ✅
**Problem**: Email links were going to `localhost:3000` instead of `localhost:3012`

**Solution**: 
- Added `BASE_URL=http://localhost:3012` to `.env` file
- Updated all default fallbacks from `3000` to `3012` in `utils/documentSigning.js`

### 2. Authentication Error ✅
**Problem**: Clicking email link returned `{"error":"Not authenticated"}`

**Root Causes**:
- `requireAuth` middleware was checking `req.session.userId` but app uses `req.session.user.id`
- Returning JSON error instead of redirecting to login page
- Not preserving the return URL after login

**Solution**:
Updated `requireAuth` middleware to:
```javascript
function requireAuth(req, res, next) {
  if (!req.session || !req.session.user || !req.session.user.id) {
    // Store the original URL to redirect back after login
    req.session.returnTo = req.originalUrl;
    return res.redirect('/login');
  }
  next();
}
```

---

## How It Works Now

### Email Link Flow

1. **User receives email** with link like:
   ```
   http://localhost:3012/document-signing/sign/123/456
   ```

2. **User clicks link**:
   - If **logged in** → Goes directly to signing page ✅
   - If **not logged in** → Redirected to `/login` page 🔐

3. **After login**:
   - System redirects back to the original signing URL
   - User can now sign the document

### What Changed

**Before:**
```
Click email link → 401 JSON error → User stuck
```

**After:**
```
Click email link → Redirect to login → Login → Back to signing page ✅
```

---

## Testing

1. **Log out** from the application
2. **Click** the signing link from an email (or visit manually)
3. You should be **redirected to login page**
4. **Log in** as the user who needs to sign
5. You should be **redirected back** to the signing page
6. **Sign the document** ✅

---

## Configuration Summary

**`.env` file now includes:**
```env
BASE_URL=http://localhost:3012

SMTP_HOST=mail.mdts-apps.com
SMTP_PORT=587
SMTP_USER=noreply@mdts-apps.com
SMTP_PASS=c@r,5ysPI@&s
SMTP_FROM=noreply@mdts-apps.com
```

**For Production:**
Update `BASE_URL` to your actual domain:
```env
BASE_URL=https://yourdomain.com
```

---

## Server Status

✅ **Running on: http://localhost:3012**
✅ **Email links**: Will use correct port (3012)
✅ **Authentication**: Redirects to login instead of error
✅ **Return URL**: Preserved after login

---

## Notes

- The `req.session.returnTo` stores the original URL
- After successful login, your login route should check for this and redirect:
  ```javascript
  if (req.session.returnTo) {
    const returnTo = req.session.returnTo;
    delete req.session.returnTo;
    return res.redirect(returnTo);
  }
  ```

- If your login route doesn't handle `returnTo`, users will need to click the email link again after logging in (but it will work the second time)
