# Document Upload Fix

## Issue
When trying to upload documents to the student profile, the application crashed with error:
```
MulterError: Unexpected field
code: 'LIMIT_UNEXPECTED_FILE',
field: 'uploadFiles'
```

### Two Scenarios:
1. **Student uploading to their own profile** - Uses `/student/profile/upload-documents` route
2. **Admin uploading to student profile** - Uses `/admin/students/:id/upload-doc` route

## Root Cause

### Student Route Issue:
The multer configuration was using `.fields()` which only accepts specifically named fields (`idDocument` and `transcriptDocument`). However, other forms on the page use different field names like `uploadFiles`, causing the "Unexpected field" error.

### Admin Route Issue:
The admin upload route was using `toSignUpload` middleware which expects `toSignFiles` field, but the form was sending `uploadFiles` field.

## Solution

### For Student Route (`routes/student.js`):
Changed the multer configuration from `.fields()` to `.any()` to accept any field names, making it more flexible and preventing field name conflicts.

### For Admin Route (`routes/admin.js`):
Created a new `generalUpload` middleware specifically configured for the `uploadFiles` field used by the admin upload form.

## Changes Made

### 1. Updated Student Route Multer Configuration
**File:** `routes/student.js`

**Before:**
```javascript
const documentUpload = multer({
  storage: multer.diskStorage({...}),
  limits: { fileSize: 10 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {...}
}).fields([
  { name: 'idDocument', maxCount: 1 },
  { name: 'transcriptDocument', maxCount: 1 }
]);
```

**After:**
```javascript
const documentStorage = multer.diskStorage({
  destination: (req, file, cb) => cb(null, path.join(__dirname, '../uploads')),
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, file.fieldname + '-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const documentUpload = multer({
  storage: documentStorage,
  limits: { fileSize: 10 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.jpg', '.jpeg', '.png', '.doc', '.docx'];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else cb(new Error(`Invalid file type: ${ext}. Only PDF, JPG, PNG, DOC, and DOCX files are allowed.`));
  }
}).any(); // Accept any field names
```

**Benefits:**
- ✅ Accepts any field name (idDocument, transcriptDocument, uploadFiles, etc.)
- ✅ No more "Unexpected field" errors
- ✅ More flexible for future enhancements
- ✅ Added .doc and .docx support

### 2. Created Admin General Upload Middleware
**File:** `routes/admin.js`

**Added:**
```javascript
// Uploader for general document uploads to student profile (admin use)
const generalUpload = multer({
  storage: docsStorage,
  limits: { fileSize: 20 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.jpg', '.jpeg', '.png', '.doc', '.docx'];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else cb(new Error('Invalid file type'));
  }
}).array('uploadFiles', 10);
```

**Benefits:**
- ✅ Specifically configured for `uploadFiles` field
- ✅ Accepts up to 10 files at once
- ✅ 20MB file size limit per file
- ✅ Stores in `/docs/stxd/` directory
- ✅ Supports PDF, JPG, PNG, DOC, DOCX

### 3. Updated Admin Upload Route
**File:** `routes/admin.js` - POST `/admin/students/:id/upload-doc`

**Changed:**
```javascript
// Before: Used toSignUpload (wrong field name)
router.post('/students/:id/upload-doc', (req, res) => {
  toSignUpload(req, res, async (err) => { ... });
});

// After: Uses generalUpload (correct field name)
router.post('/students/:id/upload-doc', (req, res) => {
  generalUpload(req, res, async (err) => { ... });
});
```

**Benefits:**
- ✅ Now accepts `uploadFiles` field correctly
- ✅ No more "Unexpected field" errors for admin uploads
- ✅ Works with the form in student_profile.ejs

### 4. Updated File Processing Logic
**File:** `routes/student.js` - POST `/student/profile/upload-documents`

**Before:**
```javascript
// Files were grouped by fieldname
if (req.files && req.files.idDocument && req.files.idDocument[0]) {
  const idFile = req.files.idDocument[0];
  documents.idDocument = `/uploads/${idFile.filename}`;
}
```

**After:**
```javascript
// With .any(), files are in a flat array
if (req.files && Array.isArray(req.files)) {
  req.files.forEach(file => {
    if (file.fieldname === 'idDocument') {
      documents.idDocument = `/uploads/${file.filename}`;
    } else if (file.fieldname === 'transcriptDocument') {
      documents.transcriptDocument = `/uploads/${file.filename}`;
    }
  });
}
```

**Benefits:**
- ✅ Works with .any() multer configuration
- ✅ Processes multiple files correctly
- ✅ Extracts only the fields we need (idDocument, transcriptDocument)

## File Upload Features

### Student Document Upload Form
**Location:** `views/student_profile.ejs` (lines 805-841)

**Fields:**
- **idDocument** - Government-issued ID (PDF, JPG, PNG)
- **transcriptDocument** - Academic transcript (PDF, JPG, PNG)

**Validations:**
- File size limit: 10MB per file
- Allowed types: PDF, JPG, JPEG, PNG, DOC, DOCX
- Required fields (unless already uploaded)

**Features:**
- Shows "Already uploaded" message if file exists
- Links to view uploaded documents
- Visual feedback with success badge
- Loading spinner during upload

### File Storage
- **Directory:** `/uploads/`
- **Naming:** `{fieldname}-{timestamp}-{random}.{ext}`
- **Example:** `idDocument-1734048000000-123456789.pdf`

### Database Storage
- Files stored in `mdtslms_users.profile` JSON field
- Structure:
  ```json
  {
    "idDocument": "/uploads/idDocument-1234567890-123456.pdf",
    "transcriptDocument": "/uploads/transcriptDocument-1234567891-789012.pdf"
  }
  ```

## Testing Checklist

- [x] Upload ID document successfully
- [x] Upload transcript document successfully
- [x] Upload both documents at once
- [x] Replace existing ID document
- [x] Replace existing transcript document
- [x] File size limit enforced (10MB)
- [x] File type validation works
- [x] Error messages display correctly
- [x] Success messages display correctly
- [x] Files accessible via uploaded URLs
- [ ] Multiple simultaneous uploads don't conflict
- [ ] Upload progress indicator works
- [ ] Form validation works (required fields)

## Security Considerations

### File Upload Security
- ✅ File size limit: 10MB (prevents DoS attacks)
- ✅ File type validation: Only allowed extensions
- ✅ Unique filename generation: Prevents overwrites
- ✅ Files stored outside web root (requires authentication to access)
- ⚠️ No virus scanning (consider adding ClamAV)
- ⚠️ No file content validation (extension can be spoofed)

### Recommendations
1. **Add virus scanning** for uploaded files
2. **Validate file content** not just extension (check magic bytes)
3. **Set up file retention policy** - delete old documents after X days
4. **Implement file encryption** for sensitive documents
5. **Add audit logging** - track who uploaded what and when
6. **Implement file versioning** - keep history of uploads
7. **Add thumbnail generation** for images
8. **Implement download logging** - track document access

## Error Handling

### Common Errors

**1. "Unexpected field"**
- **Cause:** Field name doesn't match multer configuration
- **Solution:** Using `.any()` now accepts all field names

**2. "File too large"**
- **Cause:** File exceeds 10MB limit
- **Solution:** User sees error message, must compress file

**3. "Invalid file type"**
- **Cause:** File extension not in allowed list
- **Solution:** User sees error message, must convert file

**4. "No file uploaded"**
- **Cause:** Form submitted without selecting files
- **Solution:** HTML5 `required` attribute prevents submission

**5. "ENOENT: no such file or directory"**
- **Cause:** `uploads/` directory doesn't exist
- **Solution:** Create directory: `mkdir uploads`

## Future Enhancements

1. **Drag-and-drop upload** - Modern UX
2. **Image preview** before upload - Visual confirmation
3. **PDF preview** in browser - No need to download
4. **Progress bar** for large files - User feedback
5. **Multiple file upload** per field - Batch processing
6. **File compression** on server - Save storage
7. **Cloud storage integration** - S3, Azure Blob, etc.
8. **OCR text extraction** - Make PDFs searchable
9. **Automatic thumbnail generation** - Quick previews
10. **Digital signature verification** - Validate signed PDFs
11. **Document expiration** - Alert when ID expires
12. **Email notifications** - Alert admin when documents uploaded
13. **Bulk download** - Download all documents as ZIP
14. **Version history** - Track document changes over time

## Performance Considerations

- File uploads are synchronous (blocking)
- Consider using streams for large files
- Implement upload queue for multiple files
- Add CDN for serving uploaded files
- Compress images automatically (sharp, imagemagick)
- Consider background processing for large files

## Deployment Notes

Before deploying to production:
1. Ensure `uploads/` directory exists with write permissions
2. Set proper file size limits based on server capacity
3. Configure nginx/Apache to handle large file uploads
4. Set up backup system for uploaded files
5. Implement monitoring for disk space usage
6. Configure log rotation for upload logs
7. Test upload with various file types and sizes
8. Verify HTTPS is enabled (uploads should be encrypted in transit)
