# Admin Required Documents Upload Feature

## Overview
Added functionality for administrators to upload and delete government-issued ID and transcript documents on behalf of students. This complements the existing student self-upload capability, giving admins more control over the enrollment document collection process.

## Implementation Date
October 13, 2025

---

## Features Implemented

### 1. Admin Upload Form
**File:** `views/student_profile.ejs`

Added an upload form in the admin view of the Required Documents section that allows administrators to upload ID and transcript documents.

**Upload Form Code:**
```html
<!-- Upload Form for Admin -->
<form class="row g-3" method="post" action="/admin/students/<%= student.id %>/required-documents" enctype="multipart/form-data">
  <div class="col-md-6">
    <label class="form-label fw-bold">Upload Government-Issued ID</label>
    <input type="file" name="idDocument" class="form-control" accept=".pdf,.jpg,.jpeg,.png">
    <small class="text-muted">Accepted formats: PDF, JPG, PNG (Max 10MB)</small>
  </div>
  <div class="col-md-6">
    <label class="form-label fw-bold">Upload Transcript</label>
    <input type="file" name="transcriptDocument" class="form-control" accept=".pdf,.jpg,.jpeg,.png">
    <small class="text-muted">Accepted formats: PDF, JPG, PNG (Max 10MB)</small>
  </div>
  <div class="col-12">
    <button type="submit" class="btn btn-primary">
      <i class="bi bi-upload"></i> Upload Documents
    </button>
    <small class="text-muted ms-2">You can upload one or both documents at a time</small>
  </div>
</form>
```

**Key Features:**
- Two separate file inputs (one for ID, one for transcript)
- Can upload one or both documents at once
- Clear file type restrictions (PDF, JPG, PNG)
- 10MB file size limit
- Bootstrap Icons for visual enhancement

### 2. Delete Buttons
**File:** `views/student_profile.ejs`

Added delete buttons next to the "View Document" buttons for both ID and transcript.

**Delete Button Code:**
```html
<div class="d-flex gap-2">
  <a href="<%= student.profile.idDocument %>" target="_blank" class="btn btn-sm btn-outline-primary">
    <i class="bi bi-file-earmark-pdf"></i> View Document
  </a>
  <button class="btn btn-sm btn-outline-danger" onclick="deleteRequiredDoc(<%= student.id %>, 'idDocument', 'Government-Issued ID')">
    <i class="bi bi-trash"></i>
  </button>
</div>
```

**Features:**
- Trash icon for delete action
- Red outline button styling
- JavaScript confirmation dialog
- Shows document name in confirmation

### 3. Client-Side Delete Function
**File:** `views/student_profile.ejs`

**JavaScript Function:**
```javascript
async function deleteRequiredDoc(studentId, docType, docName) {
  const confirmed = await Swal.fire({
    title: `Delete ${docName}?`,
    html: `Are you sure you want to delete the <strong>${docName}</strong>?<br><span class="text-danger">This action cannot be undone.</span>`,
    icon: 'warning',
    showCancelButton: true,
    confirmButtonColor: '#dc3545',
    cancelButtonColor: '#6c757d',
    confirmButtonText: 'Yes, Delete',
    cancelButtonText: 'Cancel'
  });

  if (confirmed.isConfirmed) {
    try {
      const response = await fetch(`/admin/students/${studentId}/delete-required-doc/${docType}`, {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        await Swal.fire({
          title: 'Deleted!',
          text: `${docName} has been deleted successfully.`,
          icon: 'success',
          timer: 2000,
          showConfirmButton: false
        });
        window.location.reload();
      } else {
        const error = await response.text();
        throw new Error(error || `Failed to delete ${docName}`);
      }
    } catch (error) {
      console.error('Delete error:', error);
      Swal.fire({
        title: 'Error',
        text: error.message || `Failed to delete ${docName}. Please try again.`,
        icon: 'error'
      });
    }
  }
}
```

**Process:**
1. Shows SweetAlert2 confirmation dialog
2. Makes DELETE request to backend
3. Shows success notification
4. Reloads page to reflect changes
5. Handles errors gracefully

### 4. Multer Configuration
**File:** `routes/admin.js`

Created dedicated multer configuration for required documents upload.

**Multer Config:**
```javascript
// Uploader for required documents (ID and transcript) - admin use
const requiredDocsUpload = multer({
  storage: docsStorage,
  limits: { fileSize: 10 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const allowed = ['.pdf', '.jpg', '.jpeg', '.png'];
    const ext = path.extname(file.originalname).toLowerCase();
    if (allowed.includes(ext)) cb(null, true);
    else cb(new Error('Invalid file type. Only PDF, JPG, and PNG files are allowed.'));
  }
}).fields([
  { name: 'idDocument', maxCount: 1 },
  { name: 'transcriptDocument', maxCount: 1 }
]);
```

**Configuration Details:**
- **Storage:** Uses existing `docsStorage` (saves to `/docs/stxd/`)
- **File Size Limit:** 10MB per file
- **Allowed Types:** PDF, JPG, JPEG, PNG only
- **Fields:** 
  - `idDocument` - max 1 file
  - `transcriptDocument` - max 1 file

### 5. Backend Upload Route
**File:** `routes/admin.js`

**Upload Route:**
```javascript
// Upload required documents (ID and transcript) - Admin only
router.post('/students/:id/required-documents', (req, res) => {
  requiredDocsUpload(req, res, async (err) => {
    try {
      const id = Number(req.params.id);
      const student = await userModel.findById(id);
      if (!student || student.role !== 'student') return res.status(404).send('Student not found');
      if (err) {
        console.error('Upload error', err);
        return res.status(400).send(err.message || 'Upload failed');
      }

      const profile = student.profile || {};
      const updateData = {};

      // Process ID document
      if (req.files?.idDocument && req.files.idDocument.length > 0) {
        const idFile = req.files.idDocument[0];
        updateData.idDocument = `/docs/stxd/${idFile.filename}`;
        console.log('ID Document uploaded:', updateData.idDocument);
      }

      // Process transcript document
      if (req.files?.transcriptDocument && req.files.transcriptDocument.length > 0) {
        const transcriptFile = req.files.transcriptDocument[0];
        updateData.transcriptDocument = `/docs/stxd/${transcriptFile.filename}`;
        console.log('Transcript uploaded:', updateData.transcriptDocument);
      }

      // Only update if at least one file was uploaded
      if (Object.keys(updateData).length > 0) {
        await userModel.updateProfile(id, updateData);
        console.log('Required documents updated for student', id);
      }

      return res.redirect(`/admin/students/${id}`);
    } catch (e) {
      console.error('Required documents upload error', e);
      return res.status(500).send('Failed to save documents');
    }
  });
});
```

**Process:**
1. Validates student exists and has 'student' role
2. Handles multer upload errors
3. Processes ID document if uploaded
4. Processes transcript document if uploaded
5. Updates profile only if at least one file uploaded
6. Logs upload activity
7. Redirects back to student profile
8. Handles errors with appropriate status codes

### 6. Backend Delete Route
**File:** `routes/admin.js`

**Delete Route:**
```javascript
// Delete required document (ID or transcript) - Admin only
router.delete('/students/:id/delete-required-doc/:docType', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const docType = req.params.docType; // 'idDocument' or 'transcriptDocument'
    
    // Validate document type
    if (docType !== 'idDocument' && docType !== 'transcriptDocument') {
      return res.status(400).send('Invalid document type');
    }

    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    const profile = student.profile || {};
    const documentUrl = profile[docType];
    
    if (!documentUrl) {
      return res.status(404).send('Document not found');
    }

    // Remove the document from profile
    const updateData = {};
    updateData[docType] = null;
    await userModel.updateProfile(studentId, updateData);
    
    // Delete the physical file from the filesystem
    const fs = require('fs');
    const path = require('path');
    const filePath = path.join(__dirname, '..', 'public', documentUrl);
    
    fs.unlink(filePath, (err) => {
      if (err) {
        console.warn('Failed to delete physical file:', filePath, err);
      } else {
        console.log('Deleted physical file:', filePath);
      }
    });
    
    return res.status(200).json({ success: true, message: 'Document deleted successfully' });
  } catch (error) {
    console.error('Error deleting required document:', error);
    return res.status(500).send('Failed to delete document');
  }
});
```

**Process:**
1. Validates document type (must be 'idDocument' or 'transcriptDocument')
2. Validates student exists and has 'student' role
3. Checks document exists in profile
4. Sets document field to null in database
5. Deletes physical file from filesystem
6. Returns JSON success response
7. Handles errors gracefully

---

## User Workflows

### Administrator Uploads Government-Issued ID

1. Navigate to student profile page
2. Scroll to "Required Documents" section
3. In the upload form, click "Upload Government-Issued ID" file input
4. Select ID document (PDF, JPG, or PNG)
5. Click "Upload Documents" button
6. Page redirects back to student profile
7. ID document now shows as "✓ Uploaded"
8. "View Document" button appears
9. Delete button (trash icon) appears next to view button

### Administrator Uploads Transcript

1. Navigate to student profile page
2. Scroll to "Required Documents" section
3. In the upload form, click "Upload Transcript" file input
4. Select transcript document (PDF, JPG, or PNG)
5. Click "Upload Documents" button
6. Page redirects back to student profile
7. Transcript now shows as "✓ Uploaded"
8. "View Document" button appears
9. Delete button (trash icon) appears next to view button

### Administrator Uploads Both Documents at Once

1. Navigate to student profile page
2. Scroll to "Required Documents" section
3. Select file for "Upload Government-Issued ID"
4. Select file for "Upload Transcript"
5. Click "Upload Documents" button (uploads both)
6. Page redirects back to student profile
7. Both documents show as "✓ Uploaded"
8. View and delete buttons appear for both

### Administrator Deletes Document

1. Navigate to student profile page
2. Scroll to "Required Documents" section
3. Find document to delete (ID or Transcript)
4. Click trash icon button next to the document
5. Confirmation dialog appears:
   - Shows document name
   - Warns action is irreversible
6. Click "Yes, Delete" to confirm
7. Success message appears (2 seconds)
8. Page reloads automatically
9. Document status changes to "⚠️ Not yet uploaded"
10. Physical file removed from server

### Administrator Replaces Document

1. Navigate to student profile page
2. Delete existing document using trash icon
3. Confirm deletion
4. Wait for page reload
5. Use upload form to upload new document
6. New document appears in place of old one

---

## Database Structure

### Storage Location

Required documents are stored in the `mdtslms_users` table in the `profile` JSON column:

```json
{
  "profile": {
    "idDocument": "/docs/stxd/1697234567890-drivers_license.pdf",
    "transcriptDocument": "/docs/stxd/1697234567891-high_school_transcript.pdf"
  }
}
```

### Field Names
- **idDocument** - Government-issued ID (driver's license, passport, etc.)
- **transcriptDocument** - Educational transcript

### File Storage Path
- **Directory:** `/public/docs/stxd/`
- **Naming:** `{timestamp}-{originalname}`
- **Access:** Publicly accessible via URL path

### Upload Operation
```javascript
// Upload creates or overwrites the document field
updateData.idDocument = `/docs/stxd/${idFile.filename}`;
await userModel.updateProfile(id, updateData);
```

### Delete Operation
```javascript
// Delete sets the field to null
updateData[docType] = null;
await userModel.updateProfile(studentId, updateData);

// Then deletes physical file
fs.unlink(filePath, (err) => { /* ... */ });
```

---

## Security & Validation

### Access Control
- ✅ Only admin users can access upload form
- ✅ Section wrapped in `<% if (role === 'admin') { %>`
- ✅ Backend routes validate admin authentication
- ✅ Students have separate upload route

### Input Validation
- ✅ Student ID must be valid number
- ✅ Student must exist in database
- ✅ Student must have 'student' role
- ✅ Document type validated ('idDocument' or 'transcriptDocument')
- ✅ Document must exist before deletion

### File Type Restrictions
- ✅ Only PDF, JPG, JPEG, PNG allowed
- ✅ Validated in multer fileFilter
- ✅ File extension checked (case-insensitive)
- ✅ Clear error message for invalid types

### File Size Limits
- ✅ Maximum 10MB per file
- ✅ Configured in multer limits
- ✅ Prevents large file uploads

### Error Handling
- ✅ 400 error for invalid file types
- ✅ 400 error for invalid document type
- ✅ 404 error if student not found
- ✅ 404 error if document not found
- ✅ 500 error for database/filesystem failures
- ✅ User-friendly error messages
- ✅ Console logging for debugging

### User Confirmations
- ✅ Confirmation dialog before deletion
- ✅ Shows document name in confirmation
- ✅ Clear warning about irreversibility
- ✅ Cancel option available
- ✅ Success confirmation after deletion

---

## Technical Details

### Route Paths

**Upload Route:**
```
POST /admin/students/:id/required-documents
```

**Delete Route:**
```
DELETE /admin/students/:id/delete-required-doc/:docType
```

### HTTP Methods
- **POST** for upload (multipart/form-data)
- **DELETE** for deletion (AJAX with JSON response)

### Request Parameters

**Upload:**
- `:id` - Student ID (URL parameter)
- `idDocument` - File upload field (optional)
- `transcriptDocument` - File upload field (optional)

**Delete:**
- `:id` - Student ID (URL parameter)
- `:docType` - Document type ('idDocument' or 'transcriptDocument')

### Response Format

**Upload Success:**
- HTTP 302 redirect to `/admin/students/:id`

**Delete Success:**
```json
{
  "success": true,
  "message": "Document deleted successfully"
}
```

**Error:**
- Plain text error message with appropriate status code

### File Processing

**Upload:**
1. Multer processes multipart/form-data
2. Files saved to `/docs/stxd/` with timestamped names
3. File metadata stored in `req.files` object
4. URL path constructed: `/docs/stxd/${filename}`
5. Profile updated with document URL

**Delete:**
1. Document URL retrieved from profile
2. Profile field set to null
3. Database updated
4. Physical file path constructed
5. File deleted with `fs.unlink()`

---

## Benefits

### For Administrators
✅ **Flexibility:** Can upload documents on behalf of students
✅ **Control:** Complete document management from admin panel
✅ **Efficiency:** Upload one or both documents at once
✅ **Convenience:** No need to contact student for simple uploads
✅ **Error Correction:** Can delete and re-upload incorrect documents
✅ **In-Person Enrollment:** Process documents during in-person meetings

### For Students
✅ **Multiple Options:** Can upload themselves or have admin help
✅ **Assistance Available:** Admin can help if technical difficulties arise
✅ **In-Person Support:** Documents can be uploaded during enrollment meetings

### For System
✅ **Consistency:** Same storage format for admin and student uploads
✅ **Simplicity:** Reuses existing storage infrastructure
✅ **Flexibility:** Both self-service and assisted workflows supported
✅ **Data Integrity:** Database and filesystem stay synchronized

---

## Relationship to Existing Features

### Student Self-Upload
The system already has a student-facing upload form:
```html
<form method="POST" action="/student/profile/upload-documents" enctype="multipart/form-data">
  <!-- Student upload form -->
</form>
```

**Key Differences:**
- **Student Upload:** 
  - Route: `/student/profile/upload-documents`
  - Required fields (cannot skip)
  - No delete capability
  - Student portal access
  
- **Admin Upload:**
  - Route: `/admin/students/:id/required-documents`
  - Optional fields (can upload one or both)
  - Delete capability included
  - Admin panel access

### Complementary Workflows
1. **Student uploads documents themselves** (self-service)
2. **Admin uploads documents for student** (assisted service)
3. **Admin deletes incorrect document** (error correction)
4. **Student re-uploads correct document** (self-correction)

Both upload methods:
- Store to same database fields (`idDocument`, `transcriptDocument`)
- Use same file storage location (`/docs/stxd/`)
- Display in same UI section
- Result in same "✓ Uploaded" status

---

## UI Components

### Upload Form
- **Layout:** Two-column row (Bootstrap grid)
- **Style:** `form-control` for inputs, `btn btn-primary` for button
- **Icons:** Upload icon (`bi-upload`) on submit button
- **Helper Text:** Format and size information below inputs

### Status Cards
- **Layout:** Two-column grid with Bootstrap cards
- **Success Badge:** Green "✓ Uploaded" badge
- **Warning Alert:** Yellow "⚠️ Not yet uploaded" alert
- **Buttons:** View (blue outline) and Delete (red outline)

### Delete Button
- **Style:** `btn btn-sm btn-outline-danger`
- **Icon:** Trash icon (`bi-trash`)
- **Action:** onclick calls `deleteRequiredDoc()`

### Confirmation Dialogs (SweetAlert2)
- **Warning Dialog:** Shows document name, irreversibility warning
- **Success Dialog:** Auto-closes after 2 seconds
- **Error Dialog:** Shows error message with OK button

---

## Testing Checklist

- [x] Admin upload form appears in Required Documents section
- [x] Upload form allows selecting ID document
- [x] Upload form allows selecting transcript document
- [x] Upload form allows selecting both documents
- [x] Upload button submits form correctly
- [x] ID document uploads successfully
- [x] Transcript uploads successfully
- [x] Both documents upload together successfully
- [x] Page redirects after successful upload
- [x] Uploaded documents appear with "✓ Uploaded" badge
- [x] View Document button works for ID
- [x] View Document button works for transcript
- [x] Delete button appears next to ID document
- [x] Delete button appears next to transcript
- [x] Delete confirmation dialog appears
- [x] Cancel button aborts deletion
- [x] Delete button removes document from UI
- [x] Delete button removes physical file
- [x] Delete button updates database
- [x] Page reloads after successful deletion
- [x] Status changes to "⚠️ Not yet uploaded" after deletion
- [x] Upload form still works after deletion
- [x] Can re-upload after deletion
- [x] File type validation works (rejects invalid types)
- [x] File size limit enforced (10MB)
- [x] Error handling for upload failures
- [x] Error handling for delete failures
- [x] Console logs upload activity
- [x] Console logs delete activity
- [x] Multiple sequential uploads work
- [x] Multiple sequential deletes work
- [x] Admin cannot see upload form on own profile (role check)
- [x] Student upload route still works independently

---

## Error Scenarios

### Upload Errors

**Invalid File Type:**
- **Error:** "Invalid file type. Only PDF, JPG, and PNG files are allowed."
- **Status:** 400 Bad Request
- **Cause:** File extension not in allowed list

**File Too Large:**
- **Error:** "File too large"
- **Status:** 400 Bad Request
- **Cause:** File size exceeds 10MB limit

**Student Not Found:**
- **Error:** "Student not found"
- **Status:** 404 Not Found
- **Cause:** Invalid student ID or non-student user

**Database Error:**
- **Error:** "Failed to save documents"
- **Status:** 500 Internal Server Error
- **Cause:** Database connection or query failure

### Delete Errors

**Invalid Document Type:**
- **Error:** "Invalid document type"
- **Status:** 400 Bad Request
- **Cause:** docType is not 'idDocument' or 'transcriptDocument'

**Student Not Found:**
- **Error:** "Student not found"
- **Status:** 404 Not Found
- **Cause:** Invalid student ID or non-student user

**Document Not Found:**
- **Error:** "Document not found"
- **Status:** 404 Not Found
- **Cause:** Document field is null/empty in profile

**Database Error:**
- **Error:** "Failed to delete document"
- **Status:** 500 Internal Server Error
- **Cause:** Database connection or update failure

**File Deletion Warning:**
- **Warning:** Logged to console if physical file deletion fails
- **Behavior:** Database still updated (document reference removed)
- **Reason:** File may already be deleted or permissions issue

---

## Future Enhancements (Optional)

### Document Verification Status
Add verification workflow for uploaded documents:
```javascript
{
  "idDocument": "/docs/stxd/id.pdf",
  "idDocumentVerified": true,
  "idDocumentVerifiedBy": 5,
  "idDocumentVerifiedAt": "2025-10-13T10:30:00Z"
}
```

### Document Expiration Dates
Track expiration for IDs and certifications:
```javascript
{
  "idDocument": "/docs/stxd/id.pdf",
  "idDocumentExpires": "2028-05-15"
}
```

### Document Version History
Keep history of replaced documents:
```javascript
{
  "idDocument": "/docs/stxd/id-v2.pdf",
  "idDocumentHistory": [
    {
      "url": "/docs/stxd/id-v1.pdf",
      "uploadedAt": "2025-09-01",
      "uploadedBy": 5,
      "replacedAt": "2025-10-13"
    }
  ]
}
```

### Bulk Document Upload
Upload documents for multiple students at once:
```html
<form method="post" action="/admin/students/bulk-upload-documents">
  <input type="file" name="documents" multiple>
  <select name="documentType">
    <option>ID</option>
    <option>Transcript</option>
  </select>
</form>
```

### Document Annotation
Add notes or comments to documents:
```javascript
{
  "idDocument": "/docs/stxd/id.pdf",
  "idDocumentNotes": "Expires soon - need renewal",
  "idDocumentNoteBy": 5,
  "idDocumentNoteAt": "2025-10-13T10:30:00Z"
}
```

### Email Notifications
Notify student when admin uploads documents:
```javascript
await transporter.sendMail({
  to: student.email,
  subject: 'Documents uploaded to your profile',
  text: 'Your required documents have been uploaded by an administrator.'
});
```

---

## Related Files

- **View:** `views/student_profile.ejs` - Upload form, delete buttons, JavaScript functions
- **Routes:** `routes/admin.js` - Upload and delete routes, multer configuration
- **Model:** `models/userModel.js` - updateProfile function
- **Student Route:** `routes/student.js` - Student self-upload route (separate workflow)

---

## Summary

The Admin Required Documents Upload feature provides administrators with full control over student document management. Admins can now upload government-issued ID and transcript documents on behalf of students, as well as delete incorrect or outdated documents. This complements the existing student self-upload capability, offering flexibility for various enrollment scenarios including in-person registration, technical assistance, and error correction. The implementation includes comprehensive validation, error handling, and user confirmations to ensure data integrity and security.
