const fs = require('fs');
const path = require('path');

const simsDir = '/Users/backupadmin/Documents/GitHub/LMS/sims';

// Get all simulation directories
const simDirs = fs.readdirSync(simsDir).filter(item => {
  const fullPath = path.join(simsDir, item);
  return fs.statSync(fullPath).isDirectory();
});

console.log(`Found ${simDirs.length} simulation directories`);

let updatedCount = 0;
let errorCount = 0;

simDirs.forEach(simDir => {
  const indexPath = path.join(simsDir, simDir, 'index.html');
  
  if (!fs.existsSync(indexPath)) {
    console.log(`⚠️  No index.html in ${simDir}`);
    return;
  }
  
  try {
    let content = fs.readFileSync(indexPath, 'utf8');
    let modified = false;
    
    // Check if loading overlay already exists
    if (content.includes('unity-loading-overlay')) {
      console.log(`✓ ${simDir} already has loading overlay`);
      return;
    }
    
    // Add loading overlay to canvas-wrap
    const canvasWrapRegex = /(<div class="canvas-wrap">[\s\S]*?<canvas id="unity-canvas"[^>]*><\/canvas>)/;
    if (canvasWrapRegex.test(content)) {
      content = content.replace(
        canvasWrapRegex,
        `$1\n          <div id="unity-loading-overlay">\n            <div class="unity-spinner"></div>\n            <div class="unity-loading-text">Loading...</div>\n          </div>`
      );
      modified = true;
    }
    
    // Add code to hide loading overlay when Unity loads
    const createInstanceRegex = /(createUnityInstance\(canvas,\s*\{[\s\S]*?\}\)\.then\(instance\s*=>\s*\{\s*unityInstance\s*=\s*instance;)/;
    if (createInstanceRegex.test(content)) {
      content = content.replace(
        createInstanceRegex,
        `$1\n          // Hide loading overlay when Unity is ready\n          const loadingOverlay = document.getElementById('unity-loading-overlay');\n          if (loadingOverlay) {\n            loadingOverlay.classList.add('hidden');\n          }`
      );
      modified = true;
    }
    
    // Also add to error handler
    const errorHandlerRegex = /(\.catch\(error\s*=>\s*\{\s*console\.error\([^)]+\);)/;
    if (errorHandlerRegex.test(content)) {
      content = content.replace(
        errorHandlerRegex,
        `$1\n          // Hide loading overlay on error too\n          const loadingOverlay = document.getElementById('unity-loading-overlay');\n          if (loadingOverlay) {\n            loadingOverlay.classList.add('hidden');\n          }`
      );
      modified = true;
    }
    
    if (modified) {
      fs.writeFileSync(indexPath, content, 'utf8');
      console.log(`✓ Updated ${simDir}`);
      updatedCount++;
    } else {
      console.log(`⚠️  Could not update ${simDir} - pattern not found`);
    }
    
  } catch (error) {
    console.error(`✗ Error updating ${simDir}:`, error.message);
    errorCount++;
  }
});

console.log(`\n✓ Successfully updated ${updatedCount} files`);
if (errorCount > 0) {
  console.log(`✗ ${errorCount} files had errors`);
}
