/**
 * Signature Field Editor - Client-side JavaScript
 * Handles drag-and-drop field placement on PDF pages
 */

(function() {
  'use strict';

  let placedFields = [];
  let fieldCounter = 0;
  let draggedElement = null;
  let currentFieldConfig = null;

  // Field size presets (in percentages)
  const fieldSizes = {
    signature: { width: 25, height: 8 },
    initials: { width: 10, height: 6 },
    text: { width: 30, height: 5 },
    date: { width: 15, height: 5 },
    checkbox: { width: 5, height: 5 },
    pretext: { width: 40, height: 6 }
  };

  // Initialize drag and drop
  function initDragAndDrop() {
    const fieldItems = document.querySelectorAll('.field-item');
    const pdfPage = document.getElementById('pdfPage');
    const pdfCanvas = document.getElementById('pdfCanvas');

    if (!pdfPage) return;

    // Make field items draggable
    fieldItems.forEach(item => {
      item.setAttribute('draggable', 'true');
      
      item.addEventListener('dragstart', (e) => {
        draggedElement = {
          type: item.dataset.type,
          signer: item.dataset.signer
        };
        e.dataTransfer.effectAllowed = 'copy';
      });

      item.addEventListener('dragend', () => {
        draggedElement = null;
      });
    });

    // PDF page drop zone
    pdfPage.addEventListener('dragover', (e) => {
      e.preventDefault();
      e.dataTransfer.dropEffect = 'copy';
    });

    pdfPage.addEventListener('drop', (e) => {
      e.preventDefault();
      
      if (!draggedElement) return;

      // Calculate relative position based on the canvas
      const canvas = pdfCanvas || pdfPage.querySelector('canvas');
      if (!canvas) return;
      
      const rect = canvas.getBoundingClientRect();
      
      const x = ((e.clientX - rect.left) / rect.width) * 100;
      const y = ((e.clientY - rect.top) / rect.height) * 100;

      // Show configuration modal
      currentFieldConfig = {
        ...draggedElement,
        x,
        y
      };

      showFieldModal();
    });

    // Prevent dragging the canvas itself
    if (pdfCanvas) {
      pdfCanvas.addEventListener('dragstart', (e) => {
        e.preventDefault();
        return false;
      });
    }
  }

  // Show field configuration modal
  function showFieldModal() {
    const modal = new bootstrap.Modal(document.getElementById('fieldModal'));
    
    // Set default label
    const fieldType = currentFieldConfig.type;
    const signerType = currentFieldConfig.signer;
    const defaultLabel = fieldType === 'pretext' 
      ? 'Pre-filled Text' 
      : `${signerType.charAt(0).toUpperCase() + signerType.slice(1)} ${fieldType.charAt(0).toUpperCase() + fieldType.slice(1)}`;
    
    document.getElementById('fieldLabel').value = defaultLabel;
    document.getElementById('fieldPlaceholder').value = '';
    document.getElementById('fieldRequired').checked = fieldType !== 'pretext'; // pretext is not required
    
    // Show/hide pretext value field
    const pretextDiv = document.getElementById('pretextValueDiv');
    const pretextInput = document.getElementById('pretextValue');
    const requiredCheckbox = document.getElementById('fieldRequired');
    
    if (fieldType === 'pretext') {
      pretextDiv.style.display = 'block';
      pretextInput.value = '';
      requiredCheckbox.disabled = true; // pretext is always read-only, not "required"
    } else {
      pretextDiv.style.display = 'none';
      pretextInput.value = '';
      requiredCheckbox.disabled = false;
    }

    modal.show();
  }

  // Confirm field placement
  window.confirmField = function() {
    const label = document.getElementById('fieldLabel').value.trim();
    const placeholder = document.getElementById('fieldPlaceholder').value.trim();
    const required = document.getElementById('fieldRequired').checked;
    const fieldType = currentFieldConfig.type;

    if (!label) {
      alert('Please enter a field label');
      return;
    }
    
    // For pretext fields, require the value to be entered
    if (fieldType === 'pretext') {
      const pretextValue = document.getElementById('pretextValue').value.trim();
      if (!pretextValue) {
        alert('Please enter the pre-filled text value');
        return;
      }
    }

    // Get current page number
    const pdfPage = document.getElementById('pdfPage');
    const currentPage = pdfPage ? parseInt(pdfPage.dataset.page || '1') : 1;

    const size = fieldSizes[fieldType];
    const field = {
      id: `field-${++fieldCounter}`,
      field_type: fieldType,
      signer_type: currentFieldConfig.signer,
      field_label: label,
      page_number: currentPage,
      x_position: currentFieldConfig.x,
      y_position: currentFieldConfig.y,
      width: size.width,
      height: size.height,
      is_required: required,
      placeholder_text: placeholder
    };
    
    // Add pretext value if this is a pretext field
    if (fieldType === 'pretext') {
      field.pretext_value = document.getElementById('pretextValue').value.trim();
    }

    placedFields.push(field);
    renderField(field);

    // Close modal
    const modal = bootstrap.Modal.getInstance(document.getElementById('fieldModal'));
    modal.hide();
  };

  // Render a field on the PDF page
  function renderField(field) {
    const pdfPage = document.getElementById('pdfPage');
    const fieldDiv = document.createElement('div');
    
    fieldDiv.className = `placed-field ${field.signer_type}`;
    fieldDiv.id = field.id;
    fieldDiv.style.left = `${field.x_position}%`;
    fieldDiv.style.top = `${field.y_position}%`;
    fieldDiv.style.width = `${field.width}%`;
    fieldDiv.style.height = `${field.height}%`;
    
    // For pretext fields, show the value instead of label
    const displayText = field.field_type === 'pretext' && field.pretext_value 
      ? field.pretext_value 
      : field.field_label;
    
    fieldDiv.innerHTML = `
      <span class="field-label-text">${displayText}</span>
      <button class="remove-btn" onclick="removeField('${field.id}')">×</button>
      <div class="resize-handle resize-se" data-direction="se"></div>
      <div class="resize-handle resize-sw" data-direction="sw"></div>
      <div class="resize-handle resize-ne" data-direction="ne"></div>
      <div class="resize-handle resize-nw" data-direction="nw"></div>
      <div class="resize-handle resize-e" data-direction="e"></div>
      <div class="resize-handle resize-w" data-direction="w"></div>
      <div class="resize-handle resize-n" data-direction="n"></div>
      <div class="resize-handle resize-s" data-direction="s"></div>
    `;

    // Make field draggable to reposition
    makeDraggable(fieldDiv, field);
    
    // Make field resizable
    makeResizable(fieldDiv, field);

    pdfPage.appendChild(fieldDiv);
  }

  // Make placed field draggable for repositioning
  function makeDraggable(element, field) {
    let isDragging = false;
    let startX, startY, initialX, initialY;

    element.addEventListener('mousedown', (e) => {
      if (e.target.classList.contains('remove-btn')) return;
      
      isDragging = true;
      const rect = element.parentElement.getBoundingClientRect();
      startX = e.clientX;
      startY = e.clientY;
      initialX = parseFloat(field.x_position);
      initialY = parseFloat(field.y_position);
      
      element.style.cursor = 'grabbing';
      e.preventDefault();
    });

    document.addEventListener('mousemove', (e) => {
      if (!isDragging) return;

      const rect = element.parentElement.getBoundingClientRect();
      const deltaX = ((e.clientX - startX) / rect.width) * 100;
      const deltaY = ((e.clientY - startY) / rect.height) * 100;

      const newX = Math.max(0, Math.min(100 - field.width, initialX + deltaX));
      const newY = Math.max(0, Math.min(100 - field.height, initialY + deltaY));

      element.style.left = `${newX}%`;
      element.style.top = `${newY}%`;
      
      field.x_position = newX;
      field.y_position = newY;
    });

    document.addEventListener('mouseup', () => {
      if (isDragging) {
        isDragging = false;
        element.style.cursor = 'move';
      }
    });
  }

  // Make placed field resizable
  function makeResizable(element, field) {
    const handles = element.querySelectorAll('.resize-handle');
    
    handles.forEach(handle => {
      let isResizing = false;
      let startX, startY, initialWidth, initialHeight, initialX, initialY;
      const direction = handle.dataset.direction;
      
      handle.addEventListener('mousedown', (e) => {
        isResizing = true;
        startX = e.clientX;
        startY = e.clientY;
        initialWidth = parseFloat(field.width);
        initialHeight = parseFloat(field.height);
        initialX = parseFloat(field.x_position);
        initialY = parseFloat(field.y_position);
        
        e.stopPropagation();
        e.preventDefault();
      });
      
      document.addEventListener('mousemove', (e) => {
        if (!isResizing) return;
        
        const rect = element.parentElement.getBoundingClientRect();
        const deltaX = ((e.clientX - startX) / rect.width) * 100;
        const deltaY = ((e.clientY - startY) / rect.height) * 100;
        
        let newWidth = initialWidth;
        let newHeight = initialHeight;
        let newX = initialX;
        let newY = initialY;
        
        // Handle different resize directions
        if (direction.includes('e')) {
          newWidth = Math.max(5, Math.min(100 - initialX, initialWidth + deltaX));
        }
        if (direction.includes('w')) {
          newWidth = Math.max(5, initialWidth - deltaX);
          newX = initialX + (initialWidth - newWidth);
        }
        if (direction.includes('s')) {
          newHeight = Math.max(3, Math.min(100 - initialY, initialHeight + deltaY));
        }
        if (direction.includes('n')) {
          newHeight = Math.max(3, initialHeight - deltaY);
          newY = initialY + (initialHeight - newHeight);
        }
        
        // Ensure field stays within bounds
        newX = Math.max(0, Math.min(100 - newWidth, newX));
        newY = Math.max(0, Math.min(100 - newHeight, newY));
        
        element.style.width = `${newWidth}%`;
        element.style.height = `${newHeight}%`;
        element.style.left = `${newX}%`;
        element.style.top = `${newY}%`;
        
        field.width = newWidth;
        field.height = newHeight;
        field.x_position = newX;
        field.y_position = newY;
      });
      
      document.addEventListener('mouseup', () => {
        if (isResizing) {
          isResizing = false;
        }
      });
    });
  }

  // Remove a field
  window.removeField = function(fieldId) {
    const element = document.getElementById(fieldId);
    if (element) {
      element.remove();
    }
    placedFields = placedFields.filter(f => f.id !== fieldId);
  };

  // Clear canvas (for signature pads)
  window.clearCanvas = function(fieldId) {
    const canvas = document.getElementById(`canvas-${fieldId}`);
    if (canvas) {
      const ctx = canvas.getContext('2d');
      ctx.clearRect(0, 0, canvas.width, canvas.height);
    }
  };

  // Get all placed fields
  window.getPlacedFields = function() {
    return placedFields.map(f => ({
      field_type: f.field_type,
      signer_type: f.signer_type,
      field_label: f.field_label,
      page_number: f.page_number,
      x_position: f.x_position,
      y_position: f.y_position,
      width: f.width,
      height: f.height,
      is_required: f.is_required,
      placeholder_text: f.placeholder_text || ''
    }));
  };

  // Load fields for a specific page
  window.loadFieldsForPage = function(pageNumber) {
    // Remove all currently rendered fields
    const pdfPage = document.getElementById('pdfPage');
    const existingFieldElements = pdfPage.querySelectorAll('.placed-field');
    existingFieldElements.forEach(el => el.remove());

    // Render only fields for this page
    placedFields
      .filter(f => f.page_number === pageNumber)
      .forEach(field => renderField(field));
  };

  // Load existing fields (if editing)
  function loadExistingFields() {
    if (typeof existingFields !== 'undefined' && existingFields.length > 0) {
      existingFields.forEach((field, index) => {
        fieldCounter = Math.max(fieldCounter, index + 1);
        const fieldData = {
          id: `field-${index + 1}`,
          field_type: field.field_type,
          signer_type: field.signer_type,
          field_label: field.field_label,
          page_number: field.page_number || 1,
          x_position: parseFloat(field.x_position),
          y_position: parseFloat(field.y_position),
          width: parseFloat(field.width),
          height: parseFloat(field.height),
          is_required: field.is_required !== false,
          placeholder_text: field.placeholder_text || ''
        };
        placedFields.push(fieldData);
        
        // Only render if it's for page 1 (initial page)
        if (fieldData.page_number === 1) {
          renderField(fieldData);
        }
      });
    }
  }

  // Initialize when DOM is ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
      initDragAndDrop();
      loadExistingFields();
    });
  } else {
    initDragAndDrop();
    loadExistingFields();
  }

})();
