/**
 * Document Signing Module - Database Model
 * A plug-and-play module for managing digital document signatures
 * Similar to DocuSign functionality
 */

const pool = require('./db');

const signatureDocumentModel = {
  /**
   * Initialize database tables for document signing
   * Run this once to set up the necessary tables
   */
  async initializeTables() {
    const connection = await pool.getConnection();
    try {
      // Main signature documents table
      await connection.query(`
        CREATE TABLE IF NOT EXISTS signature_documents (
          id INT AUTO_INCREMENT PRIMARY KEY,
          student_id INT NOT NULL,
          created_by INT NOT NULL,
          original_filename VARCHAR(255) NOT NULL,
          stored_filename VARCHAR(255) NOT NULL,
          file_path VARCHAR(500) NOT NULL,
          title VARCHAR(255) NOT NULL,
          description TEXT,
          status ENUM('draft', 'awaiting_signatures', 'partially_signed', 'completed', 'cancelled') DEFAULT 'draft',
          signing_order ENUM('student_first', 'staff_first', 'any_order') DEFAULT 'any_order',
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
          updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
          completed_at TIMESTAMP NULL,
          INDEX idx_student (student_id),
          INDEX idx_created_by (created_by),
          INDEX idx_status (status)
        )
      `);

      // Signature fields table - defines areas on PDF that need interaction
      await connection.query(`
        CREATE TABLE IF NOT EXISTS signature_fields (
          id INT AUTO_INCREMENT PRIMARY KEY,
          document_id INT NOT NULL,
          field_type ENUM('signature', 'initials', 'text', 'date', 'checkbox') NOT NULL,
          signer_type ENUM('student', 'staff') NOT NULL,
          field_label VARCHAR(255) NOT NULL,
          page_number INT NOT NULL,
          x_position DECIMAL(10,2) NOT NULL,
          y_position DECIMAL(10,2) NOT NULL,
          width DECIMAL(10,2) NOT NULL,
          height DECIMAL(10,2) NOT NULL,
          is_required BOOLEAN DEFAULT TRUE,
          placeholder_text VARCHAR(255),
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
          FOREIGN KEY (document_id) REFERENCES signature_documents(id) ON DELETE CASCADE,
          INDEX idx_document (document_id),
          INDEX idx_signer_type (signer_type)
        )
      `);

      // Signature parties table - tracks who needs to sign
      await connection.query(`
        CREATE TABLE IF NOT EXISTS signature_parties (
          id INT AUTO_INCREMENT PRIMARY KEY,
          document_id INT NOT NULL,
          user_id INT NOT NULL,
          party_type ENUM('student', 'staff') NOT NULL,
          signing_order INT NOT NULL,
          status ENUM('pending', 'viewed', 'signed', 'declined') DEFAULT 'pending',
          signed_at TIMESTAMP NULL,
          signature_data TEXT,
          ip_address VARCHAR(45),
          user_agent VARCHAR(500),
          email_sent BOOLEAN DEFAULT FALSE,
          email_sent_at TIMESTAMP NULL,
          viewed_at TIMESTAMP NULL,
          FOREIGN KEY (document_id) REFERENCES signature_documents(id) ON DELETE CASCADE,
          INDEX idx_document (document_id),
          INDEX idx_user (user_id),
          INDEX idx_status (status),
          UNIQUE KEY unique_party (document_id, user_id, party_type)
        )
      `);

      // Field responses table - stores actual signed/filled data
      await connection.query(`
        CREATE TABLE IF NOT EXISTS signature_field_responses (
          id INT AUTO_INCREMENT PRIMARY KEY,
          field_id INT NOT NULL,
          party_id INT NOT NULL,
          response_value TEXT,
          response_type ENUM('signature_image', 'initials_image', 'text', 'date', 'checkbox') NOT NULL,
          responded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
          FOREIGN KEY (field_id) REFERENCES signature_fields(id) ON DELETE CASCADE,
          FOREIGN KEY (party_id) REFERENCES signature_parties(id) ON DELETE CASCADE,
          INDEX idx_field (field_id),
          INDEX idx_party (party_id)
        )
      `);

      // Audit log for document signing activities
      await connection.query(`
        CREATE TABLE IF NOT EXISTS signature_audit_log (
          id INT AUTO_INCREMENT PRIMARY KEY,
          document_id INT NOT NULL,
          user_id INT,
          action VARCHAR(100) NOT NULL,
          details TEXT,
          ip_address VARCHAR(45),
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
          FOREIGN KEY (document_id) REFERENCES signature_documents(id) ON DELETE CASCADE,
          INDEX idx_document (document_id),
          INDEX idx_user (user_id),
          INDEX idx_action (action)
        )
      `);

      // Document templates table - save documents as reusable templates
      await connection.query(`
        CREATE TABLE IF NOT EXISTS signature_document_templates (
          id INT AUTO_INCREMENT PRIMARY KEY,
          template_name VARCHAR(255) NOT NULL,
          original_filename VARCHAR(255) NOT NULL,
          stored_filename VARCHAR(255) NOT NULL,
          file_path VARCHAR(500) NOT NULL,
          description TEXT,
          created_by INT NOT NULL,
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
          updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
          INDEX idx_created_by (created_by)
        )
      `);

      // Template fields table - stores field definitions for templates
      await connection.query(`
        CREATE TABLE IF NOT EXISTS signature_template_fields (
          id INT AUTO_INCREMENT PRIMARY KEY,
          template_id INT NOT NULL,
          field_type ENUM('signature', 'initials', 'text', 'date', 'checkbox') NOT NULL,
          signer_type ENUM('student', 'staff') NOT NULL,
          field_label VARCHAR(255) NOT NULL,
          page_number INT NOT NULL,
          x_position DECIMAL(10,2) NOT NULL,
          y_position DECIMAL(10,2) NOT NULL,
          width DECIMAL(10,2) NOT NULL,
          height DECIMAL(10,2) NOT NULL,
          is_required BOOLEAN DEFAULT TRUE,
          placeholder_text VARCHAR(255),
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
          FOREIGN KEY (template_id) REFERENCES signature_document_templates(id) ON DELETE CASCADE,
          INDEX idx_template (template_id),
          INDEX idx_signer_type (signer_type)
        )
      `);

      console.log('✓ Document signing tables initialized successfully');
      return { success: true, message: 'Tables created successfully' };
    } catch (error) {
      console.error('Error initializing document signing tables:', error);
      throw error;
    } finally {
      connection.release();
    }
  },

  /**
   * Create a new signature document
   */
  async createDocument(data) {
    const query = `
      INSERT INTO signature_documents 
      (student_id, created_by, original_filename, stored_filename, file_path, title, description, signing_order)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    `;
    const [result] = await pool.query(query, [
      data.student_id,
      data.created_by,
      data.original_filename,
      data.stored_filename,
      data.file_path,
      data.title,
      data.description || '',
      data.signing_order || 'any_order'
    ]);
    return result.insertId;
  },

  /**
   * Add signature fields to a document
   */
  async addFields(documentId, fields) {
    const query = `
      INSERT INTO signature_fields 
      (document_id, field_type, signer_type, field_label, page_number, x_position, y_position, width, height, is_required, placeholder_text)
      VALUES ?
    `;
    const values = fields.map(f => [
      documentId,
      f.field_type,
      f.signer_type,
      f.field_label,
      f.page_number,
      f.x_position,
      f.y_position,
      f.width,
      f.height,
      f.is_required !== false,
      f.placeholder_text || ''
    ]);
    const [result] = await pool.query(query, [values]);
    return result;
  },

  /**
   * Add signature parties to a document
   */
  async addParties(documentId, parties) {
    const query = `
      INSERT INTO signature_parties 
      (document_id, user_id, party_type, signing_order)
      VALUES ?
    `;
    const values = parties.map(p => [
      documentId,
      p.user_id,
      p.party_type,
      p.signing_order
    ]);
    const [result] = await pool.query(query, [values]);
    return result;
  },

  /**
   * Update document status
   */
  async updateDocumentStatus(documentId, status) {
    const query = `
      UPDATE signature_documents 
      SET status = ?, completed_at = ${status === 'completed' ? 'NOW()' : 'NULL'}
      WHERE id = ?
    `;
    await pool.query(query, [status, documentId]);
  },

  /**
   * Get document by ID with all related data
   */
  async getDocumentById(documentId) {
    const [docs] = await pool.query(
      'SELECT * FROM signature_documents WHERE id = ?',
      [documentId]
    );
    
    if (docs.length === 0) return null;
    
    const document = docs[0];
    
    // Get fields with their responses (if any)
    const [fields] = await pool.query(`
      SELECT sf.*, sfr.response_value, sfr.response_type
      FROM signature_fields sf
      LEFT JOIN signature_field_responses sfr ON sf.id = sfr.field_id
      WHERE sf.document_id = ?
      ORDER BY sf.page_number, sf.y_position
    `, [documentId]);
    document.fields = fields;
    
    // Get parties
    const [parties] = await pool.query(`
      SELECT sp.*, 
        JSON_UNQUOTE(JSON_EXTRACT(u.profile, '$.firstName')) as firstName,
        JSON_UNQUOTE(JSON_EXTRACT(u.profile, '$.lastName')) as lastName,
        u.email 
      FROM signature_parties sp
      LEFT JOIN mdtslms_users u ON sp.user_id = u.id
      WHERE sp.document_id = ?
      ORDER BY sp.signing_order
    `, [documentId]);
    document.parties = parties;
    
    return document;
  },

  /**
   * Get all documents for a student
   */
  async getDocumentsByStudent(studentId) {
    const [documents] = await pool.query(
      `SELECT sd.*, 
        JSON_UNQUOTE(JSON_EXTRACT(u.profile, '$.firstName')) as creator_firstName, 
        JSON_UNQUOTE(JSON_EXTRACT(u.profile, '$.lastName')) as creator_lastName,
        (SELECT COUNT(*) FROM signature_parties WHERE document_id = sd.id AND status = 'signed') as signed_count,
        (SELECT COUNT(*) FROM signature_parties WHERE document_id = sd.id) as total_parties
      FROM signature_documents sd
      LEFT JOIN mdtslms_users u ON sd.created_by = u.id
      WHERE sd.student_id = ?
      ORDER BY sd.created_at DESC`,
      [studentId]
    );
    return documents;
  },

  /**
   * Get documents pending signature for a user
   */
  async getPendingDocumentsForUser(userId) {
    const [documents] = await pool.query(`
      SELECT sd.*, sp.id as party_id, sp.status as my_status, sp.party_type,
        JSON_UNQUOTE(JSON_EXTRACT(u.profile, '$.firstName')) as student_firstName,
        JSON_UNQUOTE(JSON_EXTRACT(u.profile, '$.lastName')) as student_lastName
      FROM signature_documents sd
      INNER JOIN signature_parties sp ON sd.id = sp.document_id
      LEFT JOIN mdtslms_users u ON sd.student_id = u.id
      WHERE sp.user_id = ? AND sp.status IN ('pending', 'viewed')
      ORDER BY sd.created_at DESC
    `, [userId]);
    return documents;
  },

  /**
   * Get fields for a specific signer
   */
  async getFieldsForSigner(documentId, signerType) {
    const [fields] = await pool.query(
      'SELECT * FROM signature_fields WHERE document_id = ? AND signer_type = ? ORDER BY page_number, y_position',
      [documentId, signerType]
    );
    return fields;
  },

  /**
   * Save field response
   */
  async saveFieldResponse(fieldId, partyId, responseValue, responseType) {
    const query = `
      INSERT INTO signature_field_responses 
      (field_id, party_id, response_value, response_type)
      VALUES (?, ?, ?, ?)
      ON DUPLICATE KEY UPDATE response_value = VALUES(response_value), responded_at = NOW()
    `;
    await pool.query(query, [fieldId, partyId, responseValue, responseType]);
  },

  /**
   * Get all responses for a party
   */
  async getResponsesForParty(partyId) {
    const [responses] = await pool.query(
      'SELECT * FROM signature_field_responses WHERE party_id = ?',
      [partyId]
    );
    return responses;
  },

  /**
   * Update party status
   */
  async updatePartyStatus(partyId, status, additionalData = {}) {
    let query = 'UPDATE signature_parties SET status = ?';
    const params = [status];
    
    if (status === 'signed') {
      query += ', signed_at = NOW()';
    }
    if (status === 'viewed' && !additionalData.viewed_at) {
      query += ', viewed_at = NOW()';
    }
    if (additionalData.signature_data) {
      query += ', signature_data = ?';
      params.push(additionalData.signature_data);
    }
    if (additionalData.ip_address) {
      query += ', ip_address = ?';
      params.push(additionalData.ip_address);
    }
    if (additionalData.user_agent) {
      query += ', user_agent = ?';
      params.push(additionalData.user_agent);
    }
    
    query += ' WHERE id = ?';
    params.push(partyId);
    
    await pool.query(query, params);
  },

  /**
   * Mark email as sent for a party
   */
  async markEmailSent(partyId) {
    await pool.query(
      'UPDATE signature_parties SET email_sent = TRUE, email_sent_at = NOW() WHERE id = ?',
      [partyId]
    );
  },

  /**
   * Log an audit event
   */
  async logAudit(documentId, userId, action, details, ipAddress = null) {
    const query = `
      INSERT INTO signature_audit_log 
      (document_id, user_id, action, details, ip_address)
      VALUES (?, ?, ?, ?, ?)
    `;
    await pool.query(query, [documentId, userId, action, details, ipAddress]);
  },

  /**
   * Get audit log for a document
   */
  async getAuditLog(documentId) {
    const [logs] = await pool.query(`
      SELECT sal.*,
        JSON_UNQUOTE(JSON_EXTRACT(u.profile, '$.firstName')) as firstName,
        JSON_UNQUOTE(JSON_EXTRACT(u.profile, '$.lastName')) as lastName,
        u.email
      FROM signature_audit_log sal
      LEFT JOIN mdtslms_users u ON sal.user_id = u.id
      WHERE sal.document_id = ?
      ORDER BY sal.created_at DESC
    `, [documentId]);
    return logs;
  },

  /**
   * Check if all required parties have signed
   */
  async checkIfComplete(documentId) {
    const [result] = await pool.query(`
      SELECT 
        COUNT(*) as total_parties,
        SUM(CASE WHEN status = 'signed' THEN 1 ELSE 0 END) as signed_parties
      FROM signature_parties
      WHERE document_id = ?
    `, [documentId]);
    
    return result[0].total_parties === result[0].signed_parties;
  },

  /**
   * Delete a document (and cascade to related records)
   */
  async deleteDocument(documentId) {
    await pool.query('DELETE FROM signature_documents WHERE id = ?', [documentId]);
  },

  /**
   * Get all pending staff signatures (for admin dashboard)
   */
  async getPendingStaffSignatures() {
    const [documents] = await pool.query(`
      SELECT 
        sd.id,
        sd.title,
        sd.description,
        sd.status,
        sd.created_at,
        sp.id as party_id,
        sp.party_type,
        sp.status as party_status,
        u.email as party_email,
        JSON_UNQUOTE(JSON_EXTRACT(u.profile, '$.firstName')) as party_firstName,
        JSON_UNQUOTE(JSON_EXTRACT(u.profile, '$.lastName')) as party_lastName,
        JSON_UNQUOTE(JSON_EXTRACT(student.profile, '$.firstName')) as student_firstName,
        JSON_UNQUOTE(JSON_EXTRACT(student.profile, '$.lastName')) as student_lastName
      FROM signature_documents sd
      INNER JOIN signature_parties sp ON sd.id = sp.document_id
      LEFT JOIN mdtslms_users u ON sp.user_id = u.id
      LEFT JOIN mdtslms_users student ON sd.student_id = student.id
      WHERE sp.party_type = 'staff' 
        AND sp.status IN ('pending', 'viewed')
        AND sd.status IN ('awaiting_signatures', 'partially_signed')
      ORDER BY sd.created_at DESC
    `);
    return documents;
  },

  /**
   * TEMPLATE METHODS
   */

  /**
   * Save a document as a template
   */
  async saveAsTemplate(documentId, templateName, createdBy) {
    const connection = await pool.getConnection();
    try {
      await connection.beginTransaction();

      // Get the document details
      const [documents] = await connection.query(
        'SELECT * FROM signature_documents WHERE id = ?',
        [documentId]
      );
      
      if (documents.length === 0) {
        throw new Error('Document not found');
      }

      const doc = documents[0];

      // Create the template
      const [templateResult] = await connection.query(`
        INSERT INTO signature_document_templates 
        (template_name, original_filename, stored_filename, file_path, description, created_by)
        VALUES (?, ?, ?, ?, ?, ?)
      `, [
        templateName,
        doc.original_filename,
        doc.stored_filename,
        doc.file_path,
        doc.description,
        createdBy
      ]);

      const templateId = templateResult.insertId;

      // Copy the fields from the document to the template
      const [fields] = await connection.query(
        'SELECT * FROM signature_fields WHERE document_id = ?',
        [documentId]
      );

      for (const field of fields) {
        await connection.query(`
          INSERT INTO signature_template_fields
          (template_id, field_type, signer_type, field_label, page_number, 
           x_position, y_position, width, height, is_required, placeholder_text)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `, [
          templateId,
          field.field_type,
          field.signer_type,
          field.field_label,
          field.page_number,
          field.x_position,
          field.y_position,
          field.width,
          field.height,
          field.is_required,
          field.placeholder_text
        ]);
      }

      await connection.commit();
      return { templateId, success: true };
    } catch (error) {
      await connection.rollback();
      throw error;
    } finally {
      connection.release();
    }
  },

  /**
   * Get all templates
   */
  async getAllTemplates() {
    const [templates] = await pool.query(`
      SELECT 
        sdt.*,
        JSON_UNQUOTE(JSON_EXTRACT(u.profile, '$.firstName')) as creator_firstName,
        JSON_UNQUOTE(JSON_EXTRACT(u.profile, '$.lastName')) as creator_lastName,
        (SELECT COUNT(*) FROM signature_template_fields WHERE template_id = sdt.id) as field_count
      FROM signature_document_templates sdt
      LEFT JOIN mdtslms_users u ON sdt.created_by = u.id
      ORDER BY sdt.created_at DESC
    `);
    return templates;
  },

  /**
   * Get template by ID with fields
   */
  async getTemplateById(templateId) {
    const [templates] = await pool.query(
      'SELECT * FROM signature_document_templates WHERE id = ?',
      [templateId]
    );
    
    if (templates.length === 0) {
      return null;
    }

    const template = templates[0];

    // Get fields for this template
    const [fields] = await pool.query(
      'SELECT * FROM signature_template_fields WHERE template_id = ? ORDER BY page_number, y_position',
      [templateId]
    );

    template.fields = fields;
    return template;
  },

  /**
   * Delete a template
   */
  async deleteTemplate(templateId) {
    const fs = require('fs').promises;
    
    // Get template info first to delete the file
    const [templates] = await pool.query(
      'SELECT file_path FROM signature_document_templates WHERE id = ?',
      [templateId]
    );
    
    // Delete from database
    await pool.query('DELETE FROM signature_document_templates WHERE id = ?', [templateId]);
    
    // Delete the physical file if it exists
    if (templates.length > 0 && templates[0].file_path) {
      try {
        await fs.unlink(templates[0].file_path);
      } catch (err) {
        console.error('Error deleting template file:', err);
        // Continue even if file deletion fails
      }
    }
  }
};

module.exports = signatureDocumentModel;
