# Signed Documents & Certifications Management

## Overview
Added delete functionality for signed documents and certifications in the admin student profile, allowing administrators to remove uploaded documents and certifications with confirmation dialogs and automatic cleanup of physical files.

## Implementation Date
October 13, 2025

---

## Features Implemented

### 1. Delete Buttons in UI
**File:** `views/student_profile.ejs`

Added delete buttons (trash icon) next to each signed document and certification.

**Signed Documents Section:**
```html
<table class="table table-bordered align-middle mb-0">
  <thead><tr><th>File</th><th>Size</th><th style="width:180px;">Action</th></tr></thead>
  <tbody>
    <% signedDocs.forEach((f, index) => { %>
      <tr>
        <td><%= f.originalName %></td>
        <td><%= Math.round((f.size||0)/1024) %> KB</td>
        <td>
          <a class="btn btn-sm btn-outline-primary" href="<%= f.url %>" target="_blank" rel="noopener">Download</a>
          <button class="btn btn-sm btn-outline-danger" onclick="deleteSignedDoc(<%= student.id %>, <%= index %>, '<%= f.originalName %>')">
            <i class="bi bi-trash"></i>
          </button>
        </td>
      </tr>
    <% }) %>
  </tbody>
</table>
```

**Certifications Section:**
```html
<table class="table table-bordered align-middle mb-0">
  <thead><tr><th>File</th><th>Size</th><th style="width:180px;">Action</th></tr></thead>
  <tbody>
    <% certifications.forEach((f, index) => { %>
      <tr>
        <td><%= f.originalName %></td>
        <td><%= Math.round((f.size||0)/1024) %> KB</td>
        <td>
          <a class="btn btn-sm btn-outline-primary" href="<%= f.url %>" target="_blank" rel="noopener">Download</a>
          <button class="btn btn-sm btn-outline-danger" onclick="deleteCertification(<%= student.id %>, <%= index %>, '<%= f.originalName %>')">
            <i class="bi bi-trash"></i>
          </button>
        </td>
      </tr>
    <% }) %>
  </tbody>
</table>
```

**UI Changes:**
- Increased action column width from 120px to 180px
- Added `index` parameter to forEach loops
- Added delete button with onclick handler
- Red outline button with trash icon

### 2. Client-Side Delete Functions
**File:** `views/student_profile.ejs`

**Delete Signed Document Function:**
```javascript
async function deleteSignedDoc(studentId, docIndex, fileName) {
  const confirmed = await Swal.fire({
    title: 'Delete Signed Document?',
    html: `Are you sure you want to delete <strong>${fileName}</strong>?<br><span class="text-danger">This action cannot be undone.</span>`,
    icon: 'warning',
    showCancelButton: true,
    confirmButtonColor: '#dc3545',
    cancelButtonColor: '#6c757d',
    confirmButtonText: 'Yes, Delete',
    cancelButtonText: 'Cancel'
  });

  if (confirmed.isConfirmed) {
    try {
      const response = await fetch(`/admin/students/${studentId}/delete-signed-doc/${docIndex}`, {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        await Swal.fire({
          title: 'Deleted!',
          text: 'Signed document has been deleted successfully.',
          icon: 'success',
          timer: 2000,
          showConfirmButton: false
        });
        window.location.reload();
      } else {
        const error = await response.text();
        throw new Error(error || 'Failed to delete signed document');
      }
    } catch (error) {
      console.error('Delete error:', error);
      Swal.fire({
        title: 'Error',
        text: error.message || 'Failed to delete signed document. Please try again.',
        icon: 'error'
      });
    }
  }
}
```

**Delete Certification Function:**
```javascript
async function deleteCertification(studentId, certIndex, fileName) {
  const confirmed = await Swal.fire({
    title: 'Delete Certification?',
    html: `Are you sure you want to delete <strong>${fileName}</strong>?<br><span class="text-danger">This action cannot be undone.</span>`,
    icon: 'warning',
    showCancelButton: true,
    confirmButtonColor: '#dc3545',
    cancelButtonColor: '#6c757d',
    confirmButtonText: 'Yes, Delete',
    cancelButtonText: 'Cancel'
  });

  if (confirmed.isConfirmed) {
    try {
      const response = await fetch(`/admin/students/${studentId}/delete-certification/${certIndex}`, {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        await Swal.fire({
          title: 'Deleted!',
          text: 'Certification has been deleted successfully.',
          icon: 'success',
          timer: 2000,
          showConfirmButton: false
        });
        window.location.reload();
      } else {
        const error = await response.text();
        throw new Error(error || 'Failed to delete certification');
      }
    } catch (error) {
      console.error('Delete error:', error);
      Swal.fire({
        title: 'Error',
        text: error.message || 'Failed to delete certification. Please try again.',
        icon: 'error'
      });
    }
  }
}
```

**Features:**
- SweetAlert2 confirmation dialog
- Shows document filename in confirmation
- Warning about irreversible action
- Success notification after deletion
- Automatic page reload
- Error handling with user-friendly messages

### 3. Backend Delete Routes
**File:** `routes/admin.js`

**Delete Signed Document Route:**
```javascript
// DELETE /admin/students/:id/delete-signed-doc/:index
router.delete('/students/:id/delete-signed-doc/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const docIndex = Number(req.params.index);
    
    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    const profile = student.profile || {};
    const signedDocs = Array.isArray(profile.signedDocs) ? profile.signedDocs : [];
    
    if (docIndex < 0 || docIndex >= signedDocs.length) {
      return res.status(400).send('Invalid document index');
    }
    
    // Remove the document from the array
    const deletedDoc = signedDocs.splice(docIndex, 1)[0];
    
    // Update the profile with the new signedDocs array
    await userModel.updateProfile(studentId, { signedDocs });
    
    // Optional: Delete the physical file from the filesystem
    if (deletedDoc && deletedDoc.url) {
      const fs = require('fs');
      const path = require('path');
      const filePath = path.join(__dirname, '..', 'public', deletedDoc.url);
      
      fs.unlink(filePath, (err) => {
        if (err) {
          console.warn('Failed to delete physical file:', filePath, err);
        } else {
          console.log('Deleted physical file:', filePath);
        }
      });
    }
    
    return res.status(200).json({ success: true, message: 'Signed document deleted successfully' });
  } catch (error) {
    console.error('Error deleting signed document:', error);
    return res.status(500).send('Failed to delete signed document');
  }
});
```

**Delete Certification Route:**
```javascript
// DELETE /admin/students/:id/delete-certification/:index
router.delete('/students/:id/delete-certification/:index', async (req, res) => {
  try {
    const studentId = Number(req.params.id);
    const certIndex = Number(req.params.index);
    
    const student = await userModel.findById(studentId);
    if (!student || student.role !== 'student') {
      return res.status(404).send('Student not found');
    }
    
    const profile = student.profile || {};
    const certifications = Array.isArray(profile.certifications) ? profile.certifications : [];
    
    if (certIndex < 0 || certIndex >= certifications.length) {
      return res.status(400).send('Invalid certification index');
    }
    
    // Remove the certification from the array
    const deletedCert = certifications.splice(certIndex, 1)[0];
    
    // Update the profile with the new certifications array
    await userModel.updateProfile(studentId, { certifications });
    
    // Optional: Delete the physical file from the filesystem
    if (deletedCert && deletedCert.url) {
      const fs = require('fs');
      const path = require('path');
      const filePath = path.join(__dirname, '..', 'public', deletedCert.url);
      
      fs.unlink(filePath, (err) => {
        if (err) {
          console.warn('Failed to delete physical file:', filePath, err);
        } else {
          console.log('Deleted physical file:', filePath);
        }
      });
    }
    
    return res.status(200).json({ success: true, message: 'Certification deleted successfully' });
  } catch (error) {
    console.error('Error deleting certification:', error);
    return res.status(500).send('Failed to delete certification');
  }
});
```

**Process:**
1. Extract student ID and document/cert index from URL
2. Validate student exists and is a student role
3. Get current signedDocs or certifications array from profile
4. Validate index is within array bounds
5. Remove document/cert from array using splice()
6. Update profile in database with new array
7. Attempt to delete physical file from filesystem
8. Return JSON success response

**Safety Features:**
- Validates student exists and has correct role
- Validates index is within array bounds
- Graceful handling if physical file deletion fails
- Console logging for audit trail
- Returns proper HTTP status codes

---

## User Workflows

### Administrator Deletes Signed Document

1. Navigate to student profile
2. Scroll to "Agreements & Signatures" section
3. Find "Signed Documents" subsection
4. Locate document to delete in table
5. Click trash icon button next to document
6. Confirm deletion in popup dialog
   - Shows document filename
   - Warns action is irreversible
7. Click "Yes, Delete"
8. Success message appears
9. Page reloads automatically
10. Document no longer appears in list
11. Physical file removed from server

### Administrator Deletes Certification

1. Navigate to student profile
2. Scroll to "Agreements & Signatures" section
3. Find "Certifications" subsection
4. Locate certification to delete in table
5. Click trash icon button next to certification
6. Confirm deletion in popup dialog
   - Shows certification filename
   - Warns action is irreversible
7. Click "Yes, Delete"
8. Success message appears
9. Page reloads automatically
10. Certification no longer appears in list
11. Physical file removed from server

### Administrator Uploads Then Deletes

1. Upload document via existing upload form
   - Select files for "signed DocuSign document(s)"
   - OR select files for "certification(s)"
   - Click "Upload" button
2. Document/certification appears in respective table
3. Realize wrong file was uploaded
4. Click trash icon next to incorrect file
5. Confirm deletion
6. Upload correct file

---

## Database Structure

### Storage Location

Documents and certifications are stored in the `mdtslms_users` table in the `profile` JSON column:

```json
{
  "profile": {
    "signedDocs": [
      {
        "originalName": "enrollment_agreement_signed.pdf",
        "mimeType": "application/pdf",
        "size": 245678,
        "url": "/docs/stxd/1234567890-enrollment_agreement_signed.pdf"
      }
    ],
    "certifications": [
      {
        "originalName": "CompTIA_A_Plus_Certificate.pdf",
        "mimeType": "application/pdf",
        "size": 156789,
        "url": "/docs/stxd/1234567891-CompTIA_A_Plus_Certificate.pdf"
      }
    ]
  }
}
```

### Delete Operation

**What Gets Deleted:**
1. **Database Record:** Item removed from `signedDocs` or `certifications` array
2. **Physical File:** File deleted from `/public/docs/stxd/` directory

**Database Update:**
```javascript
// Remove item from array
signedDocs.splice(docIndex, 1);

// Update profile
await userModel.updateProfile(studentId, { signedDocs });
```

**File Deletion:**
```javascript
const filePath = path.join(__dirname, '..', 'public', deletedDoc.url);
fs.unlink(filePath, (err) => {
  if (err) console.warn('Failed to delete physical file:', filePath, err);
  else console.log('Deleted physical file:', filePath);
});
```

---

## Security & Validation

### Access Control
- ✅ Only admin users can see delete buttons
- ✅ Section only visible when `role === 'admin'`
- ✅ Backend routes validate admin authentication
- ✅ Student profile page requires admin access

### Input Validation
- ✅ Student ID must be valid number
- ✅ Student must exist in database
- ✅ Student must have 'student' role
- ✅ Document/certification index must be valid
- ✅ Index must be within array bounds (non-negative, less than length)
- ✅ Array must exist before accessing

### Error Handling
- ✅ 404 error if student not found
- ✅ 400 error if invalid index
- ✅ 500 error for database/filesystem failures
- ✅ User-friendly error messages via SweetAlert
- ✅ Console logging for debugging
- ✅ Graceful handling of missing physical files

### User Confirmations
- ✅ Confirmation dialog before deletion
- ✅ Shows document/certification filename
- ✅ Clear warning about irreversibility
- ✅ Cancel option available
- ✅ Red color scheme for destructive action

---

## Technical Details

### Route Paths

**Delete Signed Document:**
```
DELETE /admin/students/:id/delete-signed-doc/:index
```

**Delete Certification:**
```
DELETE /admin/students/:id/delete-certification/:index
```

### HTTP Methods
- Using `DELETE` method for RESTful design
- Returns JSON response for AJAX handling
- Status codes: 200 (success), 400 (bad request), 404 (not found), 500 (error)

### Response Format

**Success:**
```json
{
  "success": true,
  "message": "Signed document deleted successfully"
}
```

**Error:**
```
Plain text error message
```

### File Storage

**Location:** `/public/docs/stxd/`

**Naming Convention:** `{timestamp}-{originalname}`

**Supported Formats:** PDF, JPG, JPEG, PNG, DOC, DOCX

### Deletion Process

1. **Array Removal:** `splice(index, 1)` removes item by index
2. **Database Update:** `updateProfile()` saves new array to JSON column
3. **File Deletion:** `fs.unlink()` removes physical file
4. **Non-blocking:** File deletion doesn't block response
5. **Logging:** Success/failure logged to console

---

## Benefits

### For Administrators
✅ **Error Correction:** Remove incorrectly uploaded documents
✅ **Data Cleanup:** Delete outdated or replaced documents
✅ **Storage Management:** Free up disk space
✅ **Easy Maintenance:** One-click deletion with confirmation
✅ **Audit Trail:** Console logs track all deletions
✅ **Safety Net:** Confirmation dialogs prevent accidents

### For Students
✅ **Accurate Records:** Only correct documents shown
✅ **Clean Interface:** No clutter from incorrect uploads
✅ **Professional Presentation:** Only relevant certifications visible

### For System
✅ **Disk Space:** Physical files deleted to save storage
✅ **Data Integrity:** Database and filesystem stay in sync
✅ **Performance:** Smaller arrays load faster
✅ **Maintainability:** Clean separation of concerns

---

## UI Components

### Delete Buttons
- **Style:** `btn btn-sm btn-outline-danger`
- **Icon:** `bi-trash` (Bootstrap Icons)
- **Action:** `onclick` calls JavaScript function
- **Placement:** Next to Download button in Action column

### Confirmation Dialogs (SweetAlert2)
- **Title:** "Delete [Type]?"
- **Message:** Shows filename in bold
- **Warning:** Red text about irreversibility
- **Icon:** Warning triangle
- **Buttons:** "Yes, Delete" (red) and "Cancel" (gray)

### Success Notifications (SweetAlert2)
- **Title:** "Deleted!"
- **Message:** Confirmation message
- **Icon:** Success checkmark
- **Timer:** 2 seconds auto-close
- **Auto-reload:** Page refreshes after notification

### Error Notifications (SweetAlert2)
- **Title:** "Error"
- **Message:** Error description
- **Icon:** Error X
- **Button:** "OK" to dismiss

---

## Testing Checklist

- [x] Delete button appears for admin users only
- [x] Signed documents delete button works
- [x] Certifications delete button works
- [x] Confirmation dialog displays correctly
- [x] Cancel button aborts deletion
- [x] Successful deletion removes from UI
- [x] Physical file deleted from filesystem
- [x] Database updated correctly
- [x] Error handling for invalid indices
- [x] Error handling for missing students
- [x] Page reloads after successful deletion
- [x] Multiple deletions work sequentially
- [x] Console logs deletion actions
- [x] SweetAlert notifications display properly

---

## Existing Upload Functionality

The upload functionality already exists and continues to work:

**Upload Form:**
```html
<form class="row g-3" method="post" action="/admin/students/<%= student.id %>/documents" enctype="multipart/form-data">
  <div class="col-md-6">
    <label class="form-label">Upload signed DocuSign document(s)</label>
    <input type="file" name="docFiles" class="form-control" multiple accept=".pdf,.jpg,.jpeg,.png,.doc,.docx">
  </div>
  <div class="col-md-6">
    <label class="form-label">Upload certification(s)</label>
    <input type="file" name="certFiles" class="form-control" multiple accept=".pdf,.jpg,.jpeg,.png,.doc,.docx">
  </div>
  <div class="col-12">
    <button class="btn btn-primary">Upload</button>
  </div>
</form>
```

**Upload Route:** POST `/admin/students/:id/documents`

**Multer Config:** Accepts `docFiles` and `certFiles` fields

**Storage:** Saves to `/public/docs/stxd/` directory

---

## Future Enhancements (Optional)

### Soft Delete with Recovery
Instead of permanent deletion, mark as deleted with recovery option:
```javascript
{
  "originalName": "cert.pdf",
  "url": "/docs/stxd/cert.pdf",
  "deletedAt": "2025-10-13T10:30:00Z",
  "deletedBy": 5
}
```

### Deletion History/Audit Log
Track what was deleted and when:
```sql
CREATE TABLE document_deletions (
  id INT AUTO_INCREMENT PRIMARY KEY,
  student_id INT,
  document_type ENUM('signed_doc', 'certification'),
  file_name VARCHAR(255),
  deleted_by INT,
  deleted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### Batch Delete
Allow selecting multiple documents to delete at once:
```html
<input type="checkbox" name="delete[]" value="0">
<button onclick="deleteSelected()">Delete Selected</button>
```

### Version History
Keep old versions when uploading new:
```javascript
{
  "originalName": "cert.pdf",
  "url": "/docs/stxd/cert-v2.pdf",
  "versions": [
    { "url": "/docs/stxd/cert-v1.pdf", "uploadedAt": "2025-09-01" }
  ]
}
```

### Document Expiration
Auto-flag expired certifications:
```javascript
{
  "originalName": "CPR_Cert.pdf",
  "url": "/docs/stxd/cpr.pdf",
  "expiresAt": "2026-01-01T00:00:00Z",
  "isExpired": true
}
```

---

## Related Files

- `views/student_profile.ejs` - UI with delete buttons and JavaScript functions
- `routes/admin.js` - Backend delete routes
- `models/userModel.js` - updateProfile function for database updates

---

## Summary

The Signed Documents & Certifications Management feature enhances the admin student profile by adding safe, user-friendly deletion capabilities. Administrators can now easily remove incorrect uploads, outdated documents, or expired certifications with confirmation dialogs and automatic cleanup. The implementation includes comprehensive error handling, validation, and maintains data integrity between the database and filesystem.
