# Enrollment Approval Restriction

## Overview
This feature ensures that only **approved students** can be enrolled by admins into classes from the student profile page. Students with pending or other non-approved status cannot be enrolled until they are approved.

## Feature Details

### Frontend Restriction
**Location:** `views/student_profile.ejs` (Lines ~212-243)

#### Implementation:
When an admin views a student's profile, the enrollment section now checks the student's approval status:

**For Non-Approved Students:**
- Enrollment form is hidden
- Warning message displayed:
  ```
  ⚠️ Student must be approved before they can be enrolled in classes.
  ```

**For Approved Students:**
- Enrollment form is visible
- Admin can select from available classes
- "Enroll Student" button functions normally

### Backend Validation
**Location:** `routes/admin.js` - POST `/students/:id/enroll` (Lines ~2805-2834)

#### Security Check:
Added server-side validation to prevent enrollment attempts via API/form manipulation:

```javascript
// Check if student is approved
if (student.status !== 'approved') {
  return res.status(403).send('Only approved students can be enrolled in classes. Please approve the student first.');
}
```

#### HTTP Status:
- **403 Forbidden** - Returned when attempting to enroll a non-approved student
- Error message clearly indicates approval is required

## Student Status Values

### Approval States:
- **`'approved'`** - Student is approved and can be enrolled ✅
- **`'pending'`** - Student is pending approval (cannot enroll) ⏳
- **Other statuses** - Cannot be enrolled until status changes to 'approved' ❌

## User Interface

### Warning Alert (Non-Approved)
```html
<div class="alert alert-warning mb-0">
  <i class="bi bi-exclamation-triangle"></i> 
  Student must be <strong>approved</strong> before they can be enrolled in classes.
</div>
```

**Styling:**
- Yellow warning background
- Triangle exclamation icon
- Bold text emphasis on "approved"
- Clear, actionable message

### Enrollment Form (Approved)
The existing enrollment form remains unchanged when the student is approved:
- Dropdown to select class
- Shows class name, cohort, and start date
- Green "Enroll Student" button

## Logic Flow

### Admin Workflow:
1. Admin navigates to student profile
2. System checks `student.status`
3. **If status !== 'approved':**
   - Show warning message
   - Hide enrollment form
   - Admin must approve student first
4. **If status === 'approved':**
   - Show enrollment form
   - Admin can enroll student in classes

### Approval Process:
1. Admin clicks "Approve Student" button (existing feature)
2. Student status changes to `'approved'`
3. Page reloads
4. Enrollment section now shows enrollment form
5. Admin can now enroll student in classes

## Code Changes

### View Template Update
**File:** `views/student_profile.ejs`

**Before:**
```html
<% if (availableClasses && availableClasses.length > 0) { %>
  <form method="post" action="/admin/students/<%= student.id %>/enroll">
    <!-- Enrollment form -->
  </form>
<% } %>
```

**After:**
```html
<% if (student.status !== 'approved') { %>
  <div class="alert alert-warning mb-0">
    <i class="bi bi-exclamation-triangle"></i> 
    Student must be <strong>approved</strong> before they can be enrolled in classes.
  </div>
<% } else if (availableClasses && availableClasses.length > 0) { %>
  <form method="post" action="/admin/students/<%= student.id %>/enroll">
    <!-- Enrollment form -->
  </form>
<% } %>
```

### Backend Route Update
**File:** `routes/admin.js`

**Added Validation:**
```javascript
const student = await userModel.findById(studentId);
if (!student || student.role !== 'student') {
  return res.status(404).send('Student not found');
}

// NEW: Check if student is approved
if (student.status !== 'approved') {
  return res.status(403).send('Only approved students can be enrolled in classes. Please approve the student first.');
}

const klass = await classModel.findClassById(classId);
// ... continue enrollment
```

## Security Benefits

1. **Frontend Protection:** Prevents accidental enrollment attempts by hiding the form
2. **Backend Validation:** Ensures no bypass attempts via API manipulation
3. **Clear Messaging:** Admins understand why enrollment is blocked
4. **Data Integrity:** Only approved students are in classes
5. **Workflow Enforcement:** Requires proper approval process

## Error Handling

### Frontend:
- No form submission possible (form is hidden)
- Visual warning explains the restriction
- No confusing errors or failed submissions

### Backend:
- **403 Forbidden** status code
- Clear error message displayed
- Request blocked before database operations
- Logged for security monitoring

## Related Features

### Student Approval Button
**Location:** `views/student_profile.ejs` (Line ~115)

Condition to show approval button:
```html
<% if (role === 'admin' && student.status !== 'approved') { %>
  <form method="post" action="/admin/approve/<%= student.id %>">
    <button type="submit" class="btn btn-success">
      <i class="bi bi-check-circle"></i> Approve Student
    </button>
  </form>
<% } %>
```

### Enrollment Section Visibility
The entire "Enrolled Classes" section is only visible to admins:
```html
<% if (role === 'admin') { %>
  <section class="cardish">
    <div class="section-title">Enrolled Classes</div>
    <!-- Enrollments and enrollment form -->
  </section>
<% } %>
```

## Testing Scenarios

### Test Case 1: Pending Student
1. Create/view student with status = 'pending'
2. Navigate to student profile as admin
3. **Expected:** Warning message shown, no enrollment form
4. Attempt API call to enroll
5. **Expected:** 403 Forbidden response

### Test Case 2: Approved Student with Classes
1. Approve student (status = 'approved')
2. Navigate to student profile
3. **Expected:** Enrollment form visible
4. Select class and submit
5. **Expected:** Student enrolled successfully

### Test Case 3: Approved Student - All Classes Enrolled
1. Student already enrolled in all classes
2. Navigate to student profile
3. **Expected:** Info message "Student is enrolled in all available classes"

### Test Case 4: Approval Flow
1. Start with pending student
2. **Expected:** Warning shown
3. Click "Approve Student" button
4. Page reloads
5. **Expected:** Enrollment form now visible

## Benefits

1. **Process Enforcement:** Ensures approval workflow is followed
2. **Prevent Errors:** Avoids enrolling unapproved students
3. **Clear Communication:** Admins know why they can't enroll
4. **Data Quality:** Only approved students in classes
5. **Security:** Backend validation prevents bypass
6. **User Experience:** Clear visual feedback
7. **Consistency:** Status checks throughout system

## Future Enhancements

### Potential Improvements:
- **Quick Approve Button:** Add approve button directly in enrollment section
- **Status Badge:** Show student status badge next to name
- **Bulk Approval:** Approve multiple students and enroll in batch
- **Status History:** Track when student was approved and by whom
- **Email Notification:** Notify student when approved and enrollment available
- **Conditional Enrollment:** Allow enrollment with auto-approval option

## Dependencies

- **Bootstrap 5.3.3:** Alert styling and icons
- **Bootstrap Icons:** Warning icon (bi-exclamation-triangle)
- **Express.js:** Backend routing and validation
- **User Model:** Student status field
- **Class Model:** Enrollment methods

## Database Schema

### User Status Field
```javascript
{
  id: 123,
  name: "John Doe",
  role: "student",
  status: "approved" | "pending" | "rejected" | ...,
  // ... other fields
}
```

**Key Field:** `status` (string)
**Valid Values:** 
- `"approved"` - Can be enrolled
- `"pending"` - Cannot be enrolled
- Other values - Cannot be enrolled

## Notes

- **Backwards Compatibility:** Existing approved students continue to work
- **No Migration Required:** Uses existing status field
- **Non-Breaking:** Only adds restrictions, doesn't remove functionality
- **Flexible:** Easy to modify approved status values if needed

---

**Implemented:** October 14, 2025
**Status:** ✅ Active
**Files Modified:**
1. `views/student_profile.ejs` - Frontend enrollment restriction
2. `routes/admin.js` - Backend validation check
