# Drip Schedule Feature

## Overview
The drip schedule feature allows admins and teachers to schedule when lectures, assignments, simulations, and tests become available to students based on their individual enrollment date. Instead of using fixed calendar dates, content is unlocked relative to each student's enrollment day.

## Key Benefits
- **Flexible Enrollment**: Students can enroll at any time and receive content at the appropriate pace
- **Self-Paced Learning**: Content unlocks progressively as students advance through the course
- **Automatic Management**: No need to manually adjust dates for each cohort or individual student
- **Simple Configuration**: Use day numbers (Day 1, Day 2, etc.) instead of calendar dates

## Implementation Date
October 29, 2025

---

## How It Works

### Enrollment Tracking
When a student is added to a class, the system automatically records:
- **Student ID**: The ID of the enrolled student
- **Enrollment Date**: The exact date and time they were enrolled

This data is stored in the `studentEnrollments` array within each class record.

### Day Calculation
The system calculates the current "class day" for each student by:
1. Finding their enrollment date from `studentEnrollments`
2. Calculating the number of days between enrollment and today
3. Adding 1 (so enrollment day = Day 1)

### Content Filtering
When a student views a class, the system:
1. Retrieves all lectures, assignments, simulations, and tests
2. Filters each content type based on the student's current class day
3. Only shows content where `dripDay` is null (always available) or ≤ current class day

---

## Using the Feature

### For Admins and Teachers

#### Setting Drip Days
When adding or editing content (lectures, assignments, simulations, tests):

1. **Drip Day Field**: Enter a number representing the day the content should become available
   - `1` = Available from Day 1 (enrollment day)
   - `2` = Available from Day 2 (day after enrollment)
   - `7` = Available from Day 7 (one week after enrollment for 7-day classes)
   - Leave blank = Always available (no drip restriction)

2. **Date Field (Optional)**: Can still be used for due dates or informational purposes
   - The date field works independently of drip scheduling
   - Useful for displaying deadlines or suggested completion dates

#### Example Scenarios

**7-Day Class Structure:**
- Day 1: Welcome Lecture, Intro Simulation
- Day 2: Lesson 1 Lecture, Practice Assignment
- Day 4: Mid-week Quiz (Practice)
- Day 7: Final Test (Official/Graded)

**14-Day Class Structure:**
- Days 1-3: Foundational content
- Days 4-7: Intermediate material
- Days 8-11: Advanced topics
- Days 12-14: Final projects and assessments

#### Adding Content with Drip Schedule

**Lectures:**
```
Title: Introduction to Variables
URL: https://youtube.com/watch?v=...
Date: (optional - for reference)
Drip Day: 1
```

**Assignments:**
```
Title: Practice Problems Set 1
URL: https://docs.google.com/...
Date: (optional due date)
Drip Day: 2
```

**Simulations:**
```
Title: Network Configuration Lab
URL: /sims/network-lab/index.html
Date: (optional)
Drip Day: 5
```

**Tests:**
```
Title: Final Assessment
Due Date: (optional)
Drip Day: 7
Official: ✓ (Graded)
```

#### Managing Content

1. **Edit Content**: Click the "Edit" button next to any content item
2. **Update Drip Day**: Modify the day number in the "Day #" field
3. **Remove Drip Restriction**: Clear the "Day #" field to make content always available
4. **Content Display**: 
   - Green badge shows "Day X" next to content items with drip restrictions
   - Blue/primary badge shows optional dates

### For Students

#### Student Experience

1. **Enrollment Day Tracking**: 
   - Students see "Class Day X" at the top of their class page
   - This shows their current progress through the course

2. **Content Visibility**:
   - Only content available for their current day is displayed
   - Content "unlocks" automatically as days progress
   - No indication of future locked content to avoid confusion

3. **No Action Required**:
   - The system handles everything automatically
   - Students simply access their class and see what's currently available

---

## Technical Implementation

### Database Schema

**Class Model Updates:**
- Added `studentEnrollments` field (JSON array)
  ```json
  [
    { "studentId": 123, "enrolledAt": "2025-10-29T08:00:00.000Z" },
    { "studentId": 124, "enrolledAt": "2025-10-30T09:15:00.000Z" }
  ]
  ```

**Content Items (lectures, assignments, simulations, tests):**
- Added optional `dripDay` field (number or null)
  ```json
  {
    "id": 1,
    "title": "Week 1 Lecture",
    "url": "...",
    "date": "2025-11-01",
    "dripDay": 1
  }
  ```

### Model Functions

**classModel.js:**
- `addStudent()`: Records enrollment date when adding student
- `removeStudent()`: Cleans up enrollment record
- `filterContentByDripSchedule()`: Filters content arrays based on drip eligibility
- `getDaysSinceEnrollment()`: Calculates current class day for a student
- `updateLecture()`, `updateSimulation()`, `updateAssignment()`, `updateTest()`: Handle dripDay in updates

### Route Updates

**Admin Routes (/routes/admin.js):**
- `POST /classes/:id/lectures` - Accepts and saves dripDay
- `POST /classes/:id/lectures/:lectureId/update` - Updates dripDay
- `POST /classes/:id/simulations` - Accepts and saves dripDay
- `POST /classes/:id/simulations/:simId/update` - Updates dripDay
- `POST /classes/:id/assignments` - Accepts and saves dripDay
- `POST /classes/:id/assignments/:assignmentId/update` - Updates dripDay
- `POST /classes/:id/tests/:testId/update` - Updates dripDay

**Teacher Routes (/routes/teacher.js):**
- Same endpoints as admin routes with `/teacher` prefix

**Student Routes (/routes/student.js):**
- `GET /classes/:id` - Filters content and calculates daysSinceEnrollment

### View Updates

**view_class.ejs:**
- Added "Drip Day" input fields to all content add/edit forms
- Display day badges next to content items
- Show "Class Day X" banner for students
- Added informational text about drip scheduling

---

## Best Practices

### Content Planning

1. **Map Out Your Schedule**: Before setting drip days, create a course outline
   - Week 1 (Days 1-7): Foundation
   - Week 2 (Days 8-14): Application
   - etc.

2. **Space Content Appropriately**: 
   - Don't overwhelm students with too much on Day 1
   - Allow time for practice between new concepts
   - Schedule assessments after learning material

3. **Use Consistent Patterns**:
   - Lectures at start of new topics
   - Practice assignments 1-2 days after lectures
   - Quizzes mid-week
   - Tests at end of modules

### Testing Content Schedules

1. **Enroll a Test Student**: Create a test account and enroll in the class
2. **View as Student**: Check what content appears on Day 1
3. **Adjust Enrollment Date**: To test future days, manually update the enrollment date in the database
4. **Verify Progression**: Ensure content unlocks at the right times

### Combining with Dates

- **Drip Day**: Controls when content becomes *available*
- **Date Field**: Suggests when content should be *completed*
- **Example**: 
  - Assignment available Day 2 (drip day)
  - Due date set for Day 5 (date field)
  - Gives students 3 days to complete

---

## Troubleshooting

### Content Not Appearing for Students

**Check:**
1. Is the student actually enrolled in the class?
2. Is `dripDay` set to a number less than or equal to their current class day?
3. Check browser console for JavaScript errors
4. Verify `studentEnrollments` array exists in class data

### Wrong Day Number Displayed

**Check:**
1. Enrollment date is correctly set in `studentEnrollments`
2. Server time zone is correct
3. Date calculation logic in `getDaysSinceEnrollment()`

### Content Shows for Some Students But Not Others

**Check:**
1. Different enrollment dates in `studentEnrollments`
2. Each student's calculated class day
3. Ensure `dripDay` is set appropriately for all intended viewers

---

## Future Enhancements

Potential improvements for future versions:

1. **Bulk Content Management**: Upload CSV with content and drip days
2. **Visual Timeline**: Calendar view showing when content unlocks
3. **Templates**: Save drip schedules as templates for reuse
4. **Preview Mode**: Let teachers preview student view for different days
5. **Notifications**: Alert students when new content unlocks
6. **Analytics**: Track when students access newly unlocked content
7. **Conditional Unlocking**: Require completion of earlier content before unlocking new content
8. **Custom Pace**: Allow students to request early access (with approval)

---

## Migration Notes

### Existing Classes

- Classes created before this feature will continue to work normally
- All content without `dripDay` values will remain visible to all enrolled students
- No data migration required
- To start using drip scheduling:
  1. Edit each content item
  2. Set appropriate drip day numbers
  3. Existing enrollments get today's date as enrollment date on next update

### Backward Compatibility

- The feature is fully backward compatible
- `null` or missing `dripDay` = always available (original behavior)
- Existing routes continue to work without drip day parameters
- Students see all content if no drip schedule is set

---

## Support

For questions or issues with the drip schedule feature:

1. Check this documentation first
2. Review the console logs for errors
3. Verify database structure matches expected schema
4. Test with a simple case (single student, single content item)
5. Check that class `studentEnrollments` array is populated

---

## Summary

The drip schedule feature provides a flexible, student-centric way to manage course content delivery. By using relative day numbers instead of fixed dates, it accommodates rolling enrollments while maintaining a structured learning progression. The implementation is simple for instructors (just enter day numbers) and completely transparent for students (content appears when ready).
