# Document Template Management Guide

## Overview
This guide covers the template management and staff selection improvements for the document signing system.

## Features

### 1. Template Deletion
Admins can now delete signature document templates directly from the template selection page.

#### Location
- URL: `/document-signing/create/:studentId`
- Accessible: Admin users only

#### How to Delete a Template
1. Navigate to the document signing page for any student
2. Locate the template you want to delete in the templates list
3. Click the red trash icon button in the top-right corner of the template card
4. Confirm the deletion in the popup dialog
5. The template card will fade out and be removed

#### What Gets Deleted
- ✅ Database record from `signature_document_templates` table
- ✅ Physical PDF file from the server filesystem
- ✅ Template card from the UI with smooth animation

#### Error Handling
- If the physical file is already missing, the database record is still deleted
- Errors are logged but don't prevent the deletion from completing
- User receives success message after database deletion completes

#### Empty State
When all templates are deleted, the entire templates section is automatically hidden.

### 2. Staff Member Dropdown Improvements
The staff selection dropdown in document signing has been improved for better usability.

#### Location
- URL: `/document-signing/edit/:documentId`
- Field: Staff Member Assignment dropdown

#### Changes Made

**Before:**
```
John Doe (admin)
Jane Smith (teacher)
Bob Johnson (admin)
```

**After:**
```
Bob Johnson (bob.johnson@school.com)
John Doe (john.doe@school.com)
```

#### Features
- **Email Display**: Shows staff member's email address instead of role
- **Admins Only**: Only users with "admin" role appear in the dropdown
- **Sorted by Name**: Staff members are alphabetically sorted by first name, then last name
- **Full Name**: Shows complete first and last name for easy identification

## Technical Implementation

### Database Tables
```sql
-- Templates
signature_document_templates
  - id (INT)
  - name (VARCHAR)
  - file_path (VARCHAR)
  - created_at (DATETIME)

-- Staff Query
SELECT id,
  JSON_UNQUOTE(JSON_EXTRACT(profile, '$.firstName')) as firstName,
  JSON_UNQUOTE(JSON_EXTRACT(profile, '$.lastName')) as lastName,
  email, role 
FROM mdtslms_users 
WHERE role = "admin"
ORDER BY JSON_UNQUOTE(JSON_EXTRACT(profile, '$.firstName')), 
         JSON_UNQUOTE(JSON_EXTRACT(profile, '$.lastName'))
```

### API Endpoints

#### Delete Template
```
DELETE /document-signing/template/:templateId
```

**Authorization:** Requires admin role

**Response:**
```json
{
  "success": true,
  "message": "Template deleted successfully"
}
```

**Error Response:**
```json
{
  "success": false,
  "message": "Error deleting template"
}
```

### File System Structure
```
/home/mdtsapps/lms.mdts-apps.com/
└── uploads/
    └── signature-documents/
        ├── doc-[timestamp]-[hash].pdf
        └── templates/
            └── template-[timestamp]-[hash].pdf
```

## Security Considerations

### Template Deletion
- ✅ Requires admin authentication
- ✅ Protected by `requireAdmin` middleware
- ✅ Validates template exists before deletion
- ✅ No cascade deletion risk (templates are independent)

### Staff Assignment
- ✅ Only admins can assign staff members
- ✅ Dropdown only shows admin users
- ✅ Role verification on backend before assignment
- ✅ Session-based authentication required

## User Workflows

### Workflow: Delete Unused Template
1. Admin logs in
2. Navigates to any student's document signing page
3. Reviews available templates
4. Clicks delete button on obsolete template
5. Confirms deletion
6. Template is removed from system
7. Admin can verify template is gone by refreshing page

### Workflow: Assign Document to Staff Member
1. Admin creates or edits a signature document
2. Opens "Assign to Staff Member" modal
3. Reviews dropdown list showing admin names and emails
4. Selects appropriate admin by recognizing their email
5. Saves assignment
6. Selected admin can now access the document

## Troubleshooting

### Template Won't Delete
**Issue:** Delete button doesn't respond
- **Check:** Browser console for JavaScript errors
- **Check:** Network tab shows 401 Unauthorized → Not logged in as admin
- **Check:** Network tab shows 500 error → Check server logs

**Issue:** Template deleted from UI but still appears on refresh
- **Check:** Database record was actually deleted
- **Check:** Browser cache (hard refresh with Cmd+Shift+R)

### Staff Dropdown Issues
**Issue:** No staff members appear in dropdown
- **Check:** At least one user has role = "admin" in database
- **Check:** Admin users have firstName and lastName in profile JSON
- **Check:** Server logs for query errors

**Issue:** Dropdown shows "[object Object]"
- **Check:** Template file (admin_signature_document_edit.ejs) has correct field access
- **Fix:** Should be `<%= staff.email %>` not `<%= staff %>`

## Maintenance

### Regular Tasks
- **Weekly**: Review orphaned files in uploads/signature-documents/templates/
- **Monthly**: Check for templates with no associated documents
- **Quarterly**: Verify all template file_paths are valid

### Database Cleanup
```sql
-- Find templates with missing files (run periodically)
SELECT id, name, file_path 
FROM signature_document_templates 
WHERE file_path NOT IN (
  -- List of actual files from filesystem
);

-- Find orphaned template files (compare filesystem to DB)
-- Use server-side script to compare uploads/ folder with DB records
```

## Future Enhancements

### Potential Improvements
- [ ] Bulk delete multiple templates
- [ ] Template preview before deletion
- [ ] Undo functionality (move to trash instead of permanent delete)
- [ ] Template usage statistics (how many documents use each template)
- [ ] Template versioning system
- [ ] Staff member search/filter in dropdown
- [ ] Recently used staff members at top of dropdown
- [ ] Template categories or tags

## Related Documentation
- `DOCUMENT_SIGNING_README.md` - Overall document signing system
- `DOCUMENT_TEMPLATES_FEATURE.md` - Template creation and management
- `ADMIN_STAFF_SIGNATURE_FEATURE.md` - Staff signature workflows
- `SIGNATURE_PDF_GENERATION.md` - PDF generation process

## Support
For issues or questions about template management:
1. Check this documentation first
2. Review server logs in console
3. Check browser console for frontend errors
4. Verify database state with SQL queries
5. Contact system administrator if issues persist
