# Career Services Feature - Student Profile Integration

## Overview
A comprehensive Career Services section has been added to the Student Profile Record page, allowing faculty and administrators to track all career-related activities, interviews, documents, and mandatory employment survey results required by SCHEV/WIOA/Veteran Affairs.

## Feature Location
- **Student Profile Page**: Located towards the bottom of the page (before the Back button)
- **Access Level**: Admin and Teacher roles only
- **Route**: `/admin/students/:id` or `/teacher/students/:id`

## Components

### 1. Career Interviews Tab
Track and schedule career counseling sessions, mock interviews, and job search support.

**Features:**
- Schedule new interviews with date, time, type, and status
- Interview types include:
  - Career Counseling
  - Mock Interview
  - Job Search Support
  - Resume Review
  - Other
- Status tracking: Scheduled, Completed, Cancelled
- Add detailed notes for each interview
- Edit and delete existing interviews
- Tracks who added/modified each entry

**Data Structure:**
```javascript
{
  date: ISO Date String,
  time: "HH:MM",
  type: String,
  status: String,
  notes: String,
  addedBy: String,
  addedAt: ISO Date String,
  updatedBy: String,
  updatedAt: ISO Date String
}
```

### 2. Career Notes Tab
Document ongoing career development activities and counselor observations.

**Features:**
- Add titled notes with rich content
- Edit and delete existing notes
- Tracks author and timestamp for each note
- Displays notes in chronological order
- Ideal for documenting:
  - Career counseling sessions
  - Student progress updates
  - Action items and follow-ups
  - Skills assessments

**Data Structure:**
```javascript
{
  title: String,
  content: String,
  addedBy: String,
  date: ISO Date String,
  updatedBy: String,
  updatedAt: ISO Date String
}
```

### 3. Career Documents Tab
Upload and manage career-related documents for each student.

**Features:**
- Upload resumes, cover letters, and other career documents
- View document metadata (name, size, upload date)
- Download documents
- Delete documents (also removes physical file)
- Tracks who uploaded each document

**Supported File Types:**
- PDF, DOC, DOCX (documents)
- JPG, PNG (images)
- Maximum file size: 10MB per file

**Data Structure:**
```javascript
{
  originalName: String,
  mimeType: String,
  size: Number (bytes),
  url: String,
  uploadedAt: ISO Date String,
  uploadedBy: String
}
```

### 4. Employment Survey Tab
**COMPLIANCE FEATURE** - Record mandatory employment hiring results for regulatory reporting.

**Required By:**
- SCHEV (State Council of Higher Education for Virginia)
- WIOA (Workforce Innovation and Opportunity Act)
- Veteran Affairs

**Features:**
- Record employment status:
  - Employed Full-Time
  - Employed Part-Time
  - Self-Employed
  - Unemployed - Seeking
  - Unemployed - Not Seeking
  - Continuing Education
  - Military
  - Other
- Capture job details: title, employer, start date
- Track salary range (with privacy options)
- Indicate if employment is related to training
- Record survey date and additional notes
- Edit survey results as needed
- Single survey record per student (can be updated)

**Data Structure:**
```javascript
{
  employmentStatus: String,
  jobTitle: String,
  employer: String,
  startDate: ISO Date String,
  salaryRange: String,
  relatedToTraining: Boolean,
  surveyDate: ISO Date String,
  notes: String,
  recordedBy: String,
  recordedAt: ISO Date String,
  updatedBy: String,
  updatedAt: ISO Date String
}
```

## Database Schema

All career services data is stored in the `profile` JSON field of the `mdtslms_users` table under a `careerServices` object:

```javascript
student.profile.careerServices = {
  interviews: [...],    // Array of interview objects
  notes: [...],         // Array of note objects
  documents: [...],     // Array of document metadata objects
  employmentSurvey: {}  // Single employment survey object
}
```

No database migrations required - the feature uses the existing flexible JSON profile structure.

## API Endpoints

### Interview Management
- `POST /admin/students/:id/career-services/interview` - Schedule new interview
- `PUT /admin/students/:id/career-services/interview/:index` - Update interview
- `DELETE /admin/students/:id/career-services/interview/:index` - Delete interview

### Notes Management
- `POST /admin/students/:id/career-services/note` - Add new note
- `PUT /admin/students/:id/career-services/note/:index` - Update note
- `DELETE /admin/students/:id/career-services/note/:index` - Delete note

### Document Management
- `POST /admin/students/:id/career-services/document` - Upload document (multipart/form-data)
- `DELETE /admin/students/:id/career-services/document/:index` - Delete document

### Employment Survey
- `POST /admin/students/:id/career-services/employment-survey` - Create survey record
- `PUT /admin/students/:id/career-services/employment-survey` - Update survey record

All endpoints require admin or teacher authentication via session middleware.

## User Interface

### Design
- Uses Bootstrap 5 tab navigation for organized sections
- Consistent with existing student profile styling
- Orange accent bars and primary color scheme
- Responsive tables and forms
- SweetAlert2 modals for all CRUD operations

### User Experience
- All operations use AJAX requests (no page reloads for edits/deletes)
- Success/error feedback via SweetAlert modals
- Confirmation dialogs for destructive actions
- Inline editing via modal dialogs
- Auto-reload after successful operations to reflect changes

## File Storage

Career service documents are stored in:
```
/public/docs/stxd/[filename]
```

This reuses the existing document upload infrastructure (`generalDocUpload` multer middleware).

## Security Considerations

1. **Access Control**: Only admin and teacher roles can view/modify career services data
2. **File Validation**: Uses existing multer configuration for file type and size restrictions
3. **Input Sanitization**: All text inputs are sanitized before storage
4. **Authentication**: All endpoints require active session with appropriate role
5. **File Deletion**: Physical files are properly deleted when records are removed

## Compliance & Reporting

### SCHEV/WIOA/VA Requirements
The Employment Survey tab specifically addresses mandatory reporting requirements:

1. **Employment Status Tracking**: Captures detailed employment outcomes
2. **Training Relevance**: Records whether employment is related to program training
3. **Wage Information**: Optional salary range capture for reporting
4. **Survey Date**: Tracks when data was collected for compliance timelines
5. **Audit Trail**: Records who entered/modified data and when

### Reporting Recommendations
For compliance reporting, query students by:
- Survey completion status
- Employment outcomes by status
- Training-related employment percentage
- Survey collection dates for time-period reporting

Example query logic:
```javascript
const students = await getAllStudents();
const employed = students.filter(s => 
  s.profile?.careerServices?.employmentSurvey?.employmentStatus?.includes('Employed')
);
const trainingRelated = employed.filter(s =>
  s.profile?.careerServices?.employmentSurvey?.relatedToTraining === true
);
```

## Future Enhancements

Potential additions to consider:
1. **Bulk Export**: Export all career services data to CSV/Excel for reporting
2. **Email Integration**: Send interview confirmations and reminders
3. **Calendar Integration**: Sync interviews to admin/teacher calendars
4. **Document Templates**: Store resume/cover letter templates for students
5. **Job Board Integration**: Link to job postings and track applications
6. **Alumni Network**: Connect students with alumni for mentoring
7. **Analytics Dashboard**: Visualize employment outcomes and trends
8. **Automated Surveys**: Send employment surveys via email with form links
9. **Certification Tracking**: Link career certifications to career services
10. **Employer Partnerships**: Track employer relationships and placements

## Testing Checklist

- [ ] Schedule a career interview for a student
- [ ] Edit interview details (date, status, notes)
- [ ] Delete an interview
- [ ] Add career counseling notes
- [ ] Edit existing notes
- [ ] Delete a note
- [ ] Upload a resume document
- [ ] Download a document
- [ ] Delete a document (verify physical file deletion)
- [ ] Record employment survey results
- [ ] Edit employment survey
- [ ] Verify data persistence across page reloads
- [ ] Test with teacher role permissions
- [ ] Test that students cannot see this section
- [ ] Verify all validation and error handling

## Related Files

**Views:**
- `/views/student_profile.ejs` - Main UI implementation

**Routes:**
- `/routes/admin.js` - All API endpoints (lines 7668-8114)

**Models:**
- `/models/userModel.js` - Profile update methods (existing)

**Middleware:**
- Existing authentication middleware
- Existing `generalDocUpload` multer configuration

## Support & Maintenance

For issues or questions:
1. Check server logs for error messages
2. Verify file permissions on `/public/docs/stxd/` directory
3. Confirm student profile JSON structure is intact
4. Test API endpoints directly with tools like Postman
5. Review browser console for client-side JavaScript errors

---

**Implementation Date**: November 21, 2025
**Version**: 1.0
**Status**: Production Ready
